﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Windows.Forms;
using System.Xaml;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UIControls.Xaml
{
    /// <summary>
    /// An extended XamlType class.
    /// </summary>
    public class EffectMakerXamlType : XamlType
    {
        /// <summary>
        /// Initializes the EffectMakerXamlType instance.
        /// </summary>
        /// <param name="underlyingType">Related runtime type.</param>
        /// <param name="schemaContext">Current schema context provided by the XAML engine.</param>
        public EffectMakerXamlType(Type underlyingType, XamlSchemaContext schemaContext)
            : base(underlyingType, schemaContext)
        {
        }

        /// <summary>
        /// Route the element content construction.
        /// </summary>
        /// <returns>Returns an instance of PropertyXamlMember routing the content to the
        /// 'Controls' property, or returns the result of the base implementation.</returns>
        protected override XamlMember LookupContentProperty()
        {
            if (typeof(IControl).IsAssignableFrom(this.UnderlyingType))
            {
                var propertyInfo = UnderlyingType.GetProperty("Controls", typeof(IIndexableCollection<ILogicalTreeElement>));
                return new PropertyXamlMember(propertyInfo, SchemaContext);
            }
            else if (typeof(Control).IsAssignableFrom(this.UnderlyingType))
            {
                var propertyInfo = UnderlyingType.GetProperty("Controls", typeof(Control.ControlCollection));
                return new PropertyXamlMember(propertyInfo, SchemaContext);
            }

            return base.LookupContentProperty();
        }

        /// <summary>
        /// Route the member attribution for regular properties.
        /// </summary>
        /// <param name="name">The name of the member to lookup.</param>
        /// <param name="skipReadOnlyCheck">Tells wheter the readonly check
        /// is skipped or not.</param>
        /// <returns>Returns an instance of PropertyXamlMember if the member is found on the
        /// type, or the result of the base implementation otherwise.</returns>
        protected override XamlMember LookupMember(string name, bool skipReadOnlyCheck)
        {
            var propertyInfo = UnderlyingType.GetProperty(name);
            if (propertyInfo != null)
            {
                return new PropertyXamlMember(propertyInfo, SchemaContext);
            }

            return base.LookupMember(name, skipReadOnlyCheck);
        }

        /// <summary>
        /// Route the member attribution for attached properties.
        /// </summary>
        /// <param name="name">The name of the member to lookup.</param>
        /// <returns>Returns an instance of AttachedPropertyXamlMember if the getter or the setter
        /// is found on the type, or the result of the base implementation otherwise.</returns>
        protected override XamlMember LookupAttachableMember(string name)
        {
            var getMethod = this.UnderlyingType.GetMethod(
                "Get" + name,
                BindingFlags.Public | BindingFlags.Static);

            var setMethod = this.UnderlyingType.GetMethod(
                "Set" + name,
                BindingFlags.Public | BindingFlags.Static);

            if (getMethod != null || setMethod != null)
            {
                return new AttachedPropertyXamlMember(name, getMethod, setMethod, SchemaContext);
            }

            return base.LookupAttachableMember(name);
        }
    }
}
