﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Globalization;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.ValueConverters
{
    /// <summary>
    /// An IValueConverter that converts CheckState to boolean back and forth.
    /// </summary>
    public class CollapsedVisibilityValueConverter : IValueConverter
    {
        /// <summary>
        /// Convert boolean to CheckState, true means Checked and false means Unchecked.
        /// </summary>
        /// <param name="value">The boolean value to convert.
        /// If the value is not a boolean, null is returned.</param>
        /// <param name="targetType">The type to convert to.</param>
        /// <param name="parameter">A custom parameter.</param>
        /// <returns>Returns Checked if the value is true, Unchecked if the value is false,
        /// and null if the value is not of type boolean.</returns>
        public object Convert(
            object value,
            Type targetType,
            object parameter)
        {
            if ((value is bool) == false)
            {
                return null;
            }

            var v = (bool)value;
            return v ? Visibility.Visible : Visibility.Collapsed;
        }

        /// <summary>
        /// Convert CheckState to boolean, Checked and Indeterminate means true and
        /// Unchecked means false.
        /// </summary>
        /// <param name="value">The CheckState value to convert.
        /// If the value is not a CheckState enum, null is returned.</param>
        /// <param name="targetType">The type to convert to.</param>
        /// <param name="parameter">A custom parameter.</param>
        /// <returns>Returns true if the value is Checked or Indeterminate,
        /// false if the value is Unchecked,
        /// and null if the value is not of type CheckState num.</returns>
        public object ConvertBack(
            object value,
            Type targetType,
            object parameter)
        {
            if (Enum.IsDefined(typeof(Visibility), value) == false)
            {
                return null;
            }

            var v = (Visibility)Enum.ToObject(typeof(Visibility), value);

            return v == Visibility.Visible;
        }
    }
}
