﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Xaml;

namespace EffectMaker.UIControls.SpecificControls
{
    /// <summary>
    /// An extended UIUserControl that loads control based on XAML file.
    /// </summary>
    public abstract class UIXamlControl : UIUserControl
    {
        /// <summary>
        /// Stores the flag locking the manual control add capability.
        /// </summary>
        private bool controlAddAllowed;

        /// <summary>
        /// Gets the root control loaded from XAML file.
        /// </summary>
        protected IControl RootChildControl { get; private set; }

        /// <summary>
        /// Gets the flattened controls loaded from XAML file.
        /// Only the named controls are available.
        /// </summary>
        protected IDictionary<string, IControl> FlattenedControls { get; private set; }

        /// <summary>
        /// Called when the user control is loaded.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnLoad(EventArgs e)
        {
            EffectMaker.UIControls.Layout.LayoutEngineBase.SuspendLayout();

            base.OnLoad(e);

            var assembly = this.GetType().Assembly;
            var relativeName = string.Format("XamlControls.{0}.xaml", this.GetType().Name);

            IControl rootControl;

            using (Stream stream = ResourceUtility.LoadEmbedded(assembly, relativeName))
            {
                rootControl = XamlLoader.Load(stream, assembly) as IControl;
                if (rootControl == null)
                {
                    var msg = string.Format("Invalid resource type '{0}'.", relativeName);
                    throw new InvalidOperationException(msg);
                }
            }

            var controls = rootControl.FlattenControlTree();

            this.RootChildControl = rootControl;
            this.FlattenedControls = controls;

            ((Control)rootControl).Dock = DockStyle.Fill;

            this.controlAddAllowed = true;

            try
            {
                this.AddChildControls(rootControl);
            }
            finally
            {
                this.controlAddAllowed = false;
            }

            this.OnXamlLoaded(rootControl, controls);

            EffectMaker.UIControls.Layout.LayoutEngineBase.ResumeLayout();
        }

        /// <summary>
        /// When overridden, called once the XAML file has been loaded successfully.
        /// </summary>
        /// <param name="rootControl">The root control defined in the XAML file.</param>
        /// <param name="controls">The named controls contained in the XAML file.</param>
        protected virtual void OnXamlLoaded(
            IControl rootControl,
            IDictionary<string, IControl> controls)
        {
        }

        /// <summary>
        /// Called when a child control is added.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnControlAdded(ControlEventArgs e)
        {
            if (this.controlAddAllowed == false)
            {
                var msg = "Forbidden to add controls manually. It must be done through XAML.";
                throw new InvalidOperationException(msg);
            }

            base.OnControlAdded(e);
        }
    }
}
