﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.Windows.Forms.Layout;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Layout
{
    /// <summary>
    /// コントロールを左右にレイアウトします。
    /// </summary>
    public class SideBySidePanesLayoutEngine : LayoutEngineBase
    {
        /// <summary>
        /// 一列に配置するかどうかのフラグ。
        /// </summary>
        private bool isSingleColumn = false;

        /// <summary>
        /// 左枠の幅。
        /// </summary>
        private int leftPaneWidth = -1;

        /// <summary>
        /// 右枠の幅。
        /// </summary>
        private int rightPaneWidth = -1;

        /// <summary>
        /// 左右の枠の距離。
        /// </summary>
        private int sideBySideDistance = 2;

        /// <summary>
        /// 上下の枠の距離。
        /// </summary>
        private int topAndBottomDistance = 0;

        /// <summary>
        /// Padding.
        /// </summary>
        private Padding padding = new Padding(0);

        /// <summary>
        /// 前回計算時のコンテナサイズ.
        /// </summary>
        private Size cachedSize = new Size(-1, -1);

        /// <summary>
        /// 一列に配置するかどうかのフラグを取得または設定します。
        /// </summary>
        public bool IsSingleColumn
        {
            get
            {
                return this.isSingleColumn;
            }

            set
            {
                if (this.isSingleColumn != value)
                {
                    this.isSingleColumn = value;
                    this.OnRequestLayout(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// マージンを含めた左枠の幅を取得または設定します。
        /// </summary>
        public int LeftPaneWidth
        {
            get
            {
                return this.leftPaneWidth;
            }

            set
            {
                if (this.leftPaneWidth != value)
                {
                    this.leftPaneWidth = value;
                    this.OnRequestLayout(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// マージンを含めた右枠の幅を取得または設定します。
        /// </summary>
        public int RightPaneWidth
        {
            get
            {
                return this.rightPaneWidth;
            }

            set
            {
                if (this.rightPaneWidth != value)
                {
                    this.rightPaneWidth = value;
                    this.OnRequestLayout(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 左右の枠の距離を取得または設定します。
        /// </summary>
        public int SideBySideDistance
        {
            get
            {
                return this.sideBySideDistance;
            }

            set
            {
                if (value < 2)
                {
                    value = 2;
                }

                if (this.sideBySideDistance != value)
                {
                    this.sideBySideDistance = value;
                    this.OnRequestLayout(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 上下の枠の距離を取得または設定します。
        /// </summary>
        public int TopAndBottomDistance
        {
            get
            {
                return this.topAndBottomDistance;
            }

            set
            {
                if (this.topAndBottomDistance != value)
                {
                    this.topAndBottomDistance = value;
                    this.OnRequestLayout(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 分割線の位置を取得します。
        /// </summary>
        public int BorderPosition { get; private set; }

        /// <summary>
        /// UIの推薦サイズ(最小サイズ)を取得します.
        /// </summary>
        /// <param name="container">コンテナ(SideBySidePanesPanel)</param>
        /// <param name="proposedSize">有効な領域</param>
        /// <returns>UIの推薦サイズを返します.</returns>
        public Size GetPreferredSize(Control container, Size proposedSize)
        {
            ILayoutElement leftElement;
            ILayoutElement rightElement;

            // 左右に配置するコントロールを取得
            this.GetLeftAndRightElements(container.Controls, out leftElement, out rightElement);

            Rectangle preferredRectangle = new Rectangle(0, 0, proposedSize.Width, proposedSize.Height);
            Rectangle containerRectangle;

            if (this.isSingleColumn)
            {
                this.TopAndBottomLayout(
                    leftElement,
                    rightElement,
                    preferredRectangle,
                    false,
                    out containerRectangle);
            }
            else
            {
                this.SideBySideLayout(
                    leftElement,
                    rightElement,
                    preferredRectangle,
                    false,
                    out containerRectangle);
            }

            return containerRectangle.Size;
        }

        /// <summary>
        /// Perform the layout.
        /// </summary>
        /// <param name="container">Container control.</param>
        /// <param name="containerParentSize">Size of the parent of the current container.</param>
        /// <param name="layoutEventArgs">Layout event argument.</param>
        /// <returns>Returns true if the parent must perform layout too,
        /// false otherwise.</returns>
        protected override bool OnLayout(
            Control container,
            Size containerParentSize,
            LayoutEventArgs layoutEventArgs)
        {
            ILayoutElement leftElement;
            ILayoutElement rightElement;

            // 左右に配置するコントロールを取得。
            this.GetLeftAndRightElements(container.Controls, out leftElement, out rightElement);

            // Use DisplayRectangle so that parent.Padding is honored
            Rectangle parentDisplayRectangle = container.DisplayRectangle;

            Rectangle newRectangle;

            if (this.IsSingleColumn)
            {
                this.TopAndBottomLayout(
                    leftElement,
                    rightElement,
                    parentDisplayRectangle,
                    true,
                    out newRectangle);
            }
            else
            {
                this.SideBySideLayout(
                    leftElement,
                    rightElement,
                    parentDisplayRectangle,
                    true,
                    out newRectangle);
            }

            // コンテナのサイズが変わったときだけtrueを返す
            bool updateParent = false;

            if (newRectangle.Size != this.cachedSize)
            {
                this.cachedSize = newRectangle.Size;
                updateParent = true;
            }

            return updateParent;
        }

        /// <summary>
        /// 左右に配置するコントロールを取得します。
        /// </summary>
        /// <param name="controls">登録されたコントロール。</param>
        /// <param name="leftElement">左に配置するコントロールです。</param>
        /// <param name="rightElement">右に配置するコントロールです。</param>
        void GetLeftAndRightElements(
            Control.ControlCollection controls,
            out ILayoutElement leftElement,
            out ILayoutElement rightElement)
        {
            leftElement = null;
            rightElement = null;

            int paneIndex = 0;

            foreach (ILayoutElement child in controls)
            {
                if (paneIndex == 0)
                {
                    leftElement = child;
                }
                else if (paneIndex == 1)
                {
                    rightElement = child;
                }
                else
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "SideBySidePanesLayoutEngine.OnLayout : Three or more controls are added.");

                    break;
                }

                ++paneIndex;
            }

            // Collapsedのコントロールはnullにしてサイズを計算しないようにする。
            if (leftElement != null && leftElement.Visibility == Visibility.Collapsed)
            {
                leftElement = null;
            }

            if (rightElement != null && rightElement.Visibility == Visibility.Collapsed)
            {
                rightElement = null;
            }
        }

        /// <summary>
        /// コントロールを上下に配置します。
        /// </summary>
        /// <param name="leftElement">2列のとき左側に配置するコントロール。</param>
        /// <param name="rightElement">2列のとき右側に配置するコントロール。</param>
        /// <param name="displayRectangle">クライアント領域</param>
        /// <param name="applyLayout">レイアウトを適用するかどうか。</param>
        /// <param name="containerRectangle">コンテナのレイアウト。</param>
        private void TopAndBottomLayout(
            ILayoutElement leftElement,
            ILayoutElement rightElement,
            Rectangle displayRectangle,
            bool applyLayout,
            out Rectangle containerRectangle)
        {
            Rectangle availableRectangle = displayRectangle;

            Rectangle leftElementRect = new Rectangle();

            // 左コントロールを上に配置
            if (leftElement != null)
            {
                // マージン付きのサイズを取得
                Size displaySize = leftElement.GetElementDisplaySize(availableRectangle.Size);

                if (displaySize.Width < availableRectangle.Size.Width)
                {
                    displaySize.Width = availableRectangle.Size.Width;
                }

                if (this.LeftPaneWidth >= 0)
                {
                    displaySize.Width = this.LeftPaneWidth;
                }

                // 左枠の位置、サイズを計算
                leftElementRect.X = availableRectangle.X + leftElement.Margin.Left;
                leftElementRect.Y = availableRectangle.Y + leftElement.Margin.Top;
                leftElementRect.Size = displaySize;

                // 左枠のレイアウトを更新
                if (applyLayout)
                {
                    leftElement.Location = leftElementRect.Location;
                    leftElement.SetElementDisplaySize(leftElementRect.Size);
                }

                availableRectangle.Y += leftElement.Margin.Vertical + leftElement.Height + this.TopAndBottomDistance;
                availableRectangle.Height -= leftElement.Margin.Horizontal + leftElement.Height + this.TopAndBottomDistance;
            }

            Rectangle rightElementRect = new Rectangle();

            // 右コントロールを下に配置
            if (rightElement != null)
            {
                Size displaySize = rightElement.GetElementDisplaySize(availableRectangle.Size);

                if (displaySize.Width < availableRectangle.Size.Width)
                {
                    displaySize.Width = availableRectangle.Size.Width;
                }

                if (this.LeftPaneWidth >= 0)
                {
                    displaySize.Width = this.LeftPaneWidth;
                }

                // 右枠の位置、サイズを計算
                rightElementRect.X = availableRectangle.X + rightElement.Margin.Left;
                rightElementRect.Y = availableRectangle.Y + rightElement.Margin.Top;
                rightElementRect.Size = displaySize;

                // 右枠のレイアウトを更新
                if (applyLayout)
                {
                    rightElement.Location = rightElementRect.Location;
                    rightElement.SetElementDisplaySize(rightElementRect.Size);
                }
            }

            // コンテナのレイアウトを計算
            var container = (SideBySidePanesPanel)leftElement.Parent;

            containerRectangle = new Rectangle();
            containerRectangle.Location = displayRectangle.Location;
            containerRectangle.Width = container.Padding.Horizontal
                + Math.Max(leftElementRect.Width, rightElementRect.Width);
            containerRectangle.Height = container.Padding.Vertical + leftElementRect.Height
                + this.topAndBottomDistance + rightElementRect.Height;
        }

        /// <summary>
        /// コントロールを左右に配置します。
        /// </summary>
        /// <param name="leftElement">左側に配置するコントロール。</param>
        /// <param name="rightElement">右側に配置するコントロール。</param>
        /// <param name="displayRectangle">クライアント領域。</param>
        /// <param name="applyLayout">レイアウトを適用するかどうか。</param>
        /// <param name="containerRectangle">コンテナのレイアウト。</param>
        private void SideBySideLayout(
            ILayoutElement leftElement,
            ILayoutElement rightElement,
            Rectangle displayRectangle,
            bool applyLayout,
            out Rectangle containerRectangle)
        {
            // コンテナを取得
            var container = (SideBySidePanesPanel)leftElement.Parent;

            // 左右の枠の間隔、コンテナのパディングを引いた残りクライアント領域を計算
            Size availableSize = displayRectangle.Size;
            availableSize.Width -= this.SideBySideDistance + container.Padding.Horizontal;

            Rectangle leftElementRect = new Rectangle();

            if (leftElement != null)
            {
                // 左枠の位置・サイズを取得
                leftElementRect.Location = leftElement.Location;
                leftElementRect.Size = leftElement.GetElementDisplaySize(availableSize);

                if (this.LeftPaneWidth >= 0)
                {
                    leftElementRect.Width = this.LeftPaneWidth;
                }

                // 左枠の位置、サイズを計算
                leftElementRect.X = displayRectangle.X + leftElement.Margin.Left;
                leftElementRect.Y = displayRectangle.Y + leftElement.Margin.Top;

                // 左枠のレイアウトを更新
                if (applyLayout)
                {
                    leftElement.Location = leftElementRect.Location;
                    leftElement.SetElementDisplaySize(leftElementRect.Size);
                }
            }
            else if (this.LeftPaneWidth >= 0)
            {
                leftElementRect.Width = this.LeftPaneWidth;
            }

            availableSize.Width -= leftElementRect.Width;  // 残り領域から左枠のサイズ・マージンを引く

            // 中心線の位置を設定
            this.BorderPosition = displayRectangle.X + leftElementRect.Width
                                + container.Padding.Right + ((this.SideBySideDistance - 2) / 2);

            Rectangle rightElementRect = new Rectangle();

            if (rightElement != null)
            {
                // 右枠の位置・サイズを取得、設定
                rightElementRect.Location = rightElement.Location;
                rightElementRect.Size = rightElement.GetElementDisplaySize(availableSize);

                if (this.RightPaneWidth >= 0)
                {
                    rightElementRect.Width = this.RightPaneWidth;
                }

                // 右枠の位置、サイズを計算
                rightElementRect.X = displayRectangle.X + leftElementRect.Width
                                   + container.Padding.Horizontal + this.SideBySideDistance
                                   + rightElement.Margin.Left;
                rightElementRect.Y = displayRectangle.Y + rightElement.Margin.Top;

                // 右枠のレイアウトを更新
                if (applyLayout)
                {
                    rightElement.Location = rightElementRect.Location;
                    rightElement.SetElementDisplaySize(rightElementRect.Size);
                }
            }
            else if (this.RightPaneWidth >= 0)
            {
                rightElementRect.Width = this.RightPaneWidth;
            }

            // コンテナのレイアウトを計算
            containerRectangle = new Rectangle();
            containerRectangle.Location = displayRectangle.Location;
            containerRectangle.Width = container.Padding.Horizontal + leftElementRect.Width
                                     + this.sideBySideDistance + rightElementRect.Width;
            containerRectangle.Height = container.Padding.Vertical
                                      + Math.Max(leftElementRect.Height, rightElementRect.Height);
        }
    }
}
