﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Focus;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.ValueConverters;

namespace EffectMaker.UIControls.Extensions
{
    /// <summary>
    /// Class that contains extension methods related to
    /// the Control class and the IControl interface.
    /// </summary>
    public static class ControlExtensions
    {
        /// <summary>
        /// Suspends the drawing of the given control.
        /// </summary>
        /// <param name="ctrl">The control to suspend the drawing.</param>
        public static void SuspendDrawing(this Control ctrl)
        {
            if (ctrl != null && ctrl.IsHandleCreated)
            {
                EffectMaker.Foundation.Win32.Functions.SendMessage(
                    ctrl.Handle,
                    (int)EffectMaker.Foundation.Win32.WM.WM_SETREDRAW,
                    false,
                    0);
            }
        }

        /// <summary>
        /// Resumes the drawing of the given control.
        /// </summary>
        /// <param name="ctrl">The control to resume the drawing.</param>
        public static void ResumeDrawing(this Control ctrl)
        {
            if (ctrl != null && ctrl.IsHandleCreated)
            {
                EffectMaker.Foundation.Win32.Functions.SendMessage(
                    ctrl.Handle,
                    (int)EffectMaker.Foundation.Win32.WM.WM_SETREDRAW,
                    true,
                    0);
            }
        }

        /// <summary>
        /// Check if this control is an ancestor of the given control 'child'.
        /// </summary>
        /// <param name="self">This control.</param>
        /// <param name="child">The child control.</param>
        /// <returns>True if this control is the ancestor of 'child'.</returns>
        public static bool IsAncestorOf(this Control self, Control child)
        {
            Control parent = child.Parent;
            while (parent != null)
            {
                if (parent == self)
                {
                    return true;
                }

                parent = parent.Parent;
            }

            return false;
        }

        /// <summary>
        /// Retrieve the data item corresponding to the container.
        /// </summary>
        /// <param name="container">The container to look for its data item.</param>
        /// <returns>Returns the data item, or null if not found.</returns>
        public static object GetDataItemFromContainer(IControl container)
        {
            if (container == null)
            {
                return null;
            }

            var parent = container.Parent as IItemsControl;

            if (parent == null)
            {
                return null;
            }

            return parent.GetDataItemFromContainer(container);
        }

        /// <summary>
        /// Retrieve the container corresponding to the data item.
        /// </summary>
        /// <param name="itemsControl">The items control from which
        /// to look for the container.</param>
        /// <param name="dataItem">The data item to look for its data.</param>
        /// <returns>Returns the container, or null if not found.</returns>
        public static IControl GetContainerFromDataItem(
            IItemsControl itemsControl,
            object dataItem)
        {
            if (itemsControl == null)
            {
                throw new ArgumentNullException("itemsControl");
            }

            if (dataItem == null)
            {
                return null;
            }

            return itemsControl.GetContainerFromDataItem(dataItem);
        }

        /// <summary>
        /// Get a sequence of child instance of type IControl where
        /// the property Visilibity is set to Collapsed.
        /// </summary>
        /// <param name="control">The control to find the non collapsed children.</param>
        /// <returns>Returns a sequence of non collapsed IControl instances.</returns>
        public static IEnumerable<ILayoutElement> GetNonCollapsedControls(this IControl control)
        {
            var ctrl = control as Control;

            if (ctrl == null)
            {
                throw new ArgumentNullException("control");
            }

            return ctrl.Controls
                .OfType<ILayoutElement>()
                .Where(c => c.Visibility != Visibility.Collapsed);
        }

        /// <summary>
        /// Get a sequence of child instance of type IControl where
        /// the property Visilibity is set to Collapsed.
        /// </summary>
        /// <param name="control">The control to find the non collapsed children.</param>
        /// <returns>Returns a sequence of non collapsed IControl instances.</returns>
        public static IEnumerable<ILayoutElement> GetNonCollapsedControls(this Control control)
        {
            if (control == null)
            {
                throw new ArgumentNullException("control");
            }

            return control.Controls
                .OfType<ILayoutElement>()
                .Where(c => c.Visibility != Visibility.Collapsed);
        }

        /// <summary>
        /// Check if this control or any of its parent is collapsed.
        /// </summary>
        /// <param name="control">The control.</param>
        /// <returns>True if any of the parent or itself is collapsed.</returns>
        public static bool IsSelfOrParentCollapsed(this Control control)
        {
            if (control.Visible == true)
            {
                return false;
            }
            else
            {
                Control curr = control;
                while (curr != null)
                {
                    ILayoutElement layoutElement = curr as ILayoutElement;
                    if (layoutElement != null &&
                        layoutElement.Visibility == Visibility.Collapsed)
                    {
                        return true;
                    }

                    curr = curr.Parent;
                }
            }

            return false;
        }

        /// <summary>
        /// Add several child controls to a parent control.
        /// </summary>
        /// <param name="parent">The parent control.</param>
        /// <param name="childControls">The sequence of child controls.</param>
        public static void AddChildControls(this IControl parent, params IControl[] childControls)
        {
            AddChildControls(parent, (IEnumerable<IControl>)childControls);
        }

        /// <summary>
        /// Add several child controls to a parent control.
        /// </summary>
        /// <param name="parent">The parent control.</param>
        /// <param name="childControls">The sequence of child controls.</param>
        public static void AddChildControls(
            this IControl parent,
            IEnumerable<IControl> childControls)
        {
            if (parent == null)
            {
                throw new ArgumentNullException("parent");
            }

            if (childControls == null)
            {
                throw new ArgumentNullException("childControls");
            }

            var parentCtrl = parent as Control;
            if (parentCtrl != null)
            {
                parentCtrl.SuspendLayout();
            }

            foreach (var child in childControls)
            {
                var init = child as ISupportInitialize;
                if (init != null)
                {
                    init.BeginInit();
                }
                else
                {
                    var ctrl = child as Control;
                    if (ctrl != null)
                    {
                        ctrl.SuspendLayout();
                    }
                }
            }

            foreach (var child in childControls)
            {
                parent.Controls.Add(child);
            }

            foreach (var child in childControls)
            {
                var init = child as ISupportInitialize;

                if (init != null)
                {
                    init.EndInit();
                }
                else
                {
                    var ctrl = child as Control;
                    if (ctrl != null)
                    {
                        ctrl.ResumeLayout(false);
                    }
                }
            }

            if (parentCtrl != null)
            {
                parentCtrl.ResumeLayout(false);
                parentCtrl.PerformLayout();
            }
        }

        /// <summary>
        /// Set several child controls to a parent control.
        /// If there are already child controls in the parent control,
        /// this method removes all the children those are not in the
        /// provided child controls, add non-existing ones, and sort
        /// the children with the same order as in the provided child
        /// control IEnumerable.
        /// </summary>
        /// <param name="parent">The parent control.</param>
        /// <param name="childControls">The sequence of child controls.</param>
        /// <param name="minimumModification">True to do minimum modification to the child list.</param>
        public static void SetChildControls(
            this IControl parent,
            IEnumerable<IControl> childControls,
            bool minimumModification = true)
        {
            if (parent == null)
            {
                throw new ArgumentNullException("parent");
            }

            if (childControls == null)
            {
                throw new ArgumentNullException("childControls");
            }

            var parentCtrl = parent as Control;
            if (parentCtrl != null)
            {
                parentCtrl.SuspendLayout();
            }

            // ツリーノードの場合は展開状態を保持する
            var nodeCtrl = parent as TreeNode;
            if (nodeCtrl != null)
            {
                foreach (var ctrl in childControls)
                {
                    var newNode = ctrl as TreeNode;
                    if (newNode == null)
                    {
                        continue;
                    }

                    var oldNode = parent.Controls.FirstOrDefault(
                        x => x.DataContext == ctrl.DataContext) as TreeNode;
                    if (oldNode == null)
                    {
                        continue;
                    }

                    if (oldNode.IsExpanded)
                    {
                        newNode.Expand();
                    }
                    else
                    {
                        newNode.Collapse();
                    }
                }
            }

            IEnumerable<IControl> childrenToAdd = childControls;
            if (minimumModification == true)
            {
                // First remove the child controls those do not exist in the given child sequence.
                for (int i = parent.Controls.Count - 1; i >= 0; --i)
                {
                    var child = parent.Controls[i] as IControl;
                    if (child != null && childControls.Contains(child) == false)
                    {
                        parent.Controls.Remove(child);
                    }
                }

                // Find the controls those do not exist in the children list of the parent.
                childrenToAdd = childControls.Where(
                    c => parent.Controls.Contains(c) == false).ToArray();
            }
            else
            {
                // Just remove all the current children and add all the new child controls again.
                parent.Controls.Clear();
            }

            foreach (var child in childrenToAdd)
            {
                var init = child as ISupportInitialize;
                if (init != null)
                {
                    init.BeginInit();
                }
                else
                {
                    var ctrl = child as Control;
                    if (ctrl != null)
                    {
                        ctrl.SuspendLayout();
                    }
                }
            }

            // Add the children to the parent.
            foreach (var child in childrenToAdd)
            {
                parent.Controls.Add(child);
            }

            foreach (var child in childrenToAdd)
            {
                var init = child as ISupportInitialize;

                if (init != null)
                {
                    init.EndInit();
                }
                else
                {
                    var ctrl = child as Control;
                    if (ctrl != null)
                    {
                        ctrl.ResumeLayout(false);
                    }
                }
            }

            if (parentCtrl != null)
            {
                parentCtrl.ResumeLayout(false);
                parentCtrl.PerformLayout();
            }
        }

        /// <summary>
        /// Find a child control, and drill down the control
        /// hierarchy to find the bottom most one.
        /// </summary>
        /// <param name="control">The begining control to find from.</param>
        /// <param name="position">The control-space mouse coordinate to look at.</param>
        /// <returns>Returns a found child control, the given control otherwise.</returns>
        public static Control GetBottomMostChildAtPoint(this Control control, Point position)
        {
            Control found = control;

            while (true)
            {
                // positionで指定された位置にあるChildコントロールを取得。
                // TabControlは未選択のタブを返すことがあるので選択中のタブを取得。
                var tabControl = control as TabControl;
                if (tabControl != null)
                {
                    control = tabControl.SelectedTab;
                }
                else
                {
                    control = control.GetChildAtPoint(position);
                }

                if (control == null)
                {
                    break;
                }

                position = control.PointToClient(found.PointToScreen(position));
                found = control;
            }

            return found;
        }

        /// <summary>
        /// Flatten a tree of control into a dictionary of control.
        /// </summary>
        /// <param name="root">The root control of the tree.</param>
        /// <returns>Returns a dictionary or controls.</returns>
        public static IDictionary<string, IControl> FlattenControlTree(this IControl root)
        {
            var controls = new Dictionary<string, IControl>();

            FlattenControlTree(root, controls);

            return controls;
        }

        /// <summary>
        /// Creates and adds a Binder instance to the control.
        /// </summary>
        /// <param name="control">The control to bind.</param>
        /// <param name="controlPropertyName">The control property to bind.</param>
        /// <param name="dataSourcePropertyName">The data source property to bind.</param>
        /// <param name="valueConverter">The value converter.</param>
        /// <param name="converterParameter">Parameter for the value converter.</param>
        /// <returns>The added binder instance.</returns>
        public static Binder AddBinding(
            this IBindable control,
            string controlPropertyName,
            string dataSourcePropertyName,
            IValueConverter valueConverter = null,
            object converterParameter = null)
        {
            if (control == null)
            {
                throw new ArgumentNullException("control");
            }

            var binder = new Binder(control, controlPropertyName, dataSourcePropertyName);
            binder.ValueConverter = valueConverter;
            binder.ConverterParameter = converterParameter;

            control.Bindings.Add(binder);

            return binder;
        }

        /// <summary>
        /// Shortcut method to add a behavior to a control.
        /// </summary>
        /// <param name="control">The control to attach the behavior to.</param>
        /// <param name="behavior">The behavior to attach to the control.</param>
        public static void AddBehavior(this IControl control, Behavior behavior)
        {
            if (control == null)
            {
                throw new ArgumentNullException("control");
            }

            if (behavior == null)
            {
                throw new ArgumentNullException("behavior");
            }

            control.Behaviors.Add(behavior);
        }

        /// <summary>
        /// Shortcut method to add a behavior to a control.
        /// </summary>
        /// <typeparam name="T">The type of object associated to the behvior.</typeparam>
        /// <param name="control">The control to attach the behavior to.</param>
        /// <param name="behavior">The behavior to attach to the control.</param>
        public static void AddBehavior<T>(this IControl control, Behavior<T> behavior)
        {
            AddBehavior(control, (Behavior)behavior);
        }

        /// <summary>
        /// Determine if the control can receive focus.
        /// </summary>
        /// <param name="self">The control.</param>
        /// <param name="ignoreVisibility">True to ignore visible and enable flags.</param>
        /// <returns>True if the control can receive focus.</returns>
        public static bool CanReceiveFocus(this Control self, bool ignoreVisibility = false)
        {
            if (self == null)
            {
                return false;
            }

            if (ignoreVisibility == false &&
                (self.Visible == false ||
                 self.Enabled == false))
            {
                return false;
            }
            else if (self.TabStop == false)
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// Create a dictionary of controls from a control tree.
        /// </summary>
        /// <param name="root">The root control of the tree.</param>
        /// <param name="controls">The dictionary of controls to populate.</param>
        private static void FlattenControlTree(
            IControl root,
            IDictionary<string, IControl> controls)
        {
            if (string.IsNullOrWhiteSpace(root.Name) == false)
            {
                controls[root.Name] = root;
            }

            foreach (var child in root.Controls.OfType<IControl>())
            {
                FlattenControlTree(child, controls);
            }
        }
    }
}
