﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.Extenders
{
    /// <summary>
    /// This class extends the capabilities of a ILayoutElement.
    /// </summary>
    public class LayoutElementExtender : ExtenderBase, IDisposable
    {
        /// <summary>
        /// Stores the extended layout element.
        /// </summary>
        private ILayoutElement extendedElement;

        /// <summary>
        /// Backing field for Visibility property.
        /// </summary>
        private Visibility visibility = Visibility.Visible;

        /// <summary>
        /// Backing field for HorizontalAlignment property.
        /// </summary>
        private HAlignment horizontalAlignment = HAlignment.Left;

        /// <summary>
        /// Backing field for VerticalAlignment property.
        /// </summary>
        private VAlignment verticalAlignment = VAlignment.Top;

        /// <summary>
        /// Initializes the ControlExtender.
        /// </summary>
        /// <param name="extendedElement">ILayoutElement to extend the capabilitites.</param>
        public LayoutElementExtender(ILayoutElement extendedElement)
            : base(extendedElement)
        {
            this.extendedElement = extendedElement;
            this.extendedElement.VisibleChanged += this.OnExtendedElementVisibleChanged;
        }

        /// <summary>
        /// Gets or sets the visibility of the control.
        /// This property may raise a 'Visibility' change notification.
        /// </summary>
        protected internal Visibility Visibility
        {
            get
            {
                return this.visibility;
            }

            set
            {
                if (this.SetValue(
                    ref this.visibility,
                    value))
                {
                    this.extendedElement.Visible = this.visibility == Visibility.Visible;
                    if (LayoutEngineBase.IsLayoutSuspended == false)
                    {
                        this.extendedElement.PerformLayout();
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment of the control.
        /// This property may raise a 'HorizontalAlignment' change notification.
        /// </summary>
        protected internal HAlignment HorizontalAlignment
        {
            get
            {
                return this.horizontalAlignment;
            }

            set
            {
                if (this.SetValue(
                    ref this.horizontalAlignment,
                    value))
                {
                    this.extendedElement.PerformLayout();
                }
            }
        }

        /// <summary>
        /// Gets or sets the vertical alignment of the control.
        /// This property may raise a 'VerticalAlignment' change notification.
        /// </summary>
        protected internal VAlignment VerticalAlignment
        {
            get
            {
                return this.verticalAlignment;
            }

            set
            {
                if (this.SetValue(
                    ref this.verticalAlignment,
                    value))
                {
                    this.extendedElement.PerformLayout();
                }
            }
        }

        /// <summary>
        /// Dispose the ControlExtender.
        /// </summary>
        public void Dispose()
        {
            this.extendedElement.VisibleChanged -= this.OnExtendedElementVisibleChanged;
        }

        /// <summary>
        /// Raised when the extended control get visible or hidden.
        /// This method raises a 'Visible' property changed notification.
        /// </summary>
        /// <param name="sender">Caller that raise the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnExtendedElementVisibleChanged(object sender, EventArgs e)
        {
            if (this.extendedElement is System.Windows.Forms.Control)
            {
                if (((System.Windows.Forms.Control)this.extendedElement).IsHandleCreated == false)
                {
                    return;
                }
            }

            this.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                "Visible");
        }
    }
}
