﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Core;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Extenders
{
    /// <summary>
    /// A class that extend items controls.
    /// </summary>
    public class ItemsControlExtender : ExtenderBase
    {
        /// <summary>
        /// The IItemsControl instance to extend.
        /// </summary>
        private IItemsControl extendedItemsControl;

        /// <summary>
        /// Backing field for ItemsSource.
        /// </summary>
        private IEnumerable currentItemsSource;

        /// <summary>
        /// Stores the items source typed INotifyCollectionChanged that was set before it changed.
        /// </summary>
        private IEnumerable previousItemsSource;

        /// <summary>
        /// Stores a flag telling whether a child control can be added or not.
        /// </summary>
        private bool isAddingFromItemsSource;

        /// <summary>
        /// Backing field for the ItemContainerType property.
        /// </summary>
        private Type itemContainerType;

        /// <summary>
        /// Backing field for the ItemContainerSelector property.
        /// </summary>
        private IItemContainerSelector itemContainerSelector;

        /// <summary>
        /// Stores the correspondance between data item indices and container indices.
        /// Note: the first integer of the KeyValuePair stores the data item indices,
        /// and the second integer stores the container indices.
        /// </summary>
        private KeyValuePair<object, IControl>[] dataItemContainerMap;

        /// <summary>
        /// Initializes the ItemsControlExtender instance.
        /// </summary>
        /// <param name="itemsControl">IItemsControl instance to extend.</param>
        public ItemsControlExtender(IItemsControl itemsControl)
            : base(itemsControl)
        {
            this.extendedItemsControl = itemsControl;
        }

        /// <summary>
        /// Gets or sets the data source for items.
        /// This property may raise an 'ItemsSource' change notification.
        /// </summary>
        /// <remarks>
        /// DO NOT access this property directly from outside the extending control.
        /// You should ALWAYS access the ItemsSource property of the extending control
        /// in case the extending control has it's own implementation other than calling
        /// this ItemsSource property.
        /// Eg.
        /// <code>
        /// class MyControl : IItemsControl
        /// {
        ///     private ItemsControlExtender itemsControlExtender = null;
        ///
        ///     public ItemsControlExtender ItemsControlExtender
        ///     {
        ///         get { return this.itemsControlExtender; }
        ///     }
        ///
        ///     public IEnumerable ItemsSource
        ///     {
        ///         get { return this.itemsControlExtender.ItemsSource; }
        ///         set
        ///         {
        ///             this.itemsControlExtender.ItemsSource = value;
        ///             ++this.itemSourceSetCounter; // NOT EXECUTED!!!!
        ///         }
        ///     }
        /// }
        /// </code>
        /// If you do something like this:
        /// <code>
        /// var control = new MyControl();
        /// control.ItemsControlExtender.ItemsSource = new object();
        /// </code>
        /// The logic in MyControl.ItemsSource.set will not execute anymore.
        /// </remarks>
        public IEnumerable ItemsSource
        {
            get
            {
                return this.currentItemsSource;
            }

            set
            {
                if (this.SetValue(ref this.currentItemsSource, value))
                {
                    this.UpdateChildControls();
                }
            }
        }

        /// <summary>
        /// Gets or sets the type of item container to construct.
        /// </summary>
        /// <remarks>
        /// DO NOT access this property directly from outside the extending control.
        /// You should ALWAYS access the ItemContainerType property of the extending control
        /// in case the extending control has it's own implementation other than calling
        /// this ItemContainerType property.
        /// Eg.
        /// <code>
        /// class MyControl : IItemsControl
        /// {
        ///     private ItemsControlExtender itemsControlExtender = null;
        ///
        ///     public ItemsControlExtender ItemsControlExtender
        ///     {
        ///         get { return this.itemsControlExtender; }
        ///     }
        ///
        ///     public Type ItemContainerType
        ///     {
        ///         get { return this.itemsControlExtender.ItemContainerType; }
        ///         set
        ///         {
        ///             if (value.Equals(typeof(int)) == true) // NOT EXECUTED!!!!
        ///             {
        ///                 return;
        ///             }
        ///
        ///             this.ItemContainerType.ItemsSource = value;
        ///         }
        ///     }
        /// }
        /// </code>
        /// If you do something like this:
        /// <code>
        /// var control = new MyControl();
        /// control.ItemsControlExtender.ItemContainerType = typeof(int);
        /// </code>
        /// The logic in MyControl.ItemContainerType.set will not execute.
        /// </remarks>
        public Type ItemContainerType
        {
            get
            {
                return this.itemContainerType;
            }

            set
            {
                if (this.SetValue(ref this.itemContainerType, value))
                {
                    this.UpdateChildControls();
                }
            }
        }

        /// <summary>
        /// Gets or sets the item container selector.
        /// </summary>
        /// <remarks>
        /// DO NOT access this property directly from outside the extending control.
        /// You should ALWAYS access the ItemContainerType property of the extending control
        /// in case the extending control has it's own implementation other than calling
        /// this ItemContainerType property.
        /// Eg.
        /// <code>
        /// class MyControl : IItemsControl
        /// {
        ///     private ItemsControlExtender itemsControlExtender = null;
        ///
        ///     public ItemsControlExtender ItemsControlExtender
        ///     {
        ///         get { return this.itemsControlExtender; }
        ///     }
        ///
        ///     public IItemContainerSelector ItemContainerSelector
        ///     {
        ///         get { return this.itemsControlExtender.ItemContainerSelector; }
        ///         set
        ///         {
        ///             if (value.GetType().Equals(typeof(SelectorTypeIDontWant))) == true) // NOT EXECUTED!!!!
        ///             {
        ///                 return;
        ///             }
        ///
        ///             this.ItemContainerType.ItemContainerSelector = value;
        ///         }
        ///     }
        /// }
        /// </code>
        /// If you do something like this:
        /// <code>
        /// var control = new MyControl();
        /// control.ItemsControlExtender.ItemContainerSelector = new SelectorTypeIDontWant();
        /// </code>
        /// The logic in MyControl.SelectorTypeIDontWant.set will not execute.
        /// </remarks>
        public IItemContainerSelector ItemContainerSelector
        {
            get
            {
                return this.itemContainerSelector;
            }

            set
            {
                if (this.SetValue(ref this.itemContainerSelector, value))
                {
                    this.UpdateChildControls();
                }
            }
        }

        /// <summary>
        /// Throws an exception if a control is added illegally.
        /// </summary>
        protected internal void ThrowIfIllegalAdd()
        {
            if (this.currentItemsSource != null && this.isAddingFromItemsSource == false)
            {
                var msg = "Impossible to add child controls manually when ItemsSource is used";
                throw new InvalidOperationException(msg);
            }
        }

        /// <summary>
        /// Retrieve the data item corresponding to the container.
        /// </summary>
        /// <param name="container">The container to look for its data item.</param>
        /// <returns>Returns the data item, or null if not found.</returns>
        protected internal object GetDataItemFromContainer(IControl container)
        {
            if (container == null)
            {
                return null;
            }

            int index = this.dataItemContainerMap.FindIndex(kv => kv.Value == container);

            if (index == -1)
            {
                return null;
            }

            return this.dataItemContainerMap[index].Key;
        }

        /// <summary>
        /// Retrieve the container corresponding to the data item.
        /// </summary>
        /// <param name="dataItem">The data item to look for its data.</param>
        /// <returns>Returns the container, or null if not found.</returns>
        protected internal IControl GetContainerFromDataItem(object dataItem)
        {
            if (dataItem == null)
            {
                return null;
            }

            int index = this.dataItemContainerMap.FindIndex(kv => kv.Key == dataItem);

            if (index == -1)
            {
                return null;
            }

            return this.dataItemContainerMap[index].Value;
        }

        /// <summary>
        /// Clear the currently added controls to the extended control.
        /// </summary>
        /// <param name="preventFlickering">Tells whether flickering prevention
        /// code should be run or not.</param>
        private void ClearControls(bool preventFlickering)
        {
            IDisposable suspender = null;

            if (preventFlickering)
            {
                if (this.extendedItemsControl is Control)
                {
                    suspender = new DrawingSuspender((Control)this.extendedItemsControl);
                }
            }

            try
            {
                this.extendedItemsControl.Children.Clear();
            }
            finally
            {
                if (suspender != null)
                {
                    suspender.Dispose();
                }
            }
        }

        /// <summary>
        /// Clears and recreate the child items.
        /// </summary>
        private void UpdateChildControls()
        {
            if (this.currentItemsSource == null)
            {
                this.ClearControls(true);
                return;
            }

            var ncc = this.previousItemsSource as INotifyCollectionChanged;
            if (ncc != null)
            {
                ncc.CollectionChanged -= this.OnItemsControlExtenderCollectionChanged;
            }

            ncc = this.currentItemsSource as INotifyCollectionChanged;

            if (ncc != null)
            {
                ncc.CollectionChanged += this.OnItemsControlExtenderCollectionChanged;
            }

            this.ProduceAndAddItemContainers(this.currentItemsSource);

            this.previousItemsSource = this.currentItemsSource;
        }

        /// <summary>
        /// Produces an array of containers based on a sequence of data items and
        /// add them to the current children list.
        /// </summary>
        /// <param name="allItems">The sequence of all data items.</param>
        private void ProduceAndAddItemContainers(IEnumerable allItems)
        {
            if (allItems == null)
            {
                this.ClearControls(false);
                return;
            }

            this.isAddingFromItemsSource = true;

            // Always clear the child nodes then add the generated ones for UITreeNodes.
            bool doMinimumModification = !(this.extendedItemsControl is UITreeNode);

            var itemContainers = this.UpdateControlCollection(allItems);

            this.extendedItemsControl.SetChildControls(
                itemContainers,
                doMinimumModification);

            this.isAddingFromItemsSource = false;
        }

        /// <summary>
        /// Creates an array of IControl, whether the controls corresponding to input items,
        /// or with all the controls reordered.
        /// </summary>
        /// <param name="allItems">Input items to create container for.</param>
        /// <returns>Returns an array of container controls, partial or full.</returns>
        private IEnumerable<IControl> UpdateControlCollection(IEnumerable allItems)
        {
            this.dataItemContainerMap = allItems
                .Cast<object>()
                .Select(dataItem => new KeyValuePair<object, IControl>(
                    dataItem,
                    this.ProduceItemContainer(dataItem)))
                .OrderBy(kv => kv.Value, new InternalControlComparer(this.extendedItemsControl))
                .ToArray();

            return this.dataItemContainerMap
                .Select(kv => kv.Value)
                .Where(c => c != null);
        }

        /// <summary>
        /// Produces a container control for a given data item.
        /// </summary>
        /// <param name="dataItem">The data item.</param>
        /// <returns>Returns a control container. It can return null.</returns>
        private IControl ProduceItemContainer(object dataItem)
        {
            IControl itemContainer;
            try
            {
                if (this.ItemContainerType != null)
                {
                    itemContainer = Activator.CreateInstance(this.ItemContainerType) as IControl;
                }
                else if (this.ItemContainerSelector != null)
                {
                    itemContainer = this.ItemContainerSelector.SelectItemContainer(dataItem);
                }
                else
                {
                    return null;
                }
            }
            catch
            {
                return null;
            }

            if (itemContainer != null)
            {
                itemContainer.DataContext = dataItem;

                // TreeNodeだったら、ここでDataContextに関わる初期化もやる.
                if (itemContainer is UITreeNode)
                {
                    ((UITreeNode)itemContainer).OnInitialized();
                }
            }

            return itemContainer;
        }

        /// <summary>
        /// Called when the ItemSource is of type INotifyCollectionChanged and changed.
        /// </summary>
        /// <param name="sender">Caller of the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnItemsControlExtenderCollectionChanged(
            object sender,
            NotifyCollectionChangedEventArgs e)
        {
            this.UpdateChildControls();
        }

        /// <summary>
        /// Delegates IControl instances comparision to the IItemsControl instance.
        /// </summary>
        private class InternalControlComparer : IComparer<IControl>
        {
            /// <summary>
            /// Stores the IItemsControl instance.
            /// </summary>
            private IItemsControl itemsControl;

            /// <summary>
            /// Initializes the InternalControlComparer instance.
            /// </summary>
            /// <param name="itemsControl">The IItemsControl instance
            /// that manages the comparision.</param>
            public InternalControlComparer(IItemsControl itemsControl)
            {
                // garanted itemsControl cannot be null, so no need to check
                this.itemsControl = itemsControl;
            }

            /// <summary>
            /// Compares two IControl instances.
            /// </summary>
            /// <param name="x">First control.</param>
            /// <param name="y">Second control.</param>
            /// <returns>Returns 0 if equal, 1 or -1 if different to define order.</returns>
            public int Compare(IControl x, IControl y)
            {
                if (x == null || y == null)
                {
                    return 0;
                }

                return this.itemsControl.CompareControls(x, y);
            }
        }
    }
}
