﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Reflection.Emit;
using System.Runtime.CompilerServices;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.EventArguments;

namespace EffectMaker.UIControls.Extenders
{
    /// <summary>
    /// This class extends the capabilities of an ILogicalTreeElement.
    /// </summary>
    public class ExtenderBase
    {
        /// <summary>
        /// Stores auto generated dynamic method, per type.
        /// </summary>
        private static readonly Dictionary<Type, DynamicMethod> DynamicMethodCache =
            new Dictionary<Type, DynamicMethod>();

        /// <summary>
        /// Represent the dynamically generated OnPropertyChanged method.
        /// </summary>
        private PropertyChangedHandler propertyChanged;

        /// <summary>
        /// Stores the extended object instance.
        /// </summary>
        private object extendedObject;

        /// <summary>
        /// Initializes the ControlExtender.
        /// </summary>
        /// <param name="extendedObject">Object to extend the capabilitites.</param>
        public ExtenderBase(object extendedObject)
        {
            if (extendedObject == null)
            {
                throw new ArgumentNullException("extendedObject");
            }

            this.SkipPropertyChangedEvent = false;

            this.extendedObject = extendedObject;

            this.propertyChanged = this.GeneratePropertyChangedMethod();
        }

        /// <summary>
        /// Represent a call to a local PropertyChanged event.
        /// </summary>
        /// <param name="e">PropertyChanged event argument.</param>
        private delegate void PropertyChangedHandler(PropertyChangedEventArgs e);

        /// <summary>
        /// Get or set the flag indicating whether to skip PropertyChanged event.
        /// </summary>
        public bool SkipPropertyChangedEvent { get; set; }

        /// <summary>
        /// Sets the value of an internal field and provide automatic
        /// support for value changed propagation.
        /// Equality comparision is based on the Equals method of the instance,
        /// either the IEquatable(T).Equals of the base Equals methods.
        /// </summary>
        /// <typeparam name="T">The type of the field to set.</typeparam>
        /// <param name="field">The internal field member to set.</param>
        /// <param name="value">The given value to attribute to the field.</param>
        /// <param name="updateType">The type of update performed.</param>
        /// <param name="propertyName">The name of the current property to be set.</param>
        /// <returns>Returns true if the value changed, false otherwise.</returns>
        public bool SetValue<T>(
            ref T field,
            T value,
            BindingUpdateType updateType = BindingUpdateType.PropertyChanged,
            [CallerMemberName]string propertyName = null)
        {
            if (EqualityComparer<T>.Default.Equals(field, value))
            {
                return false;
            }

            if (field is ISettable && value is ISettable)
            {
                (field as ISettable).Set(value);
            }
            else
            {
                field = value;
            }

            this.NotifyPropertyChanged(updateType, propertyName);

            return true;
        }

        /// <summary>
        /// Raises the the PropertyChanged event from within the extended control.
        /// </summary>
        /// <param name="updateType">The type of update performed.</param>
        /// <param name="propertyName">The name of the property that changed.</param>
        public void NotifyPropertyChanged(
            BindingUpdateType updateType = BindingUpdateType.PropertyChanged,
            [CallerMemberName]string propertyName = null)
        {
            if (this.SkipPropertyChangedEvent == false)
            {
                this.propertyChanged(new PropertyChangedExEventArgs(propertyName, updateType));
            }
        }

        /// <summary>
        /// Dynamically generates a method bound to an external instance
        /// that can directly raise the PropertyChanged event from within it.
        /// </summary>
        /// <returns>Returns the handler to a compiled method.</returns>
        private PropertyChangedHandler GeneratePropertyChangedMethod()
        {
            var type = this.extendedObject.GetType();

            DynamicMethod dynamicMethod;

            if (DynamicMethodCache.TryGetValue(type, out dynamicMethod) == false)
            {
                dynamicMethod = this.GeneratePropertyChangedDynamicMethod(type);
                DynamicMethodCache.Add(type, dynamicMethod);
            }

            return (PropertyChangedHandler)dynamicMethod.CreateDelegate(
                typeof(PropertyChangedHandler),
                this.extendedObject);
        }

        /// <summary>
        /// Generates a dynamic method for a given external type that
        /// can directly raise the PropertyChanged event from within it.
        /// </summary>
        /// <param name="type">The type for which to generate the method.</param>
        /// <returns>Returns a dynamic method instance.</returns>
        private DynamicMethod GeneratePropertyChangedDynamicMethod(Type type)
        {
            // generate a unique name
            var name = "OnPropertyChangedAutoGenerated_" + Guid.NewGuid().ToString("N");

            // create a dynamic method
            var dynamicMethod = new DynamicMethod(
                name,
                typeof(void),
                new[] { type, typeof(PropertyChangedEventArgs) },
                type,
                true);

            var propertyChangedField = this.GetPropertyChangedField(type, "PropertyChanged");

            var invokeMethod = typeof(PropertyChangedEventHandler).GetMethod("Invoke");

            var g = dynamicMethod.GetILGenerator();

            // stores the PropertyChanged field
            var loc0 = g.DeclareLocal(typeof(PropertyChangingEventHandler));
            var label = g.DefineLabel();

            //// -----------------------------------------

            // load argument 0 (this) on the evaluation stack
            g.Emit(OpCodes.Ldarg_0);

            // load the PropertyChanged field on the evaluation stack
            g.Emit(OpCodes.Ldfld, propertyChangedField);

            // store the PropertyChanged field into local variable 0
            // C#> PropertyChangedEventHandler handler = this.PropertyChanged;
            g.Emit(OpCodes.Stloc_0);

            //// -----------------------------------------

            // load PropertyChanged field on the evaluation stack
            g.Emit(OpCodes.Ldloc_0);

            // load null on the evaluation stack
            g.Emit(OpCodes.Ldnull);

            // compare the two elements on the evaluation stack and
            // load the result on the evaluation stack
            // C#> if (handler != null)
            g.Emit(OpCodes.Ceq);

            //// -----------------------------------------

            // jump to lable is value on the evaluation stack is true
            g.Emit(OpCodes.Brtrue_S, label);

            //// -----------------------------------------

            // load PropertyChanged field on the evaluation stack
            g.Emit(OpCodes.Ldloc_0);

            // load argument 0 (this) on the evaluation stack (sender parameter)
            g.Emit(OpCodes.Ldarg_0);

            // load PropertyChangedEventArgs argument on the evaluation stack (e parameter)
            g.Emit(OpCodes.Ldarg_1);

            // call Invoke method of the PropertyChangedEventHandler delegate
            // C#> handler(this, e);
            g.Emit(OpCodes.Callvirt, invokeMethod);

            //// -----------------------------------------

            // mark label for the above declared jump instruction
            g.MarkLabel(label);

            // return
            g.Emit(OpCodes.Ret);

            return dynamicMethod;
        }

        /// <summary>
        /// Look for a specific field on a type.
        /// It also looks in the base types.
        /// </summary>
        /// <param name="type">The specific type in which to look for the field.</param>
        /// <param name="fieldName">The name of the field to look for.</param>
        /// <returns>Returns the FieldInfo instance if found, null otherwise.</returns>
        private FieldInfo GetPropertyChangedField(Type type, string fieldName)
        {
            do
            {
                var field = type.GetField(
                    "PropertyChanged",
                    BindingFlags.Instance | BindingFlags.NonPublic);

                if (field != null && field.FieldType == typeof(PropertyChangedEventHandler))
                {
                    return field;
                }

                type = type.BaseType;
            }
            while (type != typeof(object));

            return null;
        }
    }
}
