﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Debug
{
    /// <summary>
    /// An extended form that allow selection of an opened form.
    /// </summary>
    public partial class OpenedWindowsSelectorForm : Form
    {
        /// <summary>
        /// Initializes the OpenedWindowsSelectorForm instance.
        /// </summary>
        public OpenedWindowsSelectorForm()
        {
            this.InitializeComponent();

            this.RefreshFormList();
        }

        /// <summary>
        /// Gets the selected form.
        /// </summary>
        public Form SelectedForm { get; private set; }

        /// <summary>
        /// Called when the Refresh button is clicked.
        /// </summary>
        /// <param name="sender">Event called.</param>
        /// <param name="e">Event argument.</param>
        private void OnRefreshButtonClick(object sender, EventArgs e)
        {
            this.RefreshFormList();
        }

        /// <summary>
        /// Clears and re-populates the list of opened forms.
        /// </summary>
        private void RefreshFormList()
        {
            this.lstForms.Items.Clear();

            this.lstForms.Items.AddRange(Application.OpenForms
                .Cast<Form>()
                .Select(f => new FormListViewItem(f))
                .ToArray());
        }

        /// <summary>
        /// Called when the Accpt button is clicked.
        /// </summary>
        /// <param name="sender">Event called.</param>
        /// <param name="e">Event argument.</param>
        private void OnAcceptButtonClick(object sender, EventArgs e)
        {
            if (this.SetupSelectedForm())
            {
                this.DialogResult = DialogResult.OK;
                this.Close();
            }
        }

        /// <summary>
        /// Called when the list is double clicked with mouse.
        /// </summary>
        /// <param name="sender">Event called.</param>
        /// <param name="e">Event argument.</param>
        private void OnFormsListMouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (this.SetupSelectedForm())
            {
                this.DialogResult = DialogResult.OK;
                this.Close();
            }
        }

        /// <summary>
        /// Called when the Enter key is pressed on the forms list.
        /// </summary>
        /// <param name="sender">Event called.</param>
        /// <param name="e">Event argument.</param>
        private void OnFormsListKeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter && e.Modifiers == Keys.None)
            {
                if (this.SetupSelectedForm())
                {
                    this.DialogResult = DialogResult.OK;
                    this.Close();
                }
            }
        }

        /// <summary>
        /// Retrieve the selected form from the UI and set the SelectedForm property.
        /// </summary>
        /// <returns>Returns true if the selected form could be retrieved,
        /// false otherwise.</returns>
        private bool SetupSelectedForm()
        {
            if (this.lstForms.SelectedItems.Count == 0)
            {
                return false;
            }

            var item = this.lstForms.SelectedItems[0] as FormListViewItem;
            if (item == null)
            {
                return false;
            }

            this.SelectedForm = item.Form;

            return true;
        }

        /// <summary>
        /// Called when the selection of the form list changes.
        /// </summary>
        /// <param name="sender">Event called.</param>
        /// <param name="e">Event argument.</param>
        private void OnFormsListSelectedIndexChanged(object sender, EventArgs e)
        {
            this.btnAccept.Enabled = this.lstForms.SelectedItems.Count == 1;
        }
    }
}
