﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Reflection;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.ValueConverters;

namespace EffectMaker.UIControls.DataBinding
{
    /// <summary>
    /// The type of update performed by a binding.
    /// </summary>
    public enum BindingUpdateType
    {
        /// <summary>
        /// When the value of a property changes.
        /// </summary>
        PropertyChanged,

        /// <summary>
        /// When the value of a property is validated (LostFocus, SliderReleased, etc...)
        /// </summary>
        Validation,

        /// <summary>
        /// When the UpdateSource method of the Binder is directly called.
        /// </summary>
        Explicit,
    }

    /// <summary>
    /// The mode (direction) of the binding.
    /// </summary>
    public enum BindingMode
    {
        /// <summary>
        /// Binding update is performed only once.
        /// </summary>
        OneTime,

        /// <summary>
        /// The binding update is performed only to update the control using the data source.
        /// </summary>
        OneWay,

        /// <summary>
        /// The binding update is performed to update control and data source back and forth.
        /// </summary>
        TwoWay,

        /// <summary>
        /// The binding update is performed only to update the data source using the control.
        /// </summary>
        OneWayToSource,
    }

    /// <summary>
    /// Represents a data binding by storing its required metadata.
    /// </summary>
    public class Binding : System.Windows.Markup.MarkupExtension
    {
        /// <summary>
        /// Initializes a Binding instance.
        /// </summary>
        public Binding()
        {
            this.Mode = BindingMode.TwoWay;
            this.IssueCommand = false;
            this.UpdateSourceTrigger = BindingUpdateType.PropertyChanged;
        }

        /// <summary>
        /// Initializes a Binding instance.
        /// </summary>
        /// <param name="path">The name of the property of the data source.</param>
        public Binding(string path)
            : this()
        {
            this.Path = path;
        }

        /// <summary>
        /// Gets the path that define the property of the data source used for binding.
        /// </summary>
        public string Path { get; private set; }

        /// <summary>
        /// Gets or sets the matching update type.
        /// </summary>
        public BindingUpdateType UpdateSourceTrigger { get; set; }

        /// <summary>
        /// Gets or sets the binding mode.
        /// </summary>
        public BindingMode Mode { get; set; }

        /// <summary>
        /// Gets or sets the flag indicating whether to issue command
        /// when updating value to the data source.
        /// </summary>
        public bool IssueCommand { get; set; }

        /// <summary>
        /// ダイナミックではないメンバーにバインディングするかどうかを取得または設定します。
        /// </summary>
        public bool SetNonDynamicMember { get; set; }

        /// <summary>
        /// Gets or sets a value converter to customize the data processing in the Binder.
        /// </summary>
        public IValueConverter ValueConverter { get; set; }

        /// <summary>
        /// Gets or sets a custom parameter provided to
        /// the value converter when data is processed.
        /// </summary>
        public object ConverterParameter { get; set; }

        /// <summary>
        /// Produces the Binding.
        /// </summary>
        /// <param name="serviceProvider">An instance of a IServiceProvider
        /// provided by the XAML engine.</param>
        /// <returns>Returns this instance.</returns>
        public override object ProvideValue(IServiceProvider serviceProvider)
        {
            return this;
        }

        /// <summary>
        /// Creates a Binder from this Binding instance.
        /// </summary>
        /// <param name="control">The control the Binder is to be bound to.</param>
        /// <param name="controlPropertyInfo">The property of the control to be bound to.</param>
        /// <returns>Returns a Binder instance.</returns>
        public Binder CreateBinder(ILogicalTreeElement control, PropertyInfo controlPropertyInfo)
        {
            return new Binder(control, controlPropertyInfo, this.Path)
            {
                UpdateType = this.UpdateSourceTrigger,
                ValueConverter = this.ValueConverter,
                ConverterParameter = this.ConverterParameter,
                IssueCommand = this.IssueCommand,
                SetNonDynamicMember = this.SetNonDynamicMember,
            };
        }
    }
}
