﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Dynamic;
using System.Reflection;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.EventArguments;
using EffectMaker.UIControls.ValueConverters;

namespace EffectMaker.UIControls.DataBinding
{
    /// <summary>
    /// The Binder keeps an ILogicalTreeElement and a data source synchronized.
    /// </summary>
    public class Binder : IDisposable
    {
        /// <summary>
        /// Stores the property of the element that is bound.
        /// Cannot be null.
        /// </summary>
        private PropertyInfo elementPropertyInfo;

        /// <summary>
        /// Stores the property of the data source that is bound.
        /// Can be null.
        /// </summary>
        private PropertyInfo dataSourcePropertyInfo;

        /// <summary>
        /// Tells whether the update phases must be performed or skipped.
        /// </summary>
        private bool canUpdate;

        /// <summary>
        /// Tells whether to try to look for a property through dynamic resolution.
        /// </summary>
        private bool tryDynamic;

        /// <summary>
        /// Tells whether an update is in progress to avoid
        /// useless over updates and stack overflows.
        /// </summary>
        private bool interlocked;

        /// <summary>
        /// Stores the value of the previous data source in order
        /// to unregister events when data source changes.
        /// </summary>
        private INotifyPropertyChanged oldDataSource;

        /// <summary>
        /// Initializes a Binder instance.
        /// </summary>
        /// <param name="element">The element to bind to the data source.</param>
        /// <param name="elementProperty">The property of the element to be bound.</param>
        /// <param name="dataSourcePropertyName">The name of the bound property on the data source.
        /// (It can null to use the data source directly)</param>
        public Binder(
            ILogicalTreeElement element,
            PropertyInfo elementProperty,
            string dataSourcePropertyName)
        {
            this.Initialize(element, elementProperty, dataSourcePropertyName);
        }

        /// <summary>
        /// Initializes a Binder instance.
        /// </summary>
        /// <param name="element">The element to bind to the data source.</param>
        /// <param name="elementPropertyName">The name of the property
        /// of the element to be bound.</param>
        /// <param name="dataSourcePropertyName">The name of the bound property on the data source.
        /// (It can null to use the data source directly)</param>
        public Binder(
            ILogicalTreeElement element,
            string elementPropertyName,
            string dataSourcePropertyName)
        {
            PropertyInfo prop = null;

            if (element != null && string.IsNullOrWhiteSpace(elementPropertyName) == false)
            {
                prop = element.GetType().GetProperty(
                    elementPropertyName,
                    BindingFlags.Public | BindingFlags.Instance);
            }

            this.Initialize(element, prop, dataSourcePropertyName);
        }

        /// <summary>
        /// For Visual Studio use only.
        /// </summary>
        private Binder()
        {
        }

        /// <summary>
        /// Gets the element that the binder is attached to.
        /// </summary>
        public ILogicalTreeElement Element { get; private set; }

        /// <summary>
        /// Gets the name of the property of the element that is bound to the data source.
        /// </summary>
        public string ElementPropertyName { get; private set; }

        /// <summary>
        /// Get the name of the property of the data source that is bound to the element.
        /// </summary>
        public string DataSourcePropertyName { get; private set; }

        /// <summary>
        /// Gets or sets the desired matching update type.
        /// </summary>
        public BindingUpdateType UpdateType { get; set; }

        /// <summary>
        /// Gets or sets the binding mode.
        /// </summary>
        public BindingMode Mode { get; set; }

        /// <summary>
        /// Gets or sets the flag indicating whether to issue command
        /// when updating value to the data source.
        /// </summary>
        public bool IssueCommand { get; set; }

        /// <summary>
        /// ダイナミックではないメンバーにバインディングするかどうかを取得または設定します。
        /// </summary>
        public bool SetNonDynamicMember { get; set; }

        /// <summary>
        /// Gets or sets a ValueConverter that can transfom
        /// the data when passing though the binder.
        /// </summary>
        public IValueConverter ValueConverter { get; set; }

        /// <summary>
        /// Gets or sets a parameter to be passed to
        /// the ValueConverter when processing data.
        /// </summary>
        public object ConverterParameter { get; set; }

        /// <summary>
        /// Gets whether the Binder is disposed or not.
        /// </summary>
        public bool IsDisposed { get; private set; }

        /// <summary>
        /// Updates the data source according to the element state.
        /// </summary>
        /// <returns>Returns true if an update happened, false otherwise.</returns>
        public bool UpdateElement()
        {
            return this.UpdateElement(false);
        }

        /// <summary>
        /// Updates the element according to the data source state.
        /// </summary>
        /// <returns>Returns true if an update happened, false otherwise.</returns>
        public bool UpdateDataSource()
        {
            return this.UpdateDataSourceInternal(BindingUpdateType.Explicit);
        }

        /// <summary>
        /// Disposes the current Binder instance.
        /// </summary>
        public void Dispose()
        {
            if (this.IsDisposed)
            {
                return;
            }

            this.IsDisposed = true;

            this.Element.PropertyChanged -= this.OnElementPropertyChanged;
            if (this.oldDataSource != null)
            {
                this.oldDataSource.PropertyChanged -= this.OnDataSourcePropertyChanged;
            }
        }

        /// <summary>
        /// Initializes a Binder instance.
        /// </summary>
        /// <param name="element">The element to bind to the data source.</param>
        /// <param name="elementPropertyInfo">The property of the element to be bound.</param>
        /// <param name="dataSourcePropertyName">The name of the bound property on the data source.
        /// (It can null to use the data source directly)</param>
        private void Initialize(
            ILogicalTreeElement element,
            PropertyInfo elementPropertyInfo,
            string dataSourcePropertyName)
        {
            if (element == null)
            {
                throw new ArgumentNullException("element");
            }

            if (elementPropertyInfo == null)
            {
                throw new ArgumentNullException("elementPropertyInfo");
            }

            this.IssueCommand = false;
            this.SetNonDynamicMember = false;

            this.Element = element;
            this.ElementPropertyName = elementPropertyInfo.Name;
            this.DataSourcePropertyName = dataSourcePropertyName;

            this.elementPropertyInfo = elementPropertyInfo;

            this.DetermineBindingMode();

            element.PropertyChanged += this.OnElementPropertyChanged;
        }

        /// <summary>
        /// Automatically determines the binding mode according to the properties accessibilitis.
        /// </summary>
        private void DetermineBindingMode()
        {
            if (this.elementPropertyInfo.GetSetMethod(false) == null)
            {
                if (this.DataSourcePropertyName == null)
                {
                    throw new Exception("Invalid binding"); // TODO: make it localizable
                }

                this.Mode = BindingMode.OneWayToSource;
            }
            else
            {
                //// the property has read and write access

                if (this.DataSourcePropertyName == null)
                {
                    this.Mode = BindingMode.OneWay;
                }
                else
                {
                    this.Mode = BindingMode.TwoWay;
                }
            }
        }

        /// <summary>
        /// Raised when a property changed on the bound element.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The argument of the event.</param>
        private void OnElementPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            this.CheckDisposed();

            if (e.PropertyName == "DataContext")
            {
                this.UpdateDataSourcePropertyInfo();

                if (this.Mode == BindingMode.OneWayToSource)
                {
                    this.UpdateDataSourceInternal(BindingUpdateType.PropertyChanged);
                }
                else if (this.UpdateElement(true))
                {
                    var dataSource = this.Element.DataContext as INotifyPropertyChanged;

                    if (object.Equals(this.oldDataSource, dataSource) == false)
                    {
                        if (this.oldDataSource != null)
                        {
                            this.oldDataSource.PropertyChanged -=
                                this.OnDataSourcePropertyChanged;
                        }

                        if (dataSource != null)
                        {
                            dataSource.PropertyChanged +=
                                this.OnDataSourcePropertyChanged;
                        }

                        this.oldDataSource = dataSource;
                    }
                }
            }
            else if (e.PropertyName == this.elementPropertyInfo.Name)
            {
                var updateType = BindingUpdateType.PropertyChanged;
                if (e is PropertyChangedExEventArgs)
                {
                    updateType = ((PropertyChangedExEventArgs)e).BindingUpdateType;
                }

                this.UpdateDataSourceInternal(updateType);
            }
        }

        /// <summary>
        /// Raised when a property changed on data source.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The argument of the event.</param>
        private void OnDataSourcePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            this.CheckDisposed();

            if (e.PropertyName == this.DataSourcePropertyName)
            {
                this.UpdateElement();
            }
        }

        /// <summary>
        /// Updates the data source according to the element state.
        /// </summary>
        /// <param name="fromInternal">Flag indicating if this is an internal call or simply a
        /// routing for the parameterless UpdateElement method.</param>
        /// <returns>Returns true if an update happened, false otherwise.</returns>
        private bool UpdateElement(bool fromInternal)
        {
            this.CheckDisposed();

            if (this.Mode == BindingMode.OneWayToSource)
            {
                return false;
            }

            if (fromInternal == false)
            {
                this.UpdateDataSourcePropertyInfo();
            }

            if (this.UpdateElementInternal() == false)
            {
                return false;
            }

            if (this.Mode == BindingMode.OneTime)
            {
                this.Dispose();

                if (this.Element is IBindable)
                {
                    ((IBindable)this.Element).Bindings.Remove(this);
                    if (fromInternal)
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// Updates the PropertyInfo representing the property on the data source.
        /// </summary>
        private void UpdateDataSourcePropertyInfo()
        {
            var dataSource = this.Element.DataContext;

            this.dataSourcePropertyInfo = null;
            this.tryDynamic = false;
            this.canUpdate = false;

            if (dataSource == null)
            {
                return;
            }

            this.canUpdate = true;

            if (this.DataSourcePropertyName == null)
            {
                this.dataSourcePropertyInfo = null;
                this.tryDynamic = false;
            }
            else
            {
                this.dataSourcePropertyInfo = dataSource.GetType().GetProperty(
                    this.DataSourcePropertyName,
                    BindingFlags.Public | BindingFlags.Instance);
                this.tryDynamic = true;
            }
        }

        /// <summary>
        /// Internal implementation for the UpdateElement method.
        /// See UpdateElement method for more details.
        /// </summary>
        /// <returns>Returns true if an update happened, false otherwise.</returns>
        private bool UpdateElementInternal()
        {
            if (this.canUpdate == false || this.interlocked)
            {
                return false;
            }

            this.interlocked = true;
            using (new AnonymousDisposable(() => this.interlocked = false))
            {
                var value = this.Element.DataContext;

                if (value == null)
                {
                    return false;
                }

                var valueType = value.GetType();

                if (this.dataSourcePropertyInfo != null)
                {
                    value = this.dataSourcePropertyInfo.GetValue(value, null);
                    valueType = value == null ? this.dataSourcePropertyInfo.PropertyType : value.GetType();
                }
                else
                {
                    if (this.tryDynamic)
                    {
                        if (this.TryGetDynamicProperty(value, ref value) == false)
                        {
                            return false;
                        }

                        valueType = value == null ? typeof(object) : value.GetType();
                    }
                }

                if (this.ValueConverter != null)
                {
                    value = this.ValueConverter.Convert(
                        value,
                        this.elementPropertyInfo.PropertyType,
                        this.ConverterParameter);

                    valueType = value == null ? typeof(object) : value.GetType();
                }

                var canAssign = TypeConversionUtility.TryConvert(
                    valueType,
                    this.elementPropertyInfo.PropertyType,
                    ref value);

                if (canAssign == false &&
                    value == null &&
                    this.elementPropertyInfo.PropertyType.IsValueType == false)
                {
                    // null should be able to assign to non-value types.
                    canAssign = true;
                }

                if (canAssign == true)
                {
                    try
                    {
                        if (this.elementPropertyInfo.Name == "DataContext")
                        {
                            this.Element.LogicalTreeElementExtender
                                .SetDataContextThroughBinding(value);
                        }
                        else
                        {
                            this.elementPropertyInfo.SetValue(this.Element, value, null);
                        }
                    }
                    finally
                    {
                    }

                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// Internal implementation for the UpdateDataSource method.
        /// See UpdateDataSource method for more details.
        /// </summary>
        /// <param name="updateType">The type of update to perform.</param>
        /// <returns>Returns true if an update happened, false otherwise.</returns>
        private bool UpdateDataSourceInternal(BindingUpdateType updateType)
        {
            this.CheckDisposed();

            if (this.canUpdate == false || this.interlocked || this.Mode == BindingMode.OneWay)
            {
                return false;
            }

            this.interlocked = true;
            using (new AnonymousDisposable(() => this.interlocked = false))
            {
                if (this.UpdateType != updateType)
                {
                    return false;
                }

                var dataSource = this.Element.DataContext;

                if (dataSource == null)
                {
                    return false;
                }

                Type dataSourceType = null;

                if (this.dataSourcePropertyInfo == null)
                {
                    // cannot update DataContext
                    if (this.tryDynamic == false)
                    {
                        return false;
                    }

                    // trick to get the property type
                    this.TryGetDynamicPropertyType(dataSource, out dataSourceType);
                }
                else
                {
                    dataSourceType = this.dataSourcePropertyInfo.PropertyType;
                }

                var value = this.elementPropertyInfo.GetValue(this.Element, null);
                var valueType = value == null ? this.elementPropertyInfo.PropertyType : value.GetType();
                if (this.ValueConverter != null)
                {
                    value = this.ValueConverter.ConvertBack(
                        value,
                        dataSourceType,
                        this.ConverterParameter);

                    valueType = value == null ? typeof(object) : value.GetType();
                }

                var canAssign = TypeConversionUtility.TryConvert(valueType, dataSourceType, ref value);
                if (canAssign == false &&
                    value == null &&
                    dataSourceType.IsValueType == false)
                {
                    // null should be able to assign to non-value types.
                    canAssign = true;
                }

                if (canAssign)
                {
                    try
                    {
                        if (this.IssueCommand == true ||
                            this.SetNonDynamicMember == true ||
                            this.dataSourcePropertyInfo == null)
                        {
                            //// here tryDynamic is true, no need to check it

                            if (this.TrySetDynamicProperty(dataSource, value) == false)
                            {
                                return false;
                            }
                        }
                        else
                        {
                            this.dataSourcePropertyInfo.SetValue(dataSource, value, null);
                        }
                    }
                    finally
                    {
                    }

                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// Tries to determine the type of the property named by the
        /// member DataSourcePropertyName on a dynamic object.
        /// </summary>
        /// <param name="instance">Instance of the dynamic object.</param>
        /// <param name="propertyType">The determined property type.</param>
        /// <returns>Returns true if the type could be determined, false otherwise.</returns>
        private bool TryGetDynamicPropertyType(object instance, out Type propertyType)
        {
            propertyType = typeof(object);

            var dynamicObject = instance as DynamicObject;

            if (dynamicObject == null)
            {
                // property cannot be found (not dynamic)
                return false;
            }

            object readValue;
            if (dynamicObject.TryGetMember(
                new EffectMakerGetMemberBinder(this.DataSourcePropertyName),
                out readValue) == false)
            {
                // property cannot be found (property not found on dynamic)
                return false;
            }

            // value is null, impossible to determine type out of that
            if (readValue == null)
            {
                return false;
            }

            propertyType = readValue.GetType();

            return true;
        }

        /// <summary>
        /// Tries to get the value of the property named by the
        /// member DataSourcePropertyName on a dynamic object.
        /// </summary>
        /// <param name="instance">Instance of the dynamic object.</param>
        /// <param name="value">The retrieved value of the property.</param>
        /// <returns>Returns true if the value could be retrieved, false otherwise.</returns>
        private bool TryGetDynamicProperty(object instance, ref object value)
        {
            var dynamicObject = instance as DynamicObject;

            if (dynamicObject == null)
            {
                // property cannot be found (not dynamic)
                return false;
            }

            if (dynamicObject.TryGetMember(
                new EffectMakerGetMemberBinder(this.DataSourcePropertyName),
                out value) == false)
            {
                // property cannot be found (property not found on dynamic)
                return false;
            }

            return true;
        }

        /// <summary>
        /// Tries to set the value of the property named by the
        /// member DataSourcePropertyName on a dynamic object.
        /// </summary>
        /// <param name="instance">Instance of the dynamic object.</param>
        /// <param name="value">The value to set.</param>
        /// <returns>Returns true if the value could be set, false otherwise.</returns>
        private bool TrySetDynamicProperty(object instance, object value)
        {
            var dynamicObject = instance as DynamicObject;

            if (dynamicObject == null)
            {
                // property cannot be found (not dynamic)
                return false;
            }

            bool result = dynamicObject.TrySetMember(
                new EffectMakerSetMemberBinder(
                    this.DataSourcePropertyName,
                    this.SetNonDynamicMember || this.IssueCommand,
                    this.IssueCommand),
                value);

            if (result == false)
            {
                // property cannot be found (property not found on dynamic)
                return false;
            }

            return true;
        }

        /// <summary>
        /// This throws an ObjectDisposedException exception if the object is disposed.
        /// </summary>
        private void CheckDisposed()
        {
            if (this.IsDisposed)
            {
                throw new ObjectDisposedException(ToString());
            }
        }
    }
}
