﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Utility;
using EffectMaker.Foundation.Win32;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.EventArguments;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended TrackBar class.
    /// </summary>
    public class UITrackBar : TrackBar, IControl, ILayoutElement
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Mouse wheel edit confirm timer.
        /// </summary>
        private Timer wheelEditConfirmTimer = new Timer();

        /// <summary>
        /// Value changing flag.
        /// </summary>
        private bool valueChanging = false;

        /// <summary>
        /// Mouse wheel editing flag.
        /// </summary>
        private bool mouseWheelEditing = false;

        /// <summary>
        /// Mouse wheel edit enable flag.
        /// </summary>
        private bool mouseWheelEditEnabled = true;

        /// <summary>
        /// Stores the previous value when the track bar value was keeping changing.
        /// </summary>
        private int previousThrottledValue;

        /// <summary>
        /// Stores the previous value when the track bar value got validated.
        /// </summary>
        private int previousValidationValue;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UITrackBar()
        {
            this.AutoSize = true;
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);

            this.wheelEditConfirmTimer.Enabled = false;
            this.wheelEditConfirmTimer.Interval = 500;
            this.wheelEditConfirmTimer.Tick += this.OnWheelEditConfirmTimerTick;

            this.Disposed += this.OnDisposed;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// Gets or sets the value.
        /// The value is clamped to its minimum and maximum bounds if it exceeds.
        /// </summary>
        public new virtual int Value
        {
            get
            {
                return base.Value;
            }

            set
            {
                base.Value = MathUtility.Clamp(value, this.Minimum, this.Maximum);
            }
        }

        /// <summary>
        /// Gets or sets the IExecutable to be executed when the value changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// Gets or sets the custom paramter of the OnValueChangedExecutable.
        /// </summary>
        public object ValueChangedExecutableParameter { get; set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether mouse wheel can edit
        /// the track bar value or not.
        /// </summary>
        [DefaultValue(false)]
        [Description("Get or set the flag indicating whether mouse wheel can edit the track bar value or not.")]
        public bool IsMouseWheelEditEnabled
        {
            get { return this.mouseWheelEditEnabled; }
            set { this.mouseWheelEditEnabled = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// Gets a CreateParams that contains the required creation parameters
        /// when the handle to the control is created.
        /// </summary>
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams cp = base.CreateParams;
                cp.ExStyle |= 0x02000000; // WS_EX_COMPOSITED
                return cp;
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        ///  The overwritten WndProc.
        /// </summary>
        /// <param name="m">Windows message.</param>
        protected override void WndProc(ref Message m)
        {
            switch ((WM)m.Msg)
            {
                // Key down.
                case WM.WM_KEYDOWN:
                {
                    int increment = 0;
                    VK keyCode    = (VK)m.WParam;

                    // Compute the increment.
                    if (keyCode == VK.VK_UP ||
                        keyCode == VK.VK_RIGHT)
                    {
                        increment = this.SmallChange;
                    }
                    else if (keyCode == VK.VK_DOWN ||
                             keyCode == VK.VK_LEFT)
                    {
                        increment = -this.SmallChange;
                    }
                    else if (keyCode == VK.VK_PRIOR)
                    {
                        increment = this.LargeChange;
                    }
                    else if (keyCode == VK.VK_NEXT)
                    {
                        increment = -this.LargeChange;
                    }
                    else
                    {
                        break;
                    }

                    // Compute the value.
                    int value = this.Value + increment;

                    // Clamp the value into the range.
                    if (value > this.Maximum)
                    {
                        value = this.Maximum;
                    }
                    else if (value < this.Minimum)
                    {
                        value = this.Minimum;
                    }

                    // Set value.
                    if (value != this.Value)
                    {
                        this.valueChanging = true;
                        this.Value         = value;
                    }

                    return;
                }

                // Scroll event
                case WM.WM_REFLECT | WM.WM_HSCROLL:
                case WM.WM_REFLECT | WM.WM_VSCROLL:
                {
                    TB trackBarEvent = (TB)Utility.LOWORD(m.WParam);

                    if (trackBarEvent == TB.TB_ENDTRACK)
                    {
                        // Scroll finished.
                        if (this.valueChanging == true)
                        {
                            this.valueChanging = false;
                            this.OnValueChanged(null);
                        }
                    }
                    else
                    {
                        // Scrolling.
                        this.valueChanging = true;
                    }

                    break;
                }

                // Mouse wheel.
                case WM.WM_MOUSEWHEEL:
                {
                    if (this.mouseWheelEditEnabled == true)
                    {
                        int wheelDelta = Utility.SignedHIWORD(m.WParam);
                        Point mousePos = Utility.LParamToPoint(m.LParam);

                        this.OnMouseWheel(new MouseEventArgs(
                                                             MouseButtons.None,
                                                             0,
                                                             mousePos.X,
                                                             mousePos.Y,
                                                             wheelDelta));
                    }

                    return;
                }

                default:
                    break;
            }

            base.WndProc(ref m);
        }

        /// <summary>
        /// Dispose.
        /// </summary>
        /// <param name="disposing">If disposing equals true, dispose all managed.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                this.wheelEditConfirmTimer.Dispose();
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Handle MouseWheel event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            // Compute the increment.
            int increment = SmallChange * System.Math.Abs(e.Delta) / 120;
            if (e.Delta < 0)
            {
                increment *= -1;
            }
            else if (e.Delta == 0)
            {
                return;
            }

            // Set value.
            int value = this.Value + increment;

            // Clamp the value into the range.
            if (value > this.Maximum)
            {
                value = this.Maximum;
            }
            else if (value < this.Minimum)
            {
                value = this.Minimum;
            }

            // Set value.
            if (value != this.Value)
            {
                this.valueChanging = true;
                this.Value         = value;
            }

            this.wheelEditConfirmTimer.Stop();
            this.wheelEditConfirmTimer.Enabled = true;
            this.wheelEditConfirmTimer.Start();

            this.mouseWheelEditing = true;
        }

        /// <summary>
        /// Handle ValueChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnValueChanged(EventArgs e)
        {
            BindingUpdateType bindingUpdateType;

            if (e != null)
            {
                //// this is regular event, need to extend it

                e = new ValueChangedExEventArgs(
                    this.previousThrottledValue,
                    this.Value,
                    true,
                    this.ValueChangedExecutableParameter);

                this.previousThrottledValue = this.Value;

                //// with a regular event, the binding update type is PropertyChanged
                //// otherwise it means a validation, such as after mouse releasing the track bar
                bindingUpdateType = BindingUpdateType.PropertyChanged;
            }
            else
            {
                e = new ValueChangedExEventArgs(
                    this.previousValidationValue,
                    this.Value,
                    false,
                    this.ValueChangedExecutableParameter);

                this.previousValidationValue = this.Value;

                bindingUpdateType = BindingUpdateType.Validation;
            }

            // Trigger the events.
            base.OnValueChanged(e);

            this.controlExtender.NotifyPropertyChanged(bindingUpdateType, "Value");

            // run the OnValueChangedExecutable executable if available
            var executable = this.ValueChangedExecutable;
            if (executable != null && executable.CanExecute(e))
            {
                executable.Execute(e);
            }
        }

        /// <summary>
        /// Called when the UITrackBar is disposed.
        /// </summary>
        /// <param name="sender">This UITrackBar instance.</param>
        /// <param name="e">Event argument.</param>
        private void OnDisposed(object sender, EventArgs e)
        {
            this.wheelEditConfirmTimer.Stop();
            this.wheelEditConfirmTimer.Tick -= this.OnWheelEditConfirmTimerTick;
            this.wheelEditConfirmTimer.Dispose();
        }

        /// <summary>
        /// Handle Tick event for m_wheelEditConfirmTimer.
        /// There is no way to know if mouse wheel value editing is finished,
        /// so we setup a timer and wait for a period of time, if no mouse
        /// wheel is fired, treat the editing as finished to trigger the value
        /// change event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnWheelEditConfirmTimerTick(object sender, EventArgs e)
        {
            // Every time mouse wheel event is fired, the timer gets reset.
            // When this handler is called, indicating the mouse wheel event
            // hasn't been fired for a while, thus treat the value editing
            // as finished.
            if (this.mouseWheelEditing == true)
            {
                this.mouseWheelEditing = false;
                this.valueChanging     = false;

                // Trigger value changed event.
                this.OnValueChanged(null);
            }

            // Disable the timer until the next time mouse wheel event is fired.
            this.wheelEditConfirmTimer.Stop();
            this.wheelEditConfirmTimer.Enabled = false;
        }
    }
}
