﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Input;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended ToolStripSplitButton class.
    /// </summary>
    public class UIToolStripSplitButton : ToolStripSplitButton, IControl, IExecutionSource
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// The IExecutionSource extender instance.
        /// </summary>
        private ExecutionSourceExtender executionSourceExtender;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// trueならボタンを通常どおり描画する.
        /// falseならボタンをグレー表示にする.
        /// </summary>
        private bool drawEnabledButton;

        /// <summary>
        /// 左側のボタンの選択表示を描画する.
        /// </summary>
        private bool buttonSelectedImageEnabled = false;

        /// <summary>
        /// 右側のドロップダウンメニューボタンの選択表示を描画する.
        /// </summary>
        private bool dropDownButtonSelectedImageEnabled = false;

        /// <summary>
        /// The timer.
        /// </summary>
        private Stopwatch timer = new Stopwatch();

        /// <summary>
        /// Constructor.
        /// </summary>
        public UIToolStripSplitButton()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.executionSourceExtender = new ExecutionSourceExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
            this.DrawEnabledButton = true;
            this.EnableButtons = true;

            // ボタン上でマウスカーソルを動かした場合に実行される処理
            this.MouseMove += (s, e) =>
                {
                    Point mousePos = e.Location;
                    Size dropDownSize = this.DropDownButtonBounds.Size;
                    Point dropDownPoint = this.DropDownButtonBounds.Location;

                    if (mousePos.X - dropDownPoint.X < dropDownSize.Width && mousePos.X - dropDownPoint.X >= 0 &&
                        mousePos.Y - dropDownPoint.Y < dropDownSize.Height && mousePos.Y - dropDownPoint.Y >= 0)
                    {
                        this.buttonSelectedImageEnabled = false;
                        this.dropDownButtonSelectedImageEnabled = true;
                    }
                    else
                    {
                        this.buttonSelectedImageEnabled = true;
                        this.dropDownButtonSelectedImageEnabled = false;
                    }

                    this.Invalidate();
                };

            // マウスカーソルがボタンから離れた際に実行される処理
            this.MouseLeave += (s, e) =>
                {
                    this.buttonSelectedImageEnabled = false;
                    this.dropDownButtonSelectedImageEnabled = false;
                    this.Invalidate();
                };

            this.DropDownClosed += (s, e) => this.timer.Restart();
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Fired before Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler BeforeExecute
        {
            add { this.executionSourceExtender.BeforeExecute += value; }
            remove { this.executionSourceExtender.BeforeExecute -= value; }
        }

        /// <summary>
        /// Fired after Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler AfterExecute
        {
            add { this.executionSourceExtender.AfterExecute += value; }
            remove { this.executionSourceExtender.AfterExecute -= value; }
        }

        /// <summary>
        /// trueならボタンを通常どおり描画する.
        /// falseならボタンをグレー表示にする.
        /// </summary>
        public bool DrawEnabledButton
        {
            get
            {
                return this.drawEnabledButton;
            }

            set
            {
                this.drawEnabledButton = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// 画像を重ねて描画するか
        /// </summary>
        public bool OverlayImage { get; set; }

        /// <summary>
        /// 重ねて表示する画像
        /// </summary>
        public Bitmap AdditionalImage { get; set; }

        /// <summary>
        /// 左側のボタンが選択されているときの画像
        /// </summary>
        public Bitmap ButtonSelectedImage { get; set; }

        /// <summary>
        /// 左側のボタンではなく、右側のドロップボタンが選択されている時の画像
        /// </summary>
        public Bitmap ButtonNonSelectedImage { get; set; }

        /// <summary>
        /// 左側のボタンを押している時の画像.
        /// </summary>
        public Bitmap ButtonDownImage { get; set; }

        /// <summary>
        /// 右側のドロップダウンボタンが選択されているときの画像
        /// </summary>
        public Bitmap DropDownButtonSelectedImage { get; set; }

        /// <summary>
        /// 右側のドロップボタンではなく、左側のボタンが選択されている時の画像
        /// </summary>
        public Bitmap DropDownButtonNonSelectedImage { get; set; }

        /// <summary>
        /// 右側のドロップダウンボタンの標準画像
        /// </summary>
        public Bitmap DropDownButtonImage { get; set; }

        /// <summary>
        /// ドロップダウンメニューが表示されているときのボタン画像
        /// </summary>
        public Bitmap DropDownMenuOpenedImage { get; set; }

        /// <summary>
        /// 設定した画像でボタンを描画するか.
        /// </summary>
        public bool UseOriginalStyle { get; set; }

        /// <summary>
        /// 両ボタンを選択可能にするか
        /// </summary>
        public bool EnableButtons { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new NullIndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets or sets the executable to execute when clicked.
        /// </summary>
        public IExecutable Executable
        {
            get
            {
                return this.executionSourceExtender.Executable;
            }

            set
            {
                this.executionSourceExtender.Executable = value;
            }
        }

        /// <summary>
        /// Gets or sets a custom parameter to provide to the execution.
        /// </summary>
        public object ExecutionParameter
        {
            get
            {
                return this.executionSourceExtender.ExecutionParameter;
            }

            set
            {
                this.executionSourceExtender.ExecutionParameter = value;
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// ボタンのグレーオフ表示と画像の重ね書き対応のためにオーバーライト
        /// </summary>
        /// <param name="e">イベントデータを格納しているPaintEventArgs</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            if (this.UseOriginalStyle == true)
            {
                Graphics g = e.Graphics;

                // 描画する場所の左上隅のX座標とY座標
                int x = this.Margin.Left;
                int y = this.Margin.Top;

                ToolStrip toolStrip = base.Parent;
                if (toolStrip != null)
                {
                    x += toolStrip.GripMargin.Left;
                    y += toolStrip.GripMargin.Top;
                }

                this.DrawDropDownButton(g, x, y);
                this.DrawButton(g, x, y);
            }
            else
            {
                base.OnPaint(e);
            }
        }

        /// <summary>
        /// マウスボタンが押された時の処理
        /// </summary>
        /// <param name="e">マウスイベント</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            // 前回のドロップダウンリストクローズ時から直近のイベントは破棄する
            if (this.timer.IsRunning && this.timer.ElapsedMilliseconds < 100)
            {
                return;
            }

            if (this.EnableButtons == true)
            {
                base.OnMouseDown(e);
            }
        }

        /// <summary>
        /// クリックされた時の処理
        /// </summary>
        /// <param name="e">イベント</param>
        protected override void OnButtonClick(EventArgs e)
        {
            if (this.EnableButtons == true)
            {
                base.OnButtonClick(e);
            }
        }

        /// <summary>
        /// imageをNullCheckして描画する
        /// </summary>
        /// <param name="g">Graphicsインスタンス</param>
        /// <param name="image">Imageインスタンス</param>
        /// <param name="x">左上X座標</param>
        /// <param name="y">左上Y座標</param>
        private void DrawImageWithNullCheck(Graphics g, Image image, float x, float y)
        {
            if (image != null)
            {
                g.DrawImage(image, x, y, image.Width, image.Height);
            }
        }

        /// <summary>
        /// imageをNullCheckしてDisabledで描画する
        /// </summary>
        /// <param name="g">Graphicsインスタンス</param>
        /// <param name="image">Imageインスタンス</param>
        /// <param name="x">左上X座標</param>
        /// <param name="y">左上Y座標</param>
        private void DrawImageDisabledWithNullCheck(Graphics g, Image image, int x, int y)
        {
            if (image != null)
            {
                ControlPaint.DrawImageDisabled(g, image, x, y, this.BackColor);
            }
        }

        /// <summary>
        /// ドロップダウンボタンの描画
        /// </summary>
        /// <param name="g">Graphicsのインスタンス</param>
        /// <param name="x">左上X座標</param>
        /// <param name="y">左上Y座標</param>
        private void DrawDropDownButton(Graphics g, int x, int y)
        {
            if (this.DropDown.Visible == false)
            {
                //// ドロップダウンメニューが表示されていないとき

                if (this.dropDownButtonSelectedImageEnabled == true)
                {
                    //// 右側のドロップダウンボタンが選択されていた場合

                    if (this.EnableButtons == true)
                    {
                        // ドロップダウンメニューが有効になっている場合
                        this.DrawImageWithNullCheck(g, this.DropDownButtonSelectedImage, this.DropDownButtonBounds.Location.X - 1, this.DropDownButtonBounds.Location.Y);
                    }
                    else
                    {
                        // ドロップダウンメニューが有効になっていない場合
                        this.DrawImageDisabledWithNullCheck(g, this.DropDownButtonNonSelectedImage, this.DropDownButtonBounds.Location.X - 1, this.DropDownButtonBounds.Location.Y);
                    }
                }
                else if (this.buttonSelectedImageEnabled == true)
                {
                    //// 左側のボタンが選択されていた場合

                    this.DrawImageWithNullCheck(g, this.DropDownButtonNonSelectedImage, this.DropDownButtonBounds.Location.X - 1, this.DropDownButtonBounds.Location.Y);
                    if (this.EnableButtons == false)
                    {
                        // 左右のボタンが選択不可の場合は、右側のボタンの矢印を、DropDownButtonImageをグレーオフしたもので上書きする.
                        this.DrawImageDisabledWithNullCheck(g, this.DropDownButtonImage, this.DropDownButtonBounds.Location.X - 1, this.DropDownButtonBounds.Location.Y);
                    }
                }
                else
                {
                    //// 何も選択されていない場合

                    if (this.EnableButtons)
                    {
                        // 左右のボタンが選択可能なとき
                        this.DrawImageWithNullCheck(g, this.DropDownButtonImage, this.DropDownButtonBounds.Location.X - 1, this.DropDownButtonBounds.Location.Y);
                    }
                    else
                    {
                        // 左右のぼたんが選択不可能なとき
                        this.DrawImageDisabledWithNullCheck(g, this.DropDownButtonImage, this.DropDownButtonBounds.Location.X - 1, this.DropDownButtonBounds.Location.Y);
                    }
                }
            }
            else
            {
                // ドロップダウンメニューが表示されているとき
                this.DrawImageWithNullCheck(g, this.DropDownMenuOpenedImage, 0, 0);
            }
        }

        /// <summary>
        /// 右側の"その他ノード作成"ボタンのびょうが　
        /// </summary>
        /// <param name="g">Graphicsのインスタンス</param>
        /// <param name="x">左上X座標</param>
        /// <param name="y">左上Y座標</param>
        private void DrawButton(Graphics g, int x, int y)
        {
            if (this.DropDown.Visible == false)
            {
                //// ドロップダウンメニューが表示されていないとき

                if (this.dropDownButtonSelectedImageEnabled == true)
                {
                    //// 右側のドロップダウンボタンが選択されていた場合

                    this.DrawImageWithNullCheck(g, this.ButtonNonSelectedImage, 0, 0);
                }
                else if (this.buttonSelectedImageEnabled == true)
                {
                    //// 左側のボタンが選択されていた場合

                    // 左側のボタン
                    if (this.DrawEnabledButton == true && this.EnableButtons == true)
                    {
                        if (this.ButtonPressed == true)
                        {
                            // ボタンが押せる状態で、ボタンが押されている場合
                            this.DrawImageWithNullCheck(g, this.ButtonDownImage, 0, 0);
                        }
                        else
                        {
                            // ボタンが押せる状態で、ボタンが押されていない場合
                            this.DrawImageWithNullCheck(g, this.ButtonSelectedImage, 0, 0);
                        }
                    }
                    else
                    {
                        // ボタンを押すことができない場合
                        this.DrawImageDisabledWithNullCheck(g, this.ButtonNonSelectedImage, 0, 0);
                    }
                }
            }

            // 左側のボタンの描画
            if (this.DrawEnabledButton && this.EnableButtons)
            {
                this.DrawImageWithNullCheck(g, this.Image, x, y);
            }
            else
            {
                // グレーオフする場合は、元画像をグレーオフして描画
                this.DrawImageDisabledWithNullCheck(g, this.Image, x, y);
            }

            // 重ね書きする画像が設定されている場合は、描画する
            if (this.Parent != null && this.OverlayImage && this.AdditionalImage != null)
            {
                if (!this.DrawEnabledButton || !this.EnableButtons)
                {
                    // グレーオフする倍は、重ね書きする画像もグレーオフする.
                    this.DrawImageDisabledWithNullCheck(g, this.AdditionalImage, x, y);
                }
                else
                {
                    this.DrawImageWithNullCheck(g, this.AdditionalImage, x, y);
                }
            }
        }
    }
}
