﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Focus;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// En extended TextBox class.
    /// </summary>
    public class UITextBox : TextBox, IControl, ILayoutElement
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Stores the value of the Text property when the event GotFocus is raised.
        /// This is to be compare with the new Text property when the event LostFocus is raised.
        /// </summary>
        private string beginText;

        /// <summary>
        /// 複数行モード時は改行後の文字列を処理する必要があるため、このフラグを立ててOnTextChangedで確定処理を行う。
        /// </summary>
        private bool afterEnter = false;

        /// <summary>
        /// OnTextChangedメソッド内でthis.Textを編集している最中かどうか
        /// </summary>
        private bool checkingText = false;

        /// <summary>
        /// The is draw background.
        /// </summary>
        private bool isDrawBackground = true;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UITextBox()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
            this.ProhibitLineFeed = false;
            this.AllowableByteCount = -1;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// テキストが空のときに表示するコメントを取得または設定します。
        /// </summary>
        [Localizable(true)]
        public string PromptText { get; set; }

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        /// <summary>
        /// 確定時のみテキストボックスの更新を通知する場合はtrue,そうでなければfalse.
        /// </summary>
        public bool OnlyUpdatingWithValidation { get; set; }

        /// <summary>
        /// 改行コードを制限するかどうか
        /// </summary>
        public bool ProhibitLineFeed { get; set; }

        /// <summary>
        /// 入力文字列に対して許容するバイト数を取得または設定します。
        /// 文字列はUTF8でエンコードされるものとします。
        /// </summary>
        public int AllowableByteCount { get; set; }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// The is draw background.
        /// </summary>
        protected bool IsDrawBackground
        {
            get
            {
                return this.isDrawBackground;
            }

            set
            {
                this.isDrawBackground = value;
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        #region TextPropertyChanged

        /// <summary>
        /// Called when the Text propety changed.
        /// This method raises a 'Text' change notification.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnTextChanged(EventArgs e)
        {
            if (this.checkingText == true)
            {
                return;
            }

            base.OnTextChanged(e);

            // バイト数制限チェック
            if (this.AllowableByteCount > 0)
            {
                var tmp = Encoding.UTF8.GetBytes(this.Text);
                if (tmp.Length > this.AllowableByteCount)
                {
                    this.checkingText = true;
                    this.Text = Encoding.UTF8.GetString(tmp, 0, this.AllowableByteCount);
                    this.SelectionStart = this.Text.Length;
                    this.Focus();
                    this.ScrollToCaret();
                    this.checkingText = false;
                }
            }

            // this.ProhibitLineFeed==trueのときは、改行コードを削除する
            if (this.ProhibitLineFeed == true && (this.Text.Contains("\r") || this.Text.Contains("\n")))
            {
                this.checkingText = true;
                this.Text = this.Text.Replace("\r", string.Empty).Replace("\n", string.Empty);
                this.checkingText = false;
            }

            // 改行後はこちらで確定処理を行う
            if (this.afterEnter)
            {
                this.UpdateAndValidation();
                this.beginText = this.Text;
                this.afterEnter = false;
                return;
            }

            if (this.OnlyUpdatingWithValidation)
            {
                return;
            }

            this.controlExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Text");
        }

        #endregion // TextPropertyChanged

        /// <summary>
        /// クリック時のテキスト全選択
        /// </summary>
        /// <param name="e">event</param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);
            if (OptionStore.RootOptions.Interface.ClickToSelectAllInTextBox)
            {
                this.SelectAll();
            }
        }

        /// <summary>
        /// Raised when the control got focused.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnGotFocus(EventArgs e)
        {
            if (UIUserControl.DisableMenuShortcut != null)
            {
                UIUserControl.DisableMenuShortcut();
            }

            base.OnGotFocus(e);
            this.beginText = this.Text;
        }

        /// <summary>
        /// Called when the control lost the focus.
        /// This method may raise a 'Text' change notification.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnLostFocus(EventArgs e)
        {
            if (UIUserControl.EnableMenuShortcut != null)
            {
                UIUserControl.EnableMenuShortcut();
            }

            base.OnLostFocus(e);
            if (this.beginText != this.Text)
            {
                this.UpdateAndValidation();
            }
        }

        /// <summary>
        /// Called when a key is pressed.
        /// This method may raise a 'Text' change notification.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            base.OnKeyDown(e);

            if (e.KeyData == Keys.Enter)
            {
                if (this.Multiline && !this.ProhibitLineFeed)
                {
                    this.afterEnter = true;
                }
                else if (this.beginText != this.Text)
                {
                    this.UpdateAndValidation();
                    this.beginText = this.Text;
                }

                if (!this.Multiline)
                {
                    this.SelectAll();
                }
                else if (this.ProhibitLineFeed)
                {
                    // Multilineがtrueだと、this.Textが""になるOnTextChangedイベントが2,3回連続して発生してしまうので、
                    // 一時的にfalseにする。
                    this.Multiline = false;
                    this.SelectAll();
                    this.Multiline = true;
                }
            }
            else if (e.KeyData == Keys.Escape)
            {
                if (this.beginText != this.Text)
                {
                    this.UpdateAndValidation();
                    this.beginText = this.Text;
                }

                ControlFocusManager.FocusTabItemOrSubTabItem();
            }
        }

        /// <summary>
        /// The wnd proc.
        /// </summary>
        /// <param name="m">
        /// The m.
        /// </param>
        protected override void WndProc(ref Message m)
        {
            const int WmPaint = 15;
            const int WmSetfocus = 7;
            const int WmKillfocus = 8;

            switch (m.Msg)
            {
                case WmSetfocus:
                    this.IsDrawBackground = false;
                    break;

                case WmKillfocus:
                    this.IsDrawBackground = true;
                    break;
            }

            base.WndProc(ref m);

            if (m.Msg == WmPaint)
            {
                if (this.IsDrawBackground && string.IsNullOrEmpty(this.Text))
                {
                    using (var g = this.CreateGraphics())
                    {
                        this.DrawPrompt(g);
                    }
                }
            }
        }

        /// <summary>
        /// The draw prompt.
        /// </summary>
        /// <param name="g">
        /// The g.
        /// </param>
        protected void DrawPrompt(Graphics g)
        {
            if (string.IsNullOrEmpty(this.PromptText))
            {
                return;
            }

            if (string.IsNullOrEmpty(this.Text) == false)
            {
                return;
            }

            using (var promptFont = new Font(this.Font, FontStyle.Regular))
            {
                const TextFormatFlags LeftFlag = TextFormatFlags.Left | TextFormatFlags.VerticalCenter;

                TextRenderer.DrawText(
                    g,
                    this.PromptText,
                    promptFont,
                    this.ClientRectangle,
                    Color.Gray,
                    this.BackColor,
                    LeftFlag);
            }
        }

        /// <summary>
        /// 値のアップデートと確定を行います。
        /// </summary>
        private void UpdateAndValidation()
        {
            if (this.OnlyUpdatingWithValidation)
            {
                this.controlExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Text");
            }

            this.controlExtender.NotifyPropertyChanged(BindingUpdateType.Validation, "Text");
        }
    }
}
