﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;
using ControlExtensions = EffectMaker.UIControls.Extensions.ControlExtensions;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended Panel class which switches it's contents according to the
    /// available items and selected item.
    /// </summary>
    public class UISwitchPanel : Panel, IItemsControl, ILayoutElement
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// Stores the ItemsControlExtender instance.
        /// </summary>
        private ItemsControlExtender itemsControlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the SelectedContainer property.
        /// </summary>
        private ILogicalTreeElement selectedContainer;

        /// <summary>
        /// Backing field for the SelectedItem property.
        /// </summary>
        private object selectedItem;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public UISwitchPanel()
        {
            this.AutoSize = true;
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.itemsControlExtender = new ItemsControlExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
            this.BackColor = System.Drawing.Color.Transparent;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets or sets the sequence of items used as input data source.
        /// </summary>
        public IEnumerable ItemsSource
        {
            get
            {
                return this.itemsControlExtender.ItemsSource;
            }

            set
            {
                this.controlExtender.SkipPropertyChangedEvent = true;
                this.itemsControlExtender.ItemsSource = value;
                this.controlExtender.SkipPropertyChangedEvent = false;
            }
        }

        /// <summary>
        /// Gets or sets the type of container to create for data items.
        /// </summary>
        public Type ItemContainerType
        {
            get { return this.itemsControlExtender.ItemContainerType; }
            set { this.itemsControlExtender.ItemContainerType = value; }
        }

        /// <summary>
        /// Gets or sets the ItemContainerSelector instance.
        /// </summary>
        public IItemContainerSelector ItemContainerSelector
        {
            get { return this.itemsControlExtender.ItemContainerSelector; }
            set { this.itemsControlExtender.ItemContainerSelector = value; }
        }

        /// <summary>
        /// Gets or sets the selected item container.
        /// </summary>
        public ILogicalTreeElement SelectedContainer
        {
            get
            {
                return this.selectedContainer;
            }

            set
            {
                this.controlExtender.SetValue(ref this.selectedContainer, value);

                // Always update the child control visibility.
                LayoutEngineBase.SuspendLayout();

                foreach (ILogicalTreeElement child in this.Controls)
                {
                    var childLayoutElement = child as ILayoutElement;
                    if (childLayoutElement != null &&
                        child != this.selectedContainer)
                    {
                        childLayoutElement.Visibility = Visibility.Collapsed;
                    }
                }

                LayoutEngineBase.ResumeLayout();

                bool hasAnyNonCollapsedChildren = false;

                var selectedLayoutElement = this.selectedContainer as ILayoutElement;
                if (selectedLayoutElement != null)
                {
                    hasAnyNonCollapsedChildren = true;
                    selectedLayoutElement.Visibility = Visibility.Visible;
                }

                if (hasAnyNonCollapsedChildren != this.Visible)
                {
                    this.Visibility = hasAnyNonCollapsedChildren ? Visibility.Visible : Visibility.Collapsed;
                }
            }
        }

        /// <summary>
        /// Gets or sets the selected data item.
        /// </summary>
        public object SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (this.controlExtender.SetValue(ref this.selectedItem, value) == true)
                {
                    if (value != null)
                    {
                        IControl container =
                            ControlExtensions.GetContainerFromDataItem(this, value);
                        this.SelectedContainer = container;
                    }
                    else
                    {
                        this.SelectedContainer = null;
                    }
                }
            }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Compares two child controls for sorting.
        /// </summary>
        /// <param name="a">First control.</param>
        /// <param name="b">Second control.</param>
        /// <returns>Returns 0 if equal, 1 or -1 if different depending on order.</returns>
        public virtual int CompareControls(IControl a, IControl b)
        {
            return 0;
        }

        /// <summary>
        /// Retrieve the data item corresponding to the container.
        /// </summary>
        /// <param name="container">The container to look for its data item.</param>
        /// <returns>Returns the data item, or null if not found.</returns>
        public object GetDataItemFromContainer(IControl container)
        {
            return this.itemsControlExtender.GetDataItemFromContainer(container);
        }

        /// <summary>
        /// Retrieve the container corresponding to the data item.
        /// </summary>
        /// <param name="dataItem">The data item to look for its data.</param>
        /// <returns>Returns the container, or null if not found.</returns>
        public IControl GetContainerFromDataItem(object dataItem)
        {
            return this.itemsControlExtender.GetContainerFromDataItem(dataItem);
        }

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            Size preferredSize = new Size(0, 0);

            foreach (ILayoutElement child in ((IControl)this).GetNonCollapsedControls())
            {
                Size childSize = child.GetElementDisplaySize(proposedSize);

                int w = childSize.Width - child.Margin.Left + child.Left;
                int h = childSize.Height - child.Margin.Top + child.Top;

                if (w > preferredSize.Width)
                {
                    preferredSize.Width = w;
                }

                if (h > preferredSize.Height)
                {
                    preferredSize.Height = h;
                }
            }

            return preferredSize;
        }
    }
}
