﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended CheckBox class.
    /// </summary>
    public class UICheckBox : CheckBox, IControl, ILayoutElement
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// ビットマスクです.
        /// </summary>
        private ulong bitMask = 0x0001u;

        /// <summary>
        /// ビットフラグです.
        /// </summary>
        private ulong maskedValue = 0x0000u;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UICheckBox()
        {
            this.AutoSize = true;
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);

            this.TabStop = false;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// ビットマスクを取得または設定します.
        /// </summary>
        public ulong BitMask
        {
            get
            {
                return this.bitMask;
            }

            set
            {
                if (this.bitMask == value)
                {
                    return;
                }

                this.bitMask = value;
            }
        }

        /// <summary>
        /// ビットフラグを取得または設定します.
        /// </summary>
        public ulong MaskedValue
        {
            get
            {
                return this.maskedValue;
            }

            set
            {
                if (this.maskedValue == value)
                {
                    return;
                }

                this.maskedValue = value;
                this.Checked = (this.maskedValue & this.bitMask) != 0;

                this.controlExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged);
            }
        }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets or sets the IExecutable to be executed when the checked property changes.
        /// </summary>
        public IExecutable OnCheckedChangedExecutable { get; set; }

        /// <summary>
        /// Gets or sets the custom paramter of the OnValueChangedExecutable.
        /// </summary>
        public object OnCheckedChangedExecutableParameter { get; set; }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// Force the check box to show dashed rectangle around it's text when focused.
        /// ※Trunk版ではフォーカス枠を一括して非表示にしていたため、それに合わせます。
        /// </summary>
        protected override bool ShowFocusCues
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// Gets the size the control want or need to have, according to a given available space.
        /// これがないと再読み込み時にレイアウトが崩れる。要調査
        /// </summary>
        /// <param name="proposedSize">The available space.</param>
        /// <returns>Returns the desired size.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            Size textSize = TextRenderer.MeasureText(this.Text, this.Font);

            return new Size(
                this.Padding.Horizontal + 16 + textSize.Width + 5,
                this.Padding.Vertical + Math.Max(16, textSize.Height));
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        #region CheckPropertyChanged

        /// <summary>
        /// Called when the Checked (or CheckState) property changed.
        /// This method raises a 'Checked' change notification.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnCheckedChanged(EventArgs e)
        {
            base.OnCheckedChanged(e);
            this.controlExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                "Checked");

            var checkEvent = new CheckBoxChangedEventArgs(
                this.Checked,
                this.OnCheckedChangedExecutableParameter);

            if (this.Checked)
            {
                this.MaskedValue = this.MaskedValue | this.BitMask;
            }
            else
            {
                this.MaskedValue = this.MaskedValue & ~this.BitMask;
            }

            var executable = this.OnCheckedChangedExecutable;
            if (executable != null && executable.CanExecute(checkEvent))
            {
                executable.Execute(checkEvent);
            }
        }

        /// <summary>
        /// Called when the CheckState (or Checked) property changed.
        /// This method raises a 'CheckState' change notification.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnCheckStateChanged(EventArgs e)
        {
            base.OnCheckStateChanged(e);
            this.controlExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                "CheckState");
        }

        #endregion // CheckPropertyChanged
    }
}
