﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended Button class.
    /// </summary>
    public class UIButton : Button, IControl, ILayoutElement, IExecutionSource
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// The IExecutionSource extender instance.
        /// </summary>
        private ExecutionSourceExtender executionSourceExtender;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// ツールチップ
        /// </summary>
        private ToolTip toolTip = null;

        /// <summary>
        /// ツールチップ文字列
        /// </summary>
        private string toolTipText;

        /// <summary>
        /// フォーカス枠の有無
        /// </summary>
        private bool enableFocusCues = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UIButton()
        {
            this.AutoSize = true;
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.executionSourceExtender = new ExecutionSourceExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
            this.ToolTipPoint = new Point(20, -25);
            this.ToolTipDuration = 2000;

            this.EnableFocusCues = false;
            this.TabStop = false;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Fired before Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler BeforeExecute
        {
            add { this.executionSourceExtender.BeforeExecute += value; }
            remove { this.executionSourceExtender.BeforeExecute -= value; }
        }

        /// <summary>
        /// Fired after Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler AfterExecute
        {
            add { this.executionSourceExtender.AfterExecute += value; }
            remove { this.executionSourceExtender.AfterExecute -= value; }
        }

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        #region IExecutionSource implementation

        /// <summary>
        /// Gets or sets the executable to execute when clicked.
        /// </summary>
        public IExecutable Executable
        {
            get
            {
                return this.executionSourceExtender.Executable;
            }

            set
            {
                this.executionSourceExtender.Executable = value;
            }
        }

        /// <summary>
        /// Gets or sets a custom parameter to provide to the execution.
        /// </summary>
        public object ExecutionParameter
        {
            get
            {
                return this.executionSourceExtender.ExecutionParameter;
            }

            set
            {
                this.executionSourceExtender.ExecutionParameter = value;
            }
        }

        #endregion // IExecutionSource implementation

        /// <summary>
        /// ツールチップテキストを取得または設定します。
        /// </summary>
        [Localizable(true)]
        public string ToolTipText
        {
            get
            {
                return this.toolTipText;
            }

            set
            {
                this.toolTipText = value;

                if (this.toolTip == null)
                {
                    this.toolTip = new ToolTip
                    {
                        AutoPopDelay = 1200,
                        InitialDelay = 1,
                        ReshowDelay = 1
                    };

                    this.MouseEnter += (s, e) => this.ShowToolTip();
                    this.MouseLeave += (s, e) => this.toolTip.Hide(this);
                }
            }
        }

        /// <summary>
        /// ツールチップテキストを表示する座標
        /// </summary>
        public Point ToolTipPoint { get; set; }

        /// <summary>
        /// ツールチップテキストを表示する時間
        /// -1ならマウスを当てている間ずっと表示する。
        /// </summary>
        public int ToolTipDuration { get; set; }

        /// <summary>
        /// フォーカス枠表示の有無を取得または設定します。
        /// </summary>
        public bool EnableFocusCues
        {
            get
            {
                return this.enableFocusCues;
            }

            set
            {
                this.enableFocusCues = value;
                this.SetStyle(ControlStyles.Selectable, value);
            }
        }

        /// <summary>
        /// フォーカス枠の表示を無効化するオーバーライド
        /// </summary>
        protected override bool ShowFocusCues
        {
            get { return this.enableFocusCues; }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Called when the button is clicked.
        /// </summary>
        /// <param name="e">The event argument.</param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);

            this.executionSourceExtender.ExecuteOnEvent(this, e);

            // ツールチップテキストがあったら、ツールチップを表示し直す.
            if (this.toolTipText != null)
            {
                this.ShowToolTip();
            }
        }

        /// <summary>
        /// ツールチップを表示する。
        /// </summary>
        private void ShowToolTip()
        {
            if (this.ToolTipDuration < 0)
            {
                this.toolTip.Show(this.toolTipText, this, this.ToolTipPoint);
            }
            else
            {
                this.toolTip.Show(this.toolTipText, this, this.ToolTipPoint, this.ToolTipDuration);
            }
        }
    }
}
