﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Texture.Types;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// テクスチャビューア
    /// </summary>
    public partial class UITextureViewer : UserControl
    {
        /// <summary>
        /// 市松模様サイズ
        /// </summary>
        private const int CheckerSize = 10;

        /// <summary>
        /// 市松模様ビットマップ
        /// </summary>
        private Bitmap checkerBitmap = null;

        /// <summary>
        /// RGBを有効にするか
        /// </summary>
        private bool isEnableRgb = true;

        /// <summary>
        /// アルファを有効にするか
        /// </summary>
        private bool isEnableAlpha = true;

        /// <summary>
        /// 情報表示を有効にするか
        /// </summary>
        private bool isShowInfo = true;

        /// <summary>
        /// 描画スケール
        /// </summary>
        private float renderScale = 1.0f;

        /// <summary>
        /// 対象ファイルパス
        /// </summary>
        private string targetFilePath;

        /// <summary>
        /// 描画ミップマップレベル
        /// </summary>
        private int renderMipmapLevel;

        /// <summary>
        /// テクスチャビットマップ
        /// </summary>
        private Bitmap textureBitmap = null;

        /// <summary>
        /// テクスチャデータ
        /// </summary>
        private TextureData textureData = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UITextureViewer()
        {
            this.InitializeComponent();

            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;

            this.Disposed += (s, e) => OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
        }

        /// <summary>
        /// テクスチャビットマップ変更イベント
        /// </summary>
#pragma warning disable 67
        public event EventHandler TextureBitmapChanged;
#pragma warning restore 67

        /// <summary>
        /// RGBを有効にするか
        /// </summary>
        public bool IsEnableRgb
        {
            get
            {
                return this.isEnableRgb;
            }

            set
            {
                this.isEnableRgb = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// アルファを有効にするか
        /// </summary>
        public bool IsEnableAlpha
        {
            get
            {
                return this.isEnableAlpha;
            }

            set
            {
                this.isEnableAlpha = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// 情報表示を有効にするか
        /// </summary>
        public bool IsShowInfo
        {
            get
            {
                return this.isShowInfo;
            }

            set
            {
                this.isShowInfo = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// 対象ファイルパス
        /// </summary>
        public string TargetFilePath
        {
            get
            {
                return this.targetFilePath;
            }

            set
            {
                this.targetFilePath = value;
                this.UpdataTextureAync(this.targetFilePath, this.RenderMipmapLevel);
            }
        }

        /// <summary>
        /// 描画スケール
        /// </summary>
        public float RenderScale
        {
            get
            {
                return this.renderScale;
            }

            set
            {
                this.renderScale = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// 描画ミップマップレベル
        /// </summary>
        public int RenderMipmapLevel
        {
            get
            {
                return this.renderMipmapLevel;
            }

            set
            {
                this.renderMipmapLevel = value;
                this.UpdataTextureAync(this.TargetFilePath, this.RenderMipmapLevel);
            }
        }

        /// <summary>
        /// テクスチャデータ
        /// </summary>
        public TextureData TextureData
        {
            get
            {
                return this.textureData;
            }
        }

        /// <summary>
        /// SRGBフォーマットか？
        /// </summary>
        private bool IsSrgbFormat
        {
            get
            {
                if (this.textureData == null)
                {
                    return false;
                }

                return (this.textureData.PixelFormat == PixelFormats.Srgb_8_8_8_8) ||
                       (this.textureData.PixelFormat == PixelFormats.Srgb_bc1) ||
                       (this.textureData.PixelFormat == PixelFormats.Srgb_bc2) ||
                       (this.textureData.PixelFormat == PixelFormats.Srgb_bc3);
            }
        }

        /// <summary>
        /// OnLoad
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            this.DoubleBuffered = true;
        }

        /// <summary>
        /// Handle PaintBackground event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnPaintBackground(PaintEventArgs e)
        {
            if (this.checkerBitmap == null)
            {
                this.checkerBitmap = RenderUtility.CreateCheckerBoardImage(this.ClientRectangle.Width, this.ClientRectangle.Height, CheckerSize);
            }

            e.Graphics.DrawImage(this.checkerBitmap, 0, 0);
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if ((this.checkerBitmap == null) ||
                (this.checkerBitmap.Width < this.ClientRectangle.Width) ||
                (this.checkerBitmap.Height < this.ClientRectangle.Height))
            {
                if (this.checkerBitmap != null)
                {
                    this.checkerBitmap.Dispose();
                }

                this.checkerBitmap = RenderUtility.CreateCheckerBoardImage(this.ClientRectangle.Width, this.ClientRectangle.Height, CheckerSize);
            }

            base.OnSizeChanged(e);
        }

        /// <summary>
        /// 描画イベント処理
        /// </summary>
        /// <param name="e">描画イベント情報</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            this.RenderTexture(e.Graphics);
            this.RenderInformation(e.Graphics);
        }

        /// <summary>
        /// コンポーネントセレクタ文字列を作る
        /// </summary>
        /// <param name="ftxTextureData">テクスチャ</param>
        /// <returns>コンポーネントセレクタ文字列</returns>
        private static string MakeComponentString(FtxTextureData ftxTextureData)
        {
            System.Diagnostics.Debug.Assert(ftxTextureData != null, "テクスチャデータが不正");
            System.Diagnostics.Debug.Assert(ftxTextureData.CompSel != null, "コンポーネントセレクタが不正");

            var sb = new StringBuilder();
            {
                foreach (var c in ftxTextureData.CompSel)
                {
                    switch (c)
                    {
                        case ColorComponents.Item0:
                            sb.Append("0");
                            break;
                        case ColorComponents.Item1:
                            sb.Append("1");
                            break;
                        default:
                            sb.Append(c);
                            break;
                    }
                }
            }

            return sb.ToString();
        }

        /// <summary>
        /// ミップマップサイズ文字列を作る
        /// </summary>
        /// <param name="ftxTextureData">テクスチャ</param>
        /// <returns>ミップマップサイズ文字列</returns>
        private static string MakeMipmapSizeString(FtxTextureData ftxTextureData)
        {
            System.Diagnostics.Debug.Assert(ftxTextureData != null, "テクスチャデータが不正");

            return string.Format("{0} x {1}", ftxTextureData.Width, ftxTextureData.Height);
        }

        /// <summary>
        /// 非同期テクスチャ更新
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="mipmapLevel">ミップマップレベル</param>
        private void UpdataTextureAync(string filePath, int mipmapLevel)
        {
            if (this.textureBitmap != null)
            {
                this.textureBitmap.Dispose();
                this.textureBitmap = null;
                this.textureData = null;

                if (this.TextureBitmapChanged != null)
                {
                    this.TextureBitmapChanged(this, EventArgs.Empty);
                }

                this.Invalidate();
            }

            // 裏読み
            Task.Factory.StartNew(() => EffectMaker.BusinessLogic.Manager.TextureManager.Instance.LoadTexture(filePath, true))

            // 裏読み終了時
            .ContinueWith(task =>
            {
                if (task.Result.TextureData != null)
                {
                    textureBitmap = task.Result.TextureData.GenerateBitmap(mipmapLevel);
                    textureData = task.Result.TextureData;

                    if (this.TextureBitmapChanged != null)
                    {
                        this.TextureBitmapChanged(this, EventArgs.Empty);
                    }

                    Invalidate();
                }
            });
        }

        /// <summary>
        /// Render the texture.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        private void RenderTexture(Graphics g)
        {
            if (this.textureBitmap == null)
            {
                return;
            }

            if (this.textureData == null)
            {
                return;
            }

            if ((this.IsEnableRgb == false) && (this.IsEnableAlpha == false))
            {
                return;
            }

            var interpolationMode = g.InterpolationMode;

            try
            {
                g.InterpolationMode = InterpolationMode.NearestNeighbor;

                var w = (int)(this.textureBitmap.Width * this.RenderScale);
                var h = (int)(this.textureBitmap.Height * this.RenderScale);
                var x = (this.ClientRectangle.Width  - w) / 2;
                var y = (this.ClientRectangle.Height - h) / 2;

                var rect = new Rectangle(x, y, w, h);

                using (var ia = new ImageAttributes())
                {
                    ColorMatrix colorMatrix;
                    {
                        var ftxTextureData = this.textureData as FtxTextureData;

                        if (ftxTextureData == null)
                        {
                            colorMatrix = this.IsEnableRgb ? RenderUtility.CreateRgbColorMatrix() : RenderUtility.CreateAlphaColorMatrix();
                        }
                        else
                        {
                            if (this.IsEnableRgb && this.IsEnableAlpha)
                            {
                                colorMatrix = RenderUtility.CreateRgbaColorMatrix(ftxTextureData.CompSel, ftxTextureData.PixelFormat);
                            }
                            else if (this.IsEnableRgb)
                            {
                                colorMatrix = RenderUtility.CreateRgbColorMatrix(ftxTextureData.CompSel, ftxTextureData.PixelFormat);
                            }
                            else
                            {
                                colorMatrix = RenderUtility.CreateAlphaColorMatrix(ftxTextureData.CompSel, ftxTextureData.PixelFormat);
                            }
                        }
                    }

                    ia.SetColorMatrix(colorMatrix);

                    if (ColorUtility.Gamma != 1.0)
                    {
                        if (this.IsSrgbFormat)
                        {
                            if (ColorUtility.IsGammaCorrectionEnabled == false)
                            {
                                ia.SetGamma((float)ColorUtility.Gamma);
                            }
                        }
                        else
                        {
                            if (ColorUtility.IsGammaCorrectionEnabled)
                            {
                                ia.SetGamma((float)(1.0 / ColorUtility.Gamma));
                            }
                        }
                    }

                    g.DrawImage(
                        this.textureBitmap,
                        rect,
                        0,
                        0,
                        this.textureBitmap.Width,
                        this.textureBitmap.Height,
                        GraphicsUnit.Pixel,
                        ia);
                }
            }
            finally
            {
                g.InterpolationMode = interpolationMode;
            }
        }

        /// <summary>
        /// Render texture information.
        /// </summary>
        /// <param name="g">Graphics object for rendering.</param>
        private void RenderInformation(Graphics g)
        {
            if (this.IsShowInfo == false)
            {
                return;
            }

            if (this.textureData == null)
            {
                return;
            }

            var smoothingMode   = g.SmoothingMode;

            try
            {
                g.SmoothingMode   = SmoothingMode.HighQuality;

                // ファイルパス
                var filePathString = string.Format("{0}： {1}", Properties.Resources.TextureViewerFilePath, this.TargetFilePath);

                // 各種情報
                var infoString = string.Empty;
                {
                    var ftxTextureData = this.textureData as FtxTextureData;
                    if (ftxTextureData != null)
                    {
                        infoString = string.Format(
                            "{0}： {1}\n{2}： {3}\n{4}： {5}\n{6}： {7}\n{8}： {9}",
                            Properties.Resources.TextureViewerFormat,
                            ftxTextureData.PixelFormat.ToString().ToLower(),
                            Properties.Resources.TextureViewerComponent,
                            MakeComponentString(ftxTextureData),
                            Properties.Resources.TextureViewerMipmapSize,
                            MakeMipmapSizeString(ftxTextureData),
                            Properties.Resources.TextureViewerMipmapLevel,
                            ftxTextureData.MipmapCount,
                            Properties.Resources.TextureViewerCurrentMipmap,
                            this.RenderMipmapLevel);
                    }
                }

                // 描画
                {
                    const float Spacing = 5.0f;

                    var x = 10.0f;
                    var y = 10.0f;

                    using (var backgroundBrush = new SolidBrush(Color.FromArgb(128, 0, 0, 0)))
                    {
                        y += this.RenderStringWithBackground(g, backgroundBrush, filePathString, x, y);
                        y += Spacing;
                        y += this.RenderStringWithBackground(g, backgroundBrush, infoString, x, y);
                    }
                }
            }
            finally
            {
                g.SmoothingMode = smoothingMode;
            }
        }

        /// <summary>
        /// 背景付き文字列描画
        /// </summary>
        /// <param name="g">グラフィックオブジェクト</param>
        /// <param name="backgroundBrush">背景ブラシ</param>
        /// <param name="str">文字列</param>
        /// <param name="x">ｘ位置</param>
        /// <param name="y">ｙ位置</param>
        /// <returns>描画した高さ</returns>
        private float RenderStringWithBackground(Graphics g, Brush backgroundBrush, string str, float x, float y)
        {
            const float Padding = 5.0f;
            const float CornerRadius = 5.0f;

            var strSize = g.MeasureString(str, this.Font);
            var strRect = new RectangleF(x, y, Padding + strSize.Width + Padding, Padding + strSize.Height + Padding);

            // 背景の描画
            using (var backgroundPath = RenderUtility.MakeRoundRectangleGraphicsPath(strRect, CornerRadius))
            {
                g.FillPath(backgroundBrush, backgroundPath);
            }

            // 文字列の描画
            g.DrawString(str, this.Font, Brushes.White, x + Padding, y + Padding);

            return strRect.Height;
        }

        /// <summary>
        /// Handle GammaCorrectionChanged event from application.
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.Invalidate();
        }
    }
}
