﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.UIControls.Specifics.TabPages;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// Remake用の新しいタブ
    /// </summary>
    public class UIRemakeTab : UITabControl
    {
        /// <summary>
        /// タブで利用するフォント.
        /// </summary>
        private Font tabFont = new Font("MS UI Gothic", 8);

        /// <summary>
        /// コンストラクタ.
        /// </summary>
        public UIRemakeTab()
            : base()
        {
            // テーマが無効な場合はデザイン変更をキャンセルする
            if (!Foundation.Win32.Utility.IsThemeActive)
            {
                return;
            }

            // Paintイベントで描画できるようにする
            this.SetStyle(ControlStyles.UserPaint, true);

            // ダブルバッファリングを有効にする
            this.DoubleBuffered = true;

            // リサイズで再描画する
            this.ResizeRedraw = true;

            // ControlStyles.UserPaintをTrueすると、
            // SizeModeは強制的にTabSizeMode.Fixedにされる
            this.SizeMode = TabSizeMode.Fixed;
            this.ItemSize = new Size(64, 21);
            this.Appearance = TabAppearance.Normal;
            this.Multiline = true;
        }

        /// <summary>
        /// OnPaintイベント
        /// </summary>
        /// <param name="e">event.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (this.Visible == false)
            {
                return;
            }

            if (this.TabPages.Count == 0)
            {
                e.Graphics.FillRectangle(SystemBrushes.Window, this.ClientRectangle);
                return;
            }

            // TabControlの背景を塗る
            e.Graphics.FillRectangle(SystemBrushes.Control, this.ClientRectangle);

            // TabPageの枠を描画する
            if (this.SelectedIndex >= 0 && this.SelectedIndex < this.TabPages.Count)
            {
                TabPage page = this.TabPages[this.SelectedIndex];
                Rectangle pageRect = new Rectangle(
                    page.Bounds.X - 2,
                    page.Bounds.Y - 2,
                    page.Bounds.Width + 5,
                    page.Bounds.Height + 5);
                TabRenderer.DrawTabPage(e.Graphics, pageRect);
            }

            // タブを描画する
            for (int i = 0; i < this.TabPages.Count; i++)
            {
                TabPage page = this.TabPages[i];
                Rectangle tabRect = this.GetTabRect(i);

                PropertyTabPageBase propertyTabPage = this.TabPages[i] as PropertyTabPageBase;
                bool selected = this.SelectedIndex == i;

                if (this.SelectedIndex == i)
                {
                    if (this.Alignment == TabAlignment.Top)
                    {
                        tabRect.Height += 1;
                    }
                }

                Size imgSize = tabRect.Size;

                string tabText = page.Text;

                // タブの画像を作成する
                Bitmap bmp = new Bitmap(imgSize.Width, imgSize.Height);
                Graphics g = Graphics.FromImage(bmp);

                var c0 = Color.FromArgb(126, 140, 166);
                var c1 = Color.FromArgb(87, 101, 124);
                var textColor = Brushes.White;
                var borderColor = Color.FromArgb(44, 55, 75);
                if (selected)
                {
                    c0 = Color.FromArgb(239, 240, 240);
                    c1 = Color.FromArgb(159, 194, 214);
                    textColor = Brushes.Black;
                    borderColor = Color.FromArgb(137, 140, 149);
                }

                System.Drawing.Point sp = System.Windows.Forms.Cursor.Position;

                // 画面座標をクライアント座標に変換する
                System.Drawing.Point cp = this.PointToClient(sp);

                if (tabRect.Contains(cp) == true)
                {
                    borderColor = Color.FromArgb(194, 200, 210);
                }

                var rect = new Rectangle(0, 0, imgSize.Width, imgSize.Height);
                using (LinearGradientBrush gb = new LinearGradientBrush(
                    rect, c0, c1, LinearGradientMode.Vertical))
                {
                    g.FillRectangle(gb, rect);
                }

                if (propertyTabPage.IsSomeValuesNotDefault == true)
                {
                    var c2 = Color.FromArgb(255, 174, 141);
                    var orangeRect = new Rectangle(0, this.GetTabRect(i).Height / 2, this.GetTabRect(i).Width, this.GetTabRect(i).Height - 1);
                    using (var brushRed = new LinearGradientBrush(
                        orangeRect, Color.Transparent, c2, LinearGradientMode.Vertical))
                    {
                        g.FillRectangle(brushRed, orangeRect);
                    }
                }

                if (propertyTabPage.IsModified == true)
                {
                    var img = Specifics.Properties.Resources.Icon_Asterisk;
                    g.DrawImage(img, 0, 0);
                }

                using (var pen = new Pen(borderColor))
                {
                    var r0 = rect;
                    r0.Width -= 1;
                    r0.Height -= 1;
                    g.DrawRectangle(pen, r0);
                }

                if (selected)
                {
                    // ボタン下側1pixel分をControlカラーにする.
                    using (var pen = new Pen(SystemBrushes.Control))
                    {
                        g.DrawLine(
                            pen,
                            new Point(rect.Left, rect.Bottom - 1),
                            new Point(rect.Right, rect.Bottom - 1));
                    }
                }

                // テキスト表示.
                using (StringFormat sf = new StringFormat())
                {
                    sf.Alignment = StringAlignment.Center;
                    sf.LineAlignment = StringAlignment.Center;
                    g.DrawString(
                        page.Text,
                        this.tabFont,
                        textColor,
                        new RectangleF(0, 0, bmp.Width, bmp.Height),
                        sf);
                }

                g.Dispose();

                // 画像を描画する
                e.Graphics.DrawImage(bmp, tabRect.X, tabRect.Y, bmp.Width, bmp.Height);

                bmp.Dispose();
            }
        }
    }
}
