﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// A custom group box extending UIUserControl.
    /// </summary>
    public partial class UIGroupBoxEx : UIUserControl
    {
        #region Member variables

        /// <summary>
        /// Stores a flag telling wether the control as been initialized or not.
        /// </summary>
        private bool initializeDone;

        /// <summary>
        /// Backing field for the Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controls;

        /// <summary>
        /// Corner radius.
        /// </summary>
        private int cornerRadius = 6;

        /// <summary>
        /// Height of the title bar.
        /// </summary>
        private int titleBarHeight = 22;

        /// <summary>
        /// Value of the text.
        /// </summary>
        private Color textColor = Color.White;

        /// <summary>
        /// Value of the border.
        /// </summary>
        private Color borderColor = Color.FromArgb(255, 71, 91, 128);

        /// <summary>
        /// Text alignment.
        /// </summary>
        private StringFormat textFormat = new StringFormat();

        #endregion

        #region Constructor

        /// <summary>
        /// Initializes the UIGroupBoxEx instance.
        /// </summary>
        public UIGroupBoxEx()
        {
            this.InitializeComponent();

            // this.pnlContent.Margin.Top と this.titleBarHeight が同じ値になっていることを前提に
            // レイアウトを行うので、違った値を設定すると再配置のときにレイアウトが崩れる
            Padding margin = this.pnlContent.Margin;
            margin.Top = this.titleBarHeight;
            this.pnlContent.Margin = margin;

            this.DoubleBuffered = true;

            this.groupContextMenu.SetLogicalParent(this);

            this.ctxCopyGroup.Text = Properties.Resources.MenuCopyGroup;
            this.ctxPasteGroup.Text = Properties.Resources.MenuPasteGroup;
            this.ctxResetDefaultValues.Text = Properties.Resources.MenuResetGroupDefaultValues;

            this.ctxCopyGroup.AddBinding("Executable", "CopyGroupExecutable").UpdateElement();
            this.ctxPasteGroup.AddBinding("Executable", "PasteGroupExecutable").UpdateElement();
            this.ctxResetDefaultValues.AddBinding("Executable", "ResetDefaultValueExecutable").UpdateElement();

            this.groupContextMenu.Opening += (s, e) =>
            {
                if (this.ctxPasteGroup.Executable != null)
                {
                    this.Focus();
                    this.ctxPasteGroup.Enabled = this.ctxPasteGroup.Executable.CanExecute(
                        this.ctxPasteGroup.ExecutionParameter);
                }
            };

            // Align middle left
            this.textFormat.Alignment = StringAlignment.Near;
            this.textFormat.LineAlignment = StringAlignment.Center;

            this.initializeDone = true;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the radius of the rounded corners.
        /// </summary>
        [DefaultValue(4)]
        public int CornerRadius
        {
            get
            {
                return this.cornerRadius;
            }

            set
            {
                if (value == this.cornerRadius)
                {
                    return;
                }

                this.cornerRadius = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the height of the title bar.
        /// </summary>
        [DefaultValue(22)]
        public int TitleBarHeight
        {
            get
            {
                return this.titleBarHeight;
            }

            set
            {
                if (value == this.titleBarHeight)
                {
                    return;
                }

                // Resize the content.
                Padding margin = this.pnlContent.Margin;
                margin.Top = value;
                this.pnlContent.Margin = margin;

                this.pnlContent.Size = new Size(
                    this.pnlContent.Size.Width,
                    this.Height - value - 4);

                this.titleBarHeight = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Gets or sets the Text property.
        /// This correspond to the title displayed beside content.
        /// </summary>
        [Browsable(true)]
        [DefaultValue("")]
        public override string Text
        {
            get
            {
                return base.Text;
            }

            set
            {
                if (value == base.Text)
                {
                    return;
                }

                base.Text = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Gets or sets the color of the text.
        /// </summary>
        public Color TextColor
        {
            get
            {
                return this.textColor;
            }

            set
            {
                if (value == this.TextColor)
                {
                    return;
                }

                this.textColor = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Gets or sets the color of the border.
        /// </summary>
        public Color BorderColor
        {
            get
            {
                return this.borderColor;
            }

            set
            {
                if (value == this.borderColor)
                {
                    return;
                }

                this.borderColor = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Gets the collection of content controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.initializeDone)
                {
                    return this.pnlContent.Controls;
                }
                else
                {
                    if (this.controls == null)
                    {
                        this.controls = new ControlCollectionWrapper(this);
                    }

                    return this.controls;
                }
            }
        }

        #endregion

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            Size contentpanelSize = this.pnlContent.GetElementDisplaySize(proposedSize);

            return new Size(
                this.Padding.Horizontal + contentpanelSize.Width,
                this.Padding.Vertical + contentpanelSize.Height);
        }

        #region Event handlers

        /// <summary>
        /// Handle paint event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            // Get the graphics object for rendering
            Graphics g = e.Graphics;

            // Smooth the lines
            g.SmoothingMode = SmoothingMode.HighQuality;

            // Create the pen and brushes for rendering.
            Pen penBorder = new Pen(this.BorderColor);
            SolidBrush brushBorder = new SolidBrush(this.BorderColor);
            SolidBrush brushText = new SolidBrush(this.TextColor);

            // Setup the sizes.
            int controlWidth = this.Width - 1;
            int controlHeight = this.Height - 1;

            int titleBarHeight = this.TitleBarHeight - 1;

            // Setup the rectangles.
            Rectangle titleBarRect = new Rectangle(
                0,
                0,
                controlWidth,
                titleBarHeight);
            Rectangle titleTextRect = new Rectangle(
                this.CornerRadius,
                0,
                controlWidth - (this.CornerRadius * 2),
                titleBarHeight);
            Rectangle contentRect = new Rectangle(
                0,
                titleBarHeight,
                controlWidth,
                controlHeight - titleBarHeight);

            // Create path for the border.
            GraphicsPath pathBorder = new GraphicsPath();

            pathBorder.AddLine(
                contentRect.Left,
                contentRect.Bottom - this.CornerRadius,
                contentRect.Left,
                contentRect.Top);
            pathBorder.AddLine(
                contentRect.Left,
                contentRect.Top,
                contentRect.Right,
                contentRect.Top);
            pathBorder.AddLine(
                contentRect.Right,
                contentRect.Top,
                contentRect.Right,
                contentRect.Bottom - this.CornerRadius);
            pathBorder.AddArc(
                contentRect.Right - this.CornerRadius,
                contentRect.Bottom - this.CornerRadius,
                this.CornerRadius,
                this.CornerRadius,
                0.0f,
                90.0f);
            pathBorder.AddLine(
                contentRect.Right - this.CornerRadius,
                contentRect.Bottom,
                contentRect.Left + this.CornerRadius,
                contentRect.Bottom);
            pathBorder.AddArc(
                contentRect.Left,
                contentRect.Bottom - this.CornerRadius,
                this.CornerRadius,
                this.CornerRadius,
                90.0f,
                90.0f);
            pathBorder.CloseAllFigures();

            // Create path for the title bar.
            GraphicsPath pathTitle = new GraphicsPath();

            pathTitle.AddArc(
                titleBarRect.Left,
                titleBarRect.Top,
                this.CornerRadius,
                this.CornerRadius,
                180.0f,
                90.0f);
            pathTitle.AddLine(
                titleBarRect.Left + this.CornerRadius,
                titleBarRect.Top,
                titleBarRect.Right - this.CornerRadius,
                titleBarRect.Top);
            pathTitle.AddArc(
                titleBarRect.Right - this.CornerRadius,
                titleBarRect.Top,
                this.CornerRadius,
                this.CornerRadius,
                270.0f,
                90.0f);
            pathTitle.AddLine(
                titleBarRect.Right,
                titleBarRect.Top + this.CornerRadius,
                titleBarRect.Right,
                titleBarRect.Bottom);
            pathTitle.AddLine(
                titleBarRect.Right,
                titleBarRect.Bottom,
                titleBarRect.Left,
                titleBarRect.Bottom);
            pathTitle.AddLine(
                titleBarRect.Left,
                titleBarRect.Bottom,
                titleBarRect.Left,
                titleBarRect.Top + this.CornerRadius);
            pathTitle.CloseAllFigures();

            // Draw the border.
            g.DrawPath(penBorder, pathBorder);
            g.FillPath(brushBorder, pathTitle);
            g.DrawPath(penBorder, pathTitle);

            // Draw the text.
            g.DrawString(
                this.Text,
                this.Font,
                brushText,
                titleTextRect,
                this.textFormat);

            // Dispose the pen and brushes.
            penBorder.Dispose();
            brushBorder.Dispose();
            brushText.Dispose();
            pathBorder.Dispose();
            pathTitle.Dispose();
        }

        #endregion

        /// <summary>
        /// The on visible changed.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnVisibleChanged(EventArgs e)
        {
            // 本来はStackPanelのAutoSizeModeで対処するのが正しいが、
            // 表示が崩れるページがあるため、こちらで暫定対処
            if (this.Parent != null)
            {
                this.Parent.Height = 1;
            }

            base.OnVisibleChanged(e);
        }

        /// <summary>
        /// Called when a layout process is running.
        /// </summary>
        /// <param name="e">The event argument.</param>
        protected override void OnLayout(LayoutEventArgs e)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return;
            }

            base.OnLayout(e);

            this.pnlContent.SuspendLayout();

            this.pnlContent.Left = this.Padding.Left + this.pnlContent.Margin.Left;
            this.pnlContent.Top = this.Padding.Top + this.pnlContent.Margin.Top;

            this.pnlContent.Width = this.ClientSize.Width
                - (this.Padding.Horizontal + this.pnlContent.Margin.Horizontal);

            this.pnlContent.Height = this.ClientSize.Height
                - (this.Padding.Vertical + this.pnlContent.Margin.Vertical);

            this.pnlContent.ResumeLayout();
        }
    }
}
