﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.Layout;
using EffectMaker.UILogic.Manager;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// Class for the caption label of the components.
    /// </summary>
    public class UICaptionLabel : UIBorderedLabel, IDocumentLinker
    {
        #region Member variables

        /// <summary>Text color if disabled.</summary>
        private static readonly Color TextColorDisabled = Color.FromArgb(255, 150, 150, 150);

        /// <summary>Background color if disabled.</summary>
        private static readonly Color BackColorDisabled = Color.FromArgb(255, 235, 235, 235);

        /// <summary>Default background color.</summary>
        private static readonly Color DefaultLabelBackColor = Color.FromArgb(255, 255, 255, 153);

        /// <summary>Background color when data is modified.</summary>
        private static readonly Color DataModifiedBackColor = Color.FromArgb(255, 255, 220, 153);

        /// <summary>Border line color.</summary>
        private Color borderColor = Color.Black;

        /// <summary>Background color.</summary>
        private Color backColor = UICaptionLabel.DefaultLabelBackColor;

        /// <summary>Image of the star mark.</summary>
        private Image imgStarMark = null;

        /// <summary>Margin of the star mark.</summary>
        private int starMarkMargin = 1;

        /// <summary>Flag indicating if the star mark is visible.</summary>
        private bool isStarMarkVisible = false;

        /// <summary>Flag indicating if the star mark is enabled.</summary>
        private bool isStarMarkEnabled = true;

        /// <summary>Flag indicating if the bound data is its default value.</summary>
        private bool isDefaultValue = true;

        /// <summary>マウスオーバー前のカラー退避用</summary>
        private Color backupColor;

        /// <summary>マウスオーバー前のカーソル退避用</summary>
        private Cursor backupCursor;

        #endregion

        #region Constructor

        /// <summary>
        /// Default onstructor.
        /// </summary>
        public UICaptionLabel()
        {
            this.imgStarMark = Properties.Resources.StarMark;
            this.AutoSize = false;
            this.Multiline = true;
            this.TextAlign = ContentAlignment.MiddleCenter;
            this.TabStop = false;

            if (this.imgStarMark != null)
            {
                ////int paddingH = this.imgStarMark.Width + this.starMarkMargin;
                ////this.Spacing = new Padding(paddingH, 1, paddingH, 1);
                this.Spacing = new Padding(1);
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the flag indicating whether to enable the modify mark or not.
        /// If this is disabled, even when IsStarMarkVisible = true, the mark would
        /// still be hidden.
        /// </summary>
        public bool IsStarMarkEnabled
        {
            get
            {
                return this.isStarMarkEnabled;
            }

            set
            {
                if (value != this.isStarMarkEnabled)
                {
                    this.isStarMarkEnabled = value;
                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether the asterisk mark is visible.
        /// </summary>
        public bool IsStarMarkVisible
        {
            get
            {
                return this.isStarMarkVisible;
            }

            set
            {
                if (value != this.isStarMarkVisible)
                {
                    this.isStarMarkVisible = value;
                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating the bound data is its default value.
        /// </summary>
        public bool IsDefaultValue
        {
            get
            {
                return this.isDefaultValue;
            }

            set
            {
                if (this.isDefaultValue != value)
                {
                    this.isDefaultValue = value;
                    if (value == true)
                    {
                        this.BackColor = UICaptionLabel.DefaultLabelBackColor;
                    }
                    else
                    {
                        this.BackColor = UICaptionLabel.DataModifiedBackColor;
                    }
                }
            }
        }

        /// <summary>
        /// ヘルプとしてリンクするページのIDを取得または設定します。
        /// </summary>
        public string DocumentId { get; set; }

        #endregion

        #region Event handlers

        /// <summary>
        /// Handle Pain event.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
        {
            // Get the graphics object for rendering
            Graphics g = e.Graphics;

            // Smooth the lines
            g.SmoothingMode = SmoothingMode.HighQuality;

            base.OnPaint(e);

            // Render the star mark.
            if (this.isStarMarkEnabled == true &&
                this.isStarMarkVisible == true &&
                this.imgStarMark != null)
            {
                int posX = this.starMarkMargin;
                if (this.Visibility == Visibility.Hidden)
                {
                    posX = this.Width - this.imgStarMark.Width;
                }

                g.DrawImage(
                    this.imgStarMark,
                    posX,
                    /*(this.Height - this.imgStarMark.Height) / 2*/1);
            }
        }

        /// <summary>
        /// マウスオーバーイベントを処理します。文字色とカーソルを変更します。
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);
            if (string.IsNullOrEmpty(this.DocumentId))
            {
                return;
            }

            this.backupColor = this.ForeColor;
            this.backupCursor = this.Cursor;

            this.ForeColor = Color.Blue;
            this.Cursor = Cursors.Hand;
            this.Underline = true;
        }

        /// <summary>
        /// マウスリーブイベントを処理します。文字色とカーソルを復帰します。
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);
            if (string.IsNullOrEmpty(this.DocumentId))
            {
                return;
            }

            this.ForeColor = this.backupColor;
            this.Cursor = this.backupCursor;
            this.Underline = false;
        }

        /// <summary>
        /// クリックで指定されたページを開きます。
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);
            if (string.IsNullOrEmpty(this.DocumentId))
            {
                return;
            }

            if (this.DocumentId.Contains("#"))
            {
                // #を含む場合はアンカーが指定されているものとして扱う
                var splitted = this.DocumentId.Split('#');
                if (splitted.Length != 2)
                {
                    Logger.Log(LogLevels.Error, "Document Link Error.{0}", this.DocumentId);
                    return;
                }

                HelpManager.ShowHelp(splitted[0], splitted[1]);
            }
            else
            {
                HelpManager.ShowHelp(this.DocumentId);
            }
        }

        #endregion
    }
}
