﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// An extended ProjectTreeNodeBase for reserved shader implementation.
    /// </summary>
    public class ReservedShaderTreeNode : ProjectTreeNodeBase
    {
        /// <summary>
        /// 機能の有効 / 無効フラグです。
        /// </summary>
        private bool enableConvert;

        /// <summary>
        /// Initializes the ReservedShaderTreeNode instance.
        /// </summary>
        public ReservedShaderTreeNode()
        {
            this.IsDrawModifyMark = true;

            this.AddBinding("Text", "Name");
            this.AddBinding("Icon", "Icon");
            this.AddBinding("EnableConvert", "EnableConvert");
        }

        /// <summary>
        /// グレー表示のOn/Offを取得または設定します。
        /// </summary>
        public override bool IsGray
        {
            get
            {
                var parent = this.Parent as EmitterTreeNode;
                if (parent != null)
                {
                    return parent.IsGray || !this.EnableConvert;
                }
                else
                {
                    return !this.EnableConvert;
                }
            }

            set
            {
                throw new NotImplementedException();
            }
        }

        /// <summary>
        /// バイナリコンバートを有効にするかどうかのフラグを取得または設定します。
        /// </summary>
        public bool EnableConvert
        {
            get
            {
                return this.enableConvert;
            }

            set
            {
                this.LogicalTreeElementExtender.SetValue(ref this.enableConvert, value);

                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// The icon for this tree node.
        /// </summary>
        public Image Icon { get; set; }

        /// <summary>
        /// Render the node of viewer.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <returns>Returns true if it self render, or false to tall parent
        /// TreeView to perform default node rendering.</returns>
        protected internal override bool Render(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // The tree node might already be removed between the draw message is dispatched
            // and the actual rendering.
            // (the message dispatches, and the rendering happens on the next application loop)
            if (((TreeNode)this).Parent == null &&
                (this.TreeView == null || this.TreeView.Nodes.Contains(this) == false))
            {
                return true;
            }

            // ノード作成時に無効な領域の描画呼び出しが発生する
            if (bounds.Width == 0 || bounds.Height == 0)
            {
                return true;
            }

            // 背景を描画
            this.DrawBackground(bounds, state, gr);

            // 目のアイコンを描画
            ////this.DrawDisplayedIcon(bounds, state, gr);

            // 階層情報の表示位置を取得
            int locationX = ProjectTreeNodeBase.ShowButtonWidth +
                            ProjectTreeNodeBase.ShowButtonMarginRight +
                            bounds.Left;

            // 階層情報を描画
            locationX = this.DrawHierarchy(bounds, state, gr, locationX);

            // ノードアイコンを描画
            locationX = this.DrawNodeTypeIcon(bounds, state, gr, locationX, this.Icon);

            // 編集マークを描画
            locationX = this.DrawModifyMark(bounds, state, gr, locationX);

            // テキストを描画
            this.DrawText(bounds, state, gr, locationX, this.Text, this.NodeFont);

            return true;
        }

        /// <summary>
        /// コンテキストメニューのセットアップ.
        /// </summary>
        protected override void SetupContextMenu()
        {
            var docMenu = this.ContextMenuStrip as UIContextMenuStrip;

            // 機能の有効 / 無効
            var tsmiEnableConvert = new UIToolStripMenuItem();
            docMenu.Items.Add(tsmiEnableConvert);
            tsmiEnableConvert.Text = Properties.Resources.MenuItemNameEnableConvert;
            tsmiEnableConvert.ShortcutKeys = Keys.Control | Keys.Shift | Keys.H;
            tsmiEnableConvert.Image = Properties.Resources.Icon_Menu_Enable;
            tsmiEnableConvert.Click += (s, e) => { this.EnableConvert = !this.EnableConvert; };

            base.SetupContextMenu();
        }
    }
}
