﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// An extended UITreeNode that provides base implementation for project tree nodes.
    /// </summary>
    public abstract class ProjectTreeNodeBase : UITreeNode
    {
        /// <summary>
        /// 表示アイコンの大きさ。
        /// </summary>
        protected const int ShowButtonWidth = 18;

        /// <summary>
        /// 表示アイコンの右側のマージン。
        /// </summary>
        protected const int ShowButtonMarginRight = 4;

        /// <summary>
        /// 階層表示のインデント幅。
        /// </summary>
        protected const int TreeNodeHierarchyWidth = 18;

        /// <summary>
        /// ノードアイコンの右側のマージン。
        /// </summary>
        protected const int NodeTypeIconMarginRight = 20;

        /// <summary>
        /// 修正マークの右側のマージン。
        /// </summary>
        protected const int ModifyMarkMarginRight = 4;

        /// <summary>
        /// フィールドのコンテキストメニューに関するアップデートを行う
        /// </summary>
        private Func<bool> fieldContextMenuUpdaters;

        /// <summary>
        /// エミッタのコンテキストメニューに関するアップデートを行う
        /// </summary>
        private Func<bool> emitterContextMenuUpdaters;

        /// <summary>
        /// エミッタセットのコンテキストメニューに関するアップデートを行う
        /// </summary>
        private Func<bool> emitterSetContextMenuUpdaters;

        /// <summary>
        /// グレー表示のOn/Off
        /// </summary>
        private bool isGray = false;

        /// <summary>
        /// ノードの表示設定。
        /// </summary>
        private bool isDisplayed = true;

        /// <summary>
        /// The data context for the modification flags.
        /// This data context will be set to the caption label.
        /// </summary>
        private string modificationDataContext = "ModificationFlagViewModel";

        /// <summary>
        /// A flag that indicates whether the context menu for the tree node is created or not.
        /// </summary>
        private bool isContextMenuCreated = false;

        /// <summary>
        /// Initializes the ProjectTreeNodeBase instance.
        /// </summary>
        protected ProjectTreeNodeBase()
        {
            this.ItemContainerSelector = new ProjectTreeItemContainerSelector();
            this.EnableDisplayedIcon = true;
            this.IsDrawNodeTypeIcon = true;
            this.IsDrawModifyMark = true;
            this.TreeNodeModificationRenderer = new TreeNodeModificationRenderer(this);
            this.TreeNodeModificationRenderer.DataSourceModifiedChanged += this.OnDataSourceModifiedChanged;
            this.PropertyChanged += this.OnPropertyChanged;
            this.fieldContextMenuUpdaters = () => true;
            this.emitterContextMenuUpdaters = () => true;
            this.emitterSetContextMenuUpdaters = () => true;
            this.ReservedShaderContextMenuUpdaters = () => true;

            var contextMenu = new UIContextMenuStrip();
            contextMenu.SetLogicalParent(this);
            contextMenu.Opening += this.OnContextMenuOpening;
            this.ContextMenuStrip = contextMenu;

            this.Expand();

            this.AddBinding("ItemsSource", "Children");
            this.AddBinding("ContextMenuOpeningExecutable", "EvaluateCopyPasteAvailabilityExecutable");
        }

        /// <summary>
        /// Gets the region that displays the name of the node.
        /// </summary>
        public override Rectangle NameRectangle
        {
            get
            {
                // 描画アイコンのサイズを加算
                int locationX = ProjectTreeNodeBase.ShowButtonWidth + ProjectTreeNodeBase.ShowButtonMarginRight;

                // ノードタイプアイコンのサイズを加算
                if (this.IsDrawNodeTypeIcon)
                {
                    locationX += ProjectTreeNodeBase.NodeTypeIconMarginRight;
                }

                // 階層情報のサイズを加算
                locationX += ((this.Level + 1) * ProjectTreeNodeBase.TreeNodeHierarchyWidth) + 1;

                // 修正マークのサイズを加算
                if (this.IsDrawModifyMark && this.TreeNodeModificationRenderer.IsDataSourceModified)
                {
                    locationX += Properties.Resources.Icon_Asterisk.Width + ModifyMarkMarginRight;
                }

                // 名前の左端からクライアント領域の右端までの距離を取得
                // スクロールバーのサイズを考慮するためClientSizeを使用
                int width = this.TreeView.ClientSize.Width - locationX;

                return new Rectangle(locationX, Bounds.Y, width, Bounds.Height);
            }
        }

        /// <summary>
        /// 表示アイコンの描画領域を取得します.
        /// </summary>
        public virtual Rectangle ShowButtonRectangle
        {
            get
            {
                return new Rectangle(
                    0,
                    this.Bounds.Y,
                    ShowButtonWidth,
                    this.Bounds.Height);
            }
        }

        /// <summary>
        /// Gets the Executable instance to run when the context menu is being opened.
        /// </summary>
        public IExecutable ContextMenuOpeningExecutable { get; set; }

        /// <summary>
        /// グレー表示のOn/Offを取得または設定します。
        /// </summary>
        public virtual bool IsGray
        {
            get
            {
                return this.isGray;
            }

            set
            {
                this.isGray = value;

                // 再描画
                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate(true);
                }
            }
        }

        /// <summary>
        /// ノードの表示設定を取得または設定します。
        /// </summary>
        public virtual bool IsDisplayed
        {
            get { return this.isDisplayed; }
            set { this.LogicalTreeElementExtender.SetValue(ref this.isDisplayed, value); }
        }

        /// <summary>
        /// 親ノードの表示設定を取得します。
        /// </summary>
        public virtual bool IsParentDisplayed
        {
            get
            {
                var parent = this.Parent;
                while (parent != null)
                {
                    var parentNode = parent as ProjectTreeNodeBase;
                    if (parentNode != null &&
                        parentNode.isDisplayed == false)
                    {
                        return false;
                    }

                    parent = parent.Parent;
                }

                return true;
            }
        }

        /// <summary>
        /// 表示位置を制御するためにコンテキストメニューをオーバーライドではなく上書きで定義します。
        /// </summary>
        public new UIContextMenuStrip ContextMenuStrip { get; set; }

        /// <summary>
        /// Get or set the data context for the modification flags.
        /// (the data context for the caption label.)
        /// </summary>
        public string ModificationDataContext
        {
            get
            {
                return this.modificationDataContext;
            }

            set
            {
                if (this.modificationDataContext == value)
                {
                    return;
                }

                this.modificationDataContext = value;
                this.UpdateModificationDataContext();
            }
        }

        /// <summary>
        /// フィールドのコンテキストメニューに関するアップデートを行う
        /// </summary>
        protected Func<bool> FieldContextMenuUpdaters
        {
            get
            {
                return this.fieldContextMenuUpdaters;
            }

            set
            {
                this.fieldContextMenuUpdaters = value;
            }
        }

        /// <summary>
        /// エミッタのコンテキストメニューに関するアップデートを行う
        /// </summary>
        protected Func<bool> EmitterContextMenuUpdaters
        {
            get
            {
                return this.emitterContextMenuUpdaters;
            }

            set
            {
                this.emitterContextMenuUpdaters = value;
            }
        }

        /// <summary>
        /// エミッタセットのコンテキストメニューに関するアップデートを行う
        /// </summary>
        protected Func<bool> EmitterSetContextMenuUpdaters
        {
            get
            {
                return this.emitterSetContextMenuUpdaters;
            }

            set
            {
                this.emitterSetContextMenuUpdaters = value;
            }
        }

        /// <summary>
        /// エミッタプラグインのコンテキストメニューに関するアップデートを行う
        /// </summary>
        protected Func<bool> ReservedShaderContextMenuUpdaters { get; set; }

        /// <summary>
        /// 表示アイコンの有効/無効を取得または設定します。
        /// </summary>
        protected bool EnableDisplayedIcon { get; set; }

        /// <summary>
        /// ノードタイプアイコン描画のON/OFFを取得または設定します。
        /// </summary>
        protected bool IsDrawNodeTypeIcon { get; set; }

        /// <summary>
        /// 修正マーク描画のON/OFFを取得または設定します。
        /// </summary>
        protected bool IsDrawModifyMark { get; set; }

        /// <summary>
        /// 変更フラグ管理クラスです。
        /// </summary>
        protected TreeNodeModificationRenderer TreeNodeModificationRenderer { get; set; }

        /// <summary>
        /// Called when the mouse pointer is over the control and a mouse button is pressed.
        /// </summary>
        /// <param name="e">Event argument.</param>
        public override void OnMouseDown(MouseEventArgs e)
        {
            this.TreeView.Focus();

            // 表示アイコンが押されたとき表示設定を変更する。
            var bounds = new Rectangle(
                0,
                this.Bounds.Y,
                ShowButtonWidth,
                this.Bounds.Height);

            if (bounds.Contains(e.Location))
            {
                if (this.EnableDisplayedIcon == false || this.IsParentDisplayed == false)
                {
                    return;
                }

                this.IsDisplayed = !this.IsDisplayed;

                this.TreeView.Invalidate();
                return;
            }

            // ツリー展開ボタンが押されたときツリーを展開する。
            bounds.X += ShowButtonWidth + ShowButtonMarginRight + (this.Level * TreeNodeHierarchyWidth);
            bounds.Width = TreeNodeHierarchyWidth;

            if (bounds.Contains(e.Location))
            {
                if (this.IsExpanded)
                {
                    this.Collapse();
                }
                else
                {
                    this.Expand();
                }
            }

            // ここでノードの選択状態を反映させてからメニューを開く
            base.OnMouseDown(e);

            // コンテキストメニューの表示位置を制御するため手動で呼ぶ
            if (e.Button == MouseButtons.Right && this.ContextMenuStrip != null)
            {
                var nodePos = this.TreeView.PointToScreen(new Point(this.Bounds.X, this.Bounds.Y));

                this.ContextMenuStrip.Show(
                    new Point(Control.MousePosition.X + 1, nodePos.Y),
                    ToolStripDropDownDirection.Default);
            }
        }

        /// <summary>
        /// Called when the control is double clicked by the mouse.
        /// </summary>
        /// <param name="e">Event argument.</param>
        public override void OnMouseDoubleClick(MouseEventArgs e)
        {
            // ダブルクリックで2回目にボタンを押したとき、OnMouseDownイベントが
            // 発生しないのでそのときの処理をここで実装する。

            // 表示アイコンを押したとき表示設定を変更する。
            var bounds = new Rectangle(
                0,
                this.Bounds.Y,
                ShowButtonWidth,
                this.Bounds.Height);

            if (bounds.Contains(e.Location))
            {
                if (this.EnableDisplayedIcon == false || this.IsParentDisplayed == false)
                {
                    return;
                }

                this.IsDisplayed = !this.IsDisplayed;

                this.TreeView.Invalidate();
                return;
            }

            // ツリー展開ボタンを押したときツリーを展開する。
            bounds.X += ShowButtonWidth + ShowButtonMarginRight + (this.Level * TreeNodeHierarchyWidth);
            bounds.Width = TreeNodeHierarchyWidth;

            if (bounds.Contains(e.Location))
            {
                if (this.IsExpanded)
                {
                    this.Collapse();
                }
                else
                {
                    this.Expand();
                }
            }

            // ノードアイコンをダブルクリックしたときツリーを展開する。
            bounds.X += TreeNodeHierarchyWidth;
            bounds.Width = ShowButtonWidth;

            if (bounds.Contains(e.Location))
            {
                if (this.IsExpanded)
                {
                    this.Collapse();
                }
                else
                {
                    this.Expand();
                }
            }

            base.OnMouseDoubleClick(e);
        }

        /// <summary>
        /// ノードの複数選択時に条件チェックしつつトグルする
        /// </summary>
        /// <returns>トグル操作の結果</returns>
        public override bool ToggleMultiSelection()
        {
            // 同じノードしか選択できない
            var uiNode = this.TreeView.SelectedNode as ProjectTreeNodeBase;
            if (uiNode != null && this.DataContext.GetType() != uiNode.DataContext.GetType())
            {
                return this.IsMultiSelected;
            }

            // 同じエミッタセット内のノードしか複数選択できない
            if (!CheckSameParent(this, (ProjectTreeNodeBase)this.TreeView.SelectedNode))
            {
                return this.IsMultiSelected;
            }

            return base.ToggleMultiSelection();
        }

        /// <summary>
        /// Called when a child node is added.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected internal override void OnChildNodeAdded(TreeViewEventArgs e)
        {
            base.OnChildNodeAdded(e);
            this.Expand();
        }

        /// <summary>
        /// Called after the node has been instanced.
        /// </summary>
        protected internal override void OnInitialized()
        {
            base.OnInitialized();

            // コンテキストメニューを追加する。
            if (this.isContextMenuCreated == false)
            {
                // ビューアとモデルノードではコンテキストメニューを無効化
                if (this is ViewerTreeNode || this is ModelTreeNode)
                {
                    this.ContextMenu = null;
                    this.ContextMenuStrip = null;
                }
                else
                {
                    this.SetupContextMenu();
                }

                this.isContextMenuCreated = true;
            }
        }

        /// <summary>
        /// Setup the base node context menu.
        /// </summary>
        protected virtual void SetupContextMenu()
        {
            bool isNodeCopyPasteAllowed =
                this is EmitterTreeNode ||
                this is EmitterSetTreeNode ||
                this is FieldTreeNode ||
                this is PreviewTreeNode ||
                this is CustomActionTreeNode ||
                this is EmitterExtParamsTreeNode ||
                this is ReservedShaderTreeNode;

            bool canDuplicate =
                (this is CustomActionTreeNode) == false &&
                (this is EmitterExtParamsTreeNode) == false &&
                (this is FieldTreeNode) == false &&
                (this is ReservedShaderTreeNode) == false;

            if (isNodeCopyPasteAllowed == false)
            {
                return;
            }

            var copyMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuCopyNode,
                ShortcutKeys = Keys.Control | Keys.C,
                Image = Properties.Resources.Icon_Copy,
            };

            var pasteMenu = new UIToolStripMenuItem
            {
                Text = this is EmitterSetTreeNode
                    ? Properties.Resources.MenuPasteNode
                    : Properties.Resources.MenuPasteNodeContent,
                ShortcutKeys = Keys.Control | Keys.V,
                Image = Properties.Resources.Icon_Paste,
            };

            var duplicateMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuDuplicateNode,
                Image = Properties.Resources.Icon_DuplicateNode,
                ShortcutKeys = Keys.Control | Keys.D,
            };

            var deleteMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuDeleteNode,
                Image = Properties.Resources.Icon_Delete,
                ShortcutKeys = Keys.Delete,
            };

            if (this is PreviewTreeNode)
            {
                deleteMenu.Text = Properties.Resources.MenuCloseNode;
                deleteMenu.ShortcutKeys = Keys.Control | Keys.W;
            }

            deleteMenu.Bindings.Add(new UIControls.DataBinding.Binder(
                deleteMenu,
                "Enabled",
                "CanRemoveSelectedNode"));
            deleteMenu.DataContext = WorkspaceRootViewModel.Instance;
            deleteMenu.Click += (ss, ee) => WorkspaceRootViewModel.Instance.Controller.Remove();

            var menu = (UIContextMenuStrip)this.ContextMenuStrip;

            menu.Controls.Add(copyMenu);
            menu.Controls.Add(pasteMenu);
            if (canDuplicate == true)
            {
                menu.Controls.Add(duplicateMenu);
            }

            menu.Controls.Add(deleteMenu);

            copyMenu
                .AddBinding("Executable", "NodeCopyExecutable")
                .UpdateElement();

            pasteMenu
                .AddBinding("Executable", "NodePasteExecutable")
                .UpdateElement();

            if (canDuplicate == true)
            {
                duplicateMenu
                    .AddBinding("Executable", "NodeDuplicateExecutable")
                    .UpdateElement();

                // エミッタノードなら、複製メニューのOn/Offを切り替えるようにする。
                if (this is EmitterTreeNode)
                {
                    Binder duplicateMenuBinder = duplicateMenu.AddBinding("Enabled", "CanCreateNewEmitter");
                    this.EmitterContextMenuUpdaters += () => duplicateMenuBinder.UpdateElement();
                    duplicateMenuBinder.UpdateElement();
                }
            }
        }

        /// <summary>
        /// Draw the background.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        protected virtual void DrawBackground(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // 描画領域全体をクリア。
            gr.FillRectangle(Brushes.White, bounds);

            // 表示アイコンの裏は背景を描画しない。
            bounds.X += ShowButtonWidth;

            if ((state & TreeNodeStates.Selected) != 0)
            {
                // 選択時の背景を描画。
                gr.FillRectangle(Brushes.LightSteelBlue, bounds);
            }
            else if (this.IsMultiSelected)
            {
                gr.FillRectangle(Brushes.Lavender, bounds);
            }
            else if (this.IsDragging)
            {
                // ドラッグ中の背景を描画。
                gr.FillRectangle(Brushes.LightSkyBlue, bounds);
            }
            else
            {
                // デフォルトの背景を描画。
                gr.FillRectangle(Brushes.White, bounds);
            }
        }

        /// <summary>
        /// Draw the displayed icon.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        protected virtual void DrawDisplayedIcon(
            Rectangle bounds,
            TreeNodeStates state,
            Graphics gr)
        {
            if (this.EnableDisplayedIcon == false)
            {
                return;
            }

            Image img = Properties.Resources.Icon_View_ON;

            if (this.isDisplayed == false)
            {
                img = Properties.Resources.Icon_View_OFF;
            }
            else if (this.IsParentDisplayed == false)
            {
                img = Properties.Resources.Icon_View_Hide;
            }

            gr.DrawImage(
                img,
                bounds.Left,
                bounds.Top + ((bounds.Height - img.Height) / 2));
        }

        /// <summary>
        /// Draw the hierarchy.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <param name="locationX">The x-coordinate.</param>
        /// <returns>The next location.</returns>
        protected virtual int DrawHierarchy(
            Rectangle bounds,
            TreeNodeStates state,
            Graphics gr,
            int locationX)
        {
            Pen dotLinePen = new Pen(Color.Gray, 1);
            dotLinePen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dot;

            // ノードのレベルに合わせてインデントした座標を計算。
            locationX += this.Level * TreeNodeHierarchyWidth;

            // ノードの階層関係を描画。
            {
                // 階層関係の描画位置を計算。
                int middleX = locationX + (TreeNodeHierarchyWidth / 2);
                int middleY = bounds.Top + 8;

                if (this.Nodes.Count > 0 && this.IsExpanded == true)
                {
                    // Draw the line from the bottom of the icon to the child.
                    gr.DrawLine(
                        dotLinePen,
                        middleX + TreeNodeHierarchyWidth,
                        middleY,
                        middleX + TreeNodeHierarchyWidth,
                        bounds.Bottom);
                }

                if (this.NextNode == null)
                {
                    // Draw a dotted "┗" line.
                    gr.DrawLine(
                        dotLinePen,
                        middleX,
                        bounds.Top,
                        middleX,
                        middleY);
                    gr.DrawLine(
                        dotLinePen,
                        middleX,
                        middleY,
                        locationX + TreeNodeHierarchyWidth,
                        middleY);
                }
                else
                {
                    // Draw a dotted "┣" line.
                    gr.DrawLine(
                        dotLinePen,
                        middleX,
                        bounds.Top,
                        middleX,
                        bounds.Bottom);
                    gr.DrawLine(
                        dotLinePen,
                        middleX,
                        middleY,
                        locationX + TreeNodeHierarchyWidth,
                        middleY);
                }

                // ツリーを展開するアイコンを描画。
                if (this.Nodes.Count > 0)
                {
                    Image img = this.IsExpanded
                              ? Properties.Resources.Icon_Expanded
                              : Properties.Resources.Icon_Collapsed;

                    gr.DrawImageUnscaledAndClipped(
                        img,
                        new Rectangle(locationX + 2, bounds.Top, img.Width, img.Height));
                }
            }

            // インデントしてできた空間に親ノードの階層関係を描画。
            if (this.Level > 0)
            {
                var currNode = (TreeNode)this.Parent;
                int parentLocationX = locationX - TreeNodeHierarchyWidth;

                while (currNode != null)
                {
                    if (currNode.NextNode != null)
                    {
                        // 階層関係の描画位置を計算。
                        int middleX = parentLocationX + (TreeNodeHierarchyWidth / 2);

                        // Draw a dotted "┃" line.
                        gr.DrawLine(
                            dotLinePen,
                            middleX,
                            bounds.Top,
                            middleX,
                            bounds.Bottom);
                    }

                    currNode = currNode.Parent;
                    parentLocationX -= TreeNodeHierarchyWidth;
                }
            }

            dotLinePen.Dispose();

            return locationX + TreeNodeHierarchyWidth + 1;
        }

        /// <summary>
        /// ノードの種類に対応したアイコンを描画。
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <param name="locationX">The x-coordinate.</param>
        /// <param name="icon">The icon of node type.</param>
        /// <returns>The next location..</returns>
        protected virtual int DrawNodeTypeIcon(
            Rectangle bounds,
            TreeNodeStates state,
            Graphics gr,
            int locationX,
            Image icon)
        {
            if (icon == null)
            {
                return locationX + NodeTypeIconMarginRight;
            }

            int locationY = bounds.Top + ((bounds.Height - icon.Height) / 2);

            if (this.IsGray == false)
            {
                gr.DrawImage(
                    icon,
                    locationX,
                    locationY,
                    icon.Width,
                    icon.Height);
            }
            else
            {
                ControlPaint.DrawImageDisabled(
                    gr,
                    icon,
                    locationX,
                    locationY,
                    this.BackColor);
            }

            return locationX + NodeTypeIconMarginRight;
        }

        /// <summary>
        /// Draw the modify mark.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <param name="locationX">The x-coordinate.</param>
        /// <returns>The next location..</returns>
        protected virtual int DrawModifyMark(
            Rectangle bounds,
            TreeNodeStates state,
            Graphics gr,
            int locationX)
        {
            if (this.IsDrawModifyMark == false)
            {
                return locationX;
            }

            int nextLocation = locationX;

            // データに変更があったときスターマークを描画
            if (this.TreeNodeModificationRenderer.IsDataSourceModified)
            {
                Image img = Properties.Resources.Icon_Asterisk;
                int locationY = this.Bounds.Top + ((this.Bounds.Height - img.Height) / 2);

                // 機能がオフでもスターマークは常にカラー表示する
                gr.DrawImage(img, locationX, locationY);
                //// ControlPaint.DrawImageDisabled(gr, img, locationX, locationY, this.BackColor);

                nextLocation += img.Width + ModifyMarkMarginRight;
            }

            return nextLocation;
        }

        /// <summary>
        /// Draw the text.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <param name="locationX">The x-coordinate.</param>
        /// <param name="text">The text.</param>
        /// <param name="font">The font.</param>
        protected virtual void DrawText(
            Rectangle bounds,
            TreeNodeStates state,
            Graphics gr,
            int locationX,
            string text,
            Font font)
        {
            Brush textBrush = Brushes.Black;

            // フォントを設定。
            if (font == null)
            {
                font = TreeView.Font;
            }

            // ノード選択時はテキストの色を変える。
            if ((state & TreeNodeStates.Selected) != 0 ||
                (state & TreeNodeStates.Focused) != 0)
            {
                textBrush = Brushes.DarkBlue;
            }

            if (this.IsGray)
            {
                textBrush = Brushes.Gray;
            }

            // テキストの描画位置を計算。
            SizeF textSize = gr.MeasureString(text, font);
            int locationY = this.Bounds.Top + ((this.Bounds.Height - (int)textSize.Height) / 2);

            gr.DrawString(
                text,
                font,
                textBrush,
                locationX,
                locationY);
        }

        /// <summary>
        /// Gets the WorkspaceController instance.
        /// </summary>
        /// <returns>Returns the WorkspaceController instance, or null if not available.</returns>
        protected WorkspaceController GetWorkspaceController()
        {
            var dc = this.DataContext as IHierarchyObject;
            if (dc == null)
            {
                return null;
            }

            var wvm = dc.FindFarthestParentOfType<WorkspaceRootViewModel>();
            if (wvm == null)
            {
                return null;
            }

            return wvm.Controller;
        }

        /// <summary>
        /// 2つのノードが同じルートから派生したものかをチェックします。
        /// </summary>
        /// <param name="addedNode">ノード</param>
        /// <param name="selectedNode">選択済みのノード</param>
        /// <returns>一致していたらtrue,そうでなければfalse.</returns>
        private static bool CheckSameParent(ProjectTreeNodeBase addedNode, ProjectTreeNodeBase selectedNode)
        {
            if (selectedNode is WorkspaceTreeNode
                || selectedNode is ViewerTreeNode
                || selectedNode is ModelTreeNode)
            {
                return false;
            }

            while (!(addedNode is EmitterSetTreeNode))
            {
                addedNode = (ProjectTreeNodeBase)addedNode.Parent;
            }

            while (!(selectedNode is EmitterSetTreeNode))
            {
                selectedNode = (ProjectTreeNodeBase)selectedNode.Parent;
            }

            return addedNode == selectedNode;
        }

        /// <summary>
        /// Called when the context menu of this node is being opened.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnContextMenuOpening(object sender, CancelEventArgs e)
        {
            IExecutable exec = this.ContextMenuOpeningExecutable;
            if (exec != null)
            {
                exec.Execute(null);
            }

            // エミッタセットに関するコンテキストメニューを更新
            this.emitterSetContextMenuUpdaters();

            // エミッタに関するコンテキストメニューを更新
            this.emitterContextMenuUpdaters();

            // フィールドに関するコンテキストメニューを更新
            this.fieldContextMenuUpdaters();

            // エミッタプラグインに関するコンテキストメニューを更新
            this.ReservedShaderContextMenuUpdaters();
        }

        /// <summary>
        /// Handle PropertyChanged event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "DataContext")
            {
                this.UpdateModificationDataContext();
            }
            else if (e.PropertyName == "ItemsSource")
            {
                this.Children.Add(this.TreeNodeModificationRenderer);
            }
        }

        /// <summary>
        /// データソースの変更フラグが変わったときのイベントを処理します。
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDataSourceModifiedChanged(object sender, EventArgs e)
        {
            if (this.TreeView == null)
            {
                return;
            }

            // 再描画
            this.TreeView.Invalidate(true);
        }

        /// <summary>
        /// データコンテキストの変更状態をアップデート
        /// </summary>
        private void UpdateModificationDataContext()
        {
            // First remove all the binders that binds to the label's "DataContext" property.
            DataBinding.Binder[] bindersToRemove = this.TreeNodeModificationRenderer.Bindings.Where(
                item => item.ElementPropertyName == "DataContext").ToArray();

            // There should be at most one binding for the data context.
            if (bindersToRemove != null && bindersToRemove.Length > 1)
            {
                Logger.Log(
                    LogLevels.Error,
                    "ProjectTreeNodeBase.UpdateModificationDataContext : Data context binds to multiple data sources.");
            }

            foreach (DataBinding.Binder binder in bindersToRemove)
            {
                // If the bound data source is the same, don't replace the binding with the new one.
                if (binder.DataSourcePropertyName == this.modificationDataContext)
                {
                    return;
                }

                this.TreeNodeModificationRenderer.Bindings.Remove(binder);
            }

            // Add new binder to the data context property.
            this.TreeNodeModificationRenderer.AddBinding("DataContext", this.modificationDataContext);

            // Also update the modification flag target property bindings.
            this.UpdateModificationFlagTarget();
        }

        /// <summary>
        /// 変更フラグのターゲットをアップデート
        /// </summary>
        private void UpdateModificationFlagTarget()
        {
            // Remove old binders.
            {
                // Remove all the binders that binds to the label's "IsDataSourceModified" property.
                DataBinding.Binder[] bindersToRemove = this.TreeNodeModificationRenderer.Bindings.Where(
                    item => item.ElementPropertyName == "IsDataSourceModified").ToArray();
                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    this.TreeNodeModificationRenderer.Bindings.Remove(binder);
                }
            }

            // Add the new binders.
            {
                DataBinding.Binder binder = this.TreeNodeModificationRenderer.AddBinding(
                    "IsDataSourceModified",
                    "IsAnyValueModified");

                binder.Mode = DataBinding.BindingMode.OneWay;
            }
        }
    }
}
