﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// プリセットメニュー関連の処理を集約するクラスです。
    /// </summary>
    public static class PresetUtility
    {
        /// <summary>
        /// プリセットがあるかどうかを返します。
        /// </summary>
        public static bool HasPreset
        {
            get
            {
                var esetList = WorkspaceRootViewModel.Instance.WorkspaceViewModel.CreatePresetEmitterSetItems(false);
                return esetList.Count >= 2;
            }
        }

        /// <summary>
        /// プリセットメニューを作成して提示します。
        /// </summary>
        /// <param name="targetItem">プリセット一覧を追加するメニューアイテム</param>
        /// <param name="addedItems">プリセットに追加済みのアイテムリスト</param>
        /// <param name="needSeparater">セパレータを挿入する場合はtrue.</param>
        /// <param name="mode">表示する種別を示すモード.</param>
        /// <param name="makeChild">子どもを作る時はtrue,そうでなければfalse.</param>
        public static void SetupPresetMenu(
            UIToolStripMenuItem targetItem,
            List<ToolStripItem> addedItems,
            bool needSeparater,
            PresetMode mode,
            bool makeChild = false)
        {
            // 現在登録されているアイテムを全て削除
            foreach (var item in addedItems)
            {
                targetItem.DropDownItems.Remove(item);
                item.Dispose();
            }

            addedItems.Clear();

            // プリセットフォルダ内のエミッタセットリストを取得(1つは「未選択」なので2個以上でなければ処理しない)
            var esetList = WorkspaceRootViewModel.Instance.WorkspaceViewModel.CreatePresetEmitterSetItems(false);
            if (esetList.Count < 2)
            {
                return;
            }

            int esetIndex = 0, emitterIndex = 0;

            // エミッタセットリストごとに処理
            bool separaterAppeared = false;
            for (int i = 0; i < esetList.Count; i++)
            {
                var eset = esetList[i];

                // 「未選択」はスキップ
                if (eset.IsUnselected)
                {
                    ++esetIndex;
                    ++emitterIndex;
                    continue;
                }

                // セパレータの挿入
                if (i + 1 < esetList.Count && esetList[i + 1].SettingType == PresetSettingTypes.Project)
                {
                    if (separaterAppeared == false)
                    {
                        var sep = new ToolStripSeparator();
                        addedItems.Add(sep);
                        targetItem.DropDownItems.Add(sep);
                        separaterAppeared = true;
                        continue;
                    }
                }

                // エミッタセット名のアイテムを作成
                UIToolStripMenuItem menuItem = null;
                if (mode == PresetMode.EmitterSet)
                {
                    menuItem = targetItem;
                }
                else
                {
                    menuItem = new UIToolStripMenuItem
                    {
                        Image = separaterAppeared ? Properties.Resources.Icon_EmitterSet_Add_Proj : Properties.Resources.Icon_EmitterSet_Add,
                        Text = eset.Key
                    };
                    addedItems.Add(menuItem);
                    targetItem.DropDownItems.Add(menuItem);
                }

                // エミッタセット全体をプリセットとして作成するアイテムを追加
                if (mode == PresetMode.EmitterSetAndEmitter || mode == PresetMode.EmitterSet)
                {
                    var subItem = new UIToolStripMenuItem();
                    subItem.Bindings.Add(
                        new UIControls.DataBinding.Binder(subItem, "Enabled", "CanCreateNewEmitterSet"));
                    subItem.DataContext = WorkspaceRootViewModel.Instance;

                    subItem.Image = separaterAppeared ? Properties.Resources.Icon_EmitterSet_Add_Proj : Properties.Resources.Icon_EmitterSet_Add;
                    subItem.Text = eset.Key;
                    int tmp = esetIndex;
                    subItem.Click += (s, ee) =>
                    {
                        WorkspaceRootViewModel.Instance.WorkspaceViewModel.OnCreateNewEmitterSet(tmp);
                    };
                    menuItem.DropDownItems.Add(subItem);
                    if (mode == PresetMode.EmitterSet)
                    {
                        addedItems.Add(subItem);
                    }
                }

                ++esetIndex;

                if (mode == PresetMode.EmitterSet)
                {
                    continue;
                }

                // エミッタセット内のエミッタをリストとして取得し、サブアイテムとして追加
                List<string> emitterList =
                    WorkspaceRootViewModel.Instance.WorkspaceViewModel.EnumerateEmitterFromFileFast((string)eset.Value);
                foreach (var emitter in emitterList)
                {
                    var subEmitterItem = new UIToolStripMenuItem();
                    subEmitterItem.Bindings.Add(
                        new UIControls.DataBinding.Binder(subEmitterItem, "Enabled", "CanCreateNewEmitter"));
                    subEmitterItem.DataContext = WorkspaceRootViewModel.Instance;
                    subEmitterItem.Image = separaterAppeared ? Properties.Resources.Icon_Emitter_Simple_Add_Proj : Properties.Resources.Icon_Emitter_Simple_Add;
                    subEmitterItem.Text = emitter;
                    int tmpEmitter = emitterIndex;
                    if (makeChild)
                    {
                        subEmitterItem.Click += (s, ee) =>
                        {
                            WorkspaceRootViewModel.Instance.WorkspaceViewModel.OnCreateNewChildEmitter(tmpEmitter);
                        };
                    }
                    else
                    {
                        subEmitterItem.Click += (s, ee) =>
                        {
                            WorkspaceRootViewModel.Instance.WorkspaceViewModel.OnCreateNewEmitter(tmpEmitter);
                        };
                    }

                    menuItem.DropDownItems.Add(subEmitterItem);
                    ++emitterIndex;
                }
            }
        }
    }
}
