﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.DataModel;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// An extended ProjectTreeNodeBase for emitter implementation.
    /// </summary>
    public class FieldTreeNode : ProjectTreeNodeBase
    {
        /// <summary>
        /// フィールドのアイコン.
        /// </summary>
        private Bitmap fieldIcon = null;

        /// <summary>
        /// 機能の有効 / 無効フラグです。
        /// </summary>
        private bool enableConvert;

        /// <summary>
        /// Initializes the EmitterTreeNode instance.
        /// </summary>
        public FieldTreeNode()
        {
            this.IsDrawModifyMark = true;

            this.AddBinding("Text", "Name");
            this.AddBinding("EnableConvert", "EnableConvert");
        }

        /// <summary>
        /// グレー表示のOn/Offを取得または設定します。
        /// </summary>
        public override bool IsGray
        {
            get
            {
                var parent = this.Parent as EmitterTreeNode;
                if (parent != null)
                {
                    return parent.IsGray || !this.EnableConvert;
                }
                else
                {
                    return !this.EnableConvert;
                }
            }

            set
            {
                throw new NotImplementedException();
            }
        }

        /// <summary>
        /// バイナリコンバートを有効にするかどうかのフラグを取得または設定します。
        /// </summary>
        public bool EnableConvert
        {
            get
            {
                return this.enableConvert;
            }

            set
            {
                this.LogicalTreeElementExtender.SetValue(ref this.enableConvert, value);

                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// 指定したビューモデルに対応するアイコンをセットする.
        /// </summary>
        /// <param name="viewModel">ビューモデル.</param>
        public void SetIcon(FieldViewModel viewModel)
        {
            if (viewModel as AddLocationViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_AddPos;
            }
            else if (viewModel as CollisionViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_Collision;
            }
            else if (viewModel as ConvergeViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_Convergence;
            }
            else if (viewModel as CurlNoiseViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_CurlNoise;
            }
            else if (viewModel as MagnetViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_Magness;
            }
            else if (viewModel as RandomViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_Wave;
            }
            else if (viewModel as RandomFe1ViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_Random;
            }
            else if (viewModel as SpinViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_Spin;
            }
            else if (viewModel as CustomViewModel != null)
            {
                this.fieldIcon = Properties.Resources.Icon_Field_Custom;
            }

            // 白(RGB(255, 255, 255))を透過色として処理.
            this.fieldIcon.MakeTransparent(Color.White);
        }

        /// <summary>
        /// Render the node of viewer.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <returns>Returns true if it self render, or false to tall parent
        /// TreeView to perform default node rendering.</returns>
        protected internal override bool Render(
            Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // ノード作成時に無効な領域の描画呼び出しが発生する
            if (bounds.Width == 0 || bounds.Height == 0)
            {
                return true;
            }

            this.DrawBackground(bounds, state, gr);

////            this.DrawDisplayedIcon(bounds, state, gr);

            int locationX = ProjectTreeNodeBase.ShowButtonWidth +
                            ProjectTreeNodeBase.ShowButtonMarginRight +
                            bounds.Left;

            locationX = this.DrawHierarchy(bounds, state, gr, locationX);

            if (this.fieldIcon != null)
            {
                locationX = this.DrawNodeTypeIcon(
                    bounds, state, gr, locationX, this.fieldIcon);
            }

            locationX = this.DrawModifyMark(bounds, state, gr, locationX);

            this.DrawText(bounds, state, gr, locationX, this.Text, this.NodeFont);

            return true;
        }

        /// <summary>
        /// コンテキストメニューのセットアップ.
        /// </summary>
        protected override void SetupContextMenu()
        {
            var docMenu = this.ContextMenuStrip as UIContextMenuStrip;

            // 機能の有効 / 無効
            var tsmiEnableConvert = new UIToolStripMenuItem();
            tsmiEnableConvert.Text = Properties.Resources.MenuItemNameEnableConvert;
            tsmiEnableConvert.Image = Properties.Resources.Icon_Menu_Enable;

            // ランダム
            var tsmiRandomFe1 = new UIToolStripMenuItem();
            tsmiRandomFe1.Text = Properties.Resources.MenuItemNameRandomFe1;
            tsmiRandomFe1.Image = Properties.Resources.Icon_Field_Random;
            tsmiRandomFe1.ImageTransparentColor = Color.White;

            // 新ランダム
            var tsmiRandom = new UIToolStripMenuItem();
            tsmiRandom.Text = Properties.Resources.MenuItemNameRandom;
            tsmiRandom.Image = Properties.Resources.Icon_Field_Wave;
            tsmiRandom.ImageTransparentColor = Color.White;

            // 磁力
            var tsmiMagnet = new UIToolStripMenuItem();
            tsmiMagnet.Text = Properties.Resources.MenuItemNameMagness;
            tsmiMagnet.Image = Properties.Resources.Icon_Field_Magness;
            tsmiMagnet.ImageTransparentColor = Color.White;

            // スピン
            var tsmiSpin = new UIToolStripMenuItem();
            tsmiSpin.Text = Properties.Resources.MenuItemNameSpin;
            tsmiSpin.Image = Properties.Resources.Icon_Field_Spin;
            tsmiSpin.ImageTransparentColor = Color.White;

            // 収束
            var tsmiConverge = new UIToolStripMenuItem();
            tsmiConverge.Text = Properties.Resources.MenuItemNameConvergence;
            tsmiConverge.Image = Properties.Resources.Icon_Field_Convergence;
            tsmiConverge.ImageTransparentColor = Color.White;

            // 位置に加算
            var tsmiAddLocation = new UIToolStripMenuItem();
            tsmiAddLocation.Text = Properties.Resources.MenuItemNameAddPos;
            tsmiAddLocation.Image = Properties.Resources.Icon_Field_AddPos;
            tsmiAddLocation.ImageTransparentColor = Color.White;

            // シンプルコリジョン
            var tsmiCollision = new UIToolStripMenuItem();
            tsmiCollision.Text = Properties.Resources.MenuItemNameCollision;
            tsmiCollision.Image = Properties.Resources.Icon_Field_Collision;
            tsmiCollision.ImageTransparentColor = Color.White;

            // カールノイズ
            var tsmiCurlNoise = new UIToolStripMenuItem();
            tsmiCurlNoise.Text = Properties.Resources.MenuItemNameCurlNoise;
            tsmiCurlNoise.Image = Properties.Resources.Icon_Field_CurlNoise;
            tsmiCurlNoise.ImageTransparentColor = Color.White;

            // カスタムフィールド
            var tsmiCustomField = new UIToolStripMenuItem();
            tsmiCustomField.Text = Properties.Resources.MenuItemNameCustomField;
            tsmiCustomField.Image = Properties.Resources.Icon_Field_Custom;
            tsmiCustomField.ImageTransparentColor = Color.White;

            docMenu.Items.AddRange(
                new ToolStripItem[]
                {
                    tsmiEnableConvert,
                    tsmiRandomFe1,
                    tsmiRandom,
                    tsmiMagnet,
                    tsmiSpin,
                    tsmiConverge,
                    tsmiAddLocation,
                    tsmiCollision,
                    tsmiCurlNoise,
                    tsmiCustomField,
                    new ToolStripSeparator(),
                });

            // コンテキストメニューの更新と、クリック時の挙動について設定する.

            // 機能の有効 / 無効
            tsmiEnableConvert.ShortcutKeys = Keys.Control | Keys.Shift | Keys.H;
            tsmiEnableConvert.Click += (s, e) =>
            {
                this.EnableConvert = !this.EnableConvert;
            };

            // ランダムフィールド
            Binder randomFe1Binder = tsmiRandomFe1.AddBinding("Enabled", "CanCreateNewRandomFe1Field");
            tsmiRandomFe1.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.RandomFe1);
            };
            this.FieldContextMenuUpdaters += () => randomFe1Binder.UpdateElement();
            randomFe1Binder.UpdateElement();

            // 新ランダムフィールド
            Binder randomBinder = tsmiRandom.AddBinding("Enabled", "CanCreateNewRandomField");
            tsmiRandom.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Random);
            };
            this.FieldContextMenuUpdaters += () => randomBinder.UpdateElement();
            randomBinder.UpdateElement();

            // 磁力フィールド
            Binder magnetBinder = tsmiMagnet.AddBinding("Enabled", "CanCreateNewMagnetField");
            tsmiMagnet.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Magnet);
            };
            this.FieldContextMenuUpdaters += () => magnetBinder.UpdateElement();
            magnetBinder.UpdateElement();

            // スピンフィールド
            Binder spinBinder = tsmiSpin.AddBinding("Enabled", "CanCreateNewSpinField");
            tsmiSpin.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Spin);
            };
            this.FieldContextMenuUpdaters += () => spinBinder.UpdateElement();
            spinBinder.UpdateElement();

            // 収束フィールド
            Binder convergeBinder = tsmiConverge.AddBinding("Enabled", "CanCreateNewConvergeField");
            tsmiConverge.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Converge);
            };
            this.FieldContextMenuUpdaters += () => convergeBinder.UpdateElement();
            convergeBinder.UpdateElement();

            // "位置に加算"フィールド
            Binder addLocationBinder = tsmiAddLocation.AddBinding("Enabled", "CanCreateNewAddLocationField");
            tsmiAddLocation.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.AddLocation);
            };
            this.FieldContextMenuUpdaters += () => addLocationBinder.UpdateElement();
            addLocationBinder.UpdateElement();

            // シンプルコリジョンフィールド
            Binder collisionBinder = tsmiCollision.AddBinding("Enabled", "CanCreateNewCollisionField");
            tsmiCollision.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Collision);
            };
            this.FieldContextMenuUpdaters += () => collisionBinder.UpdateElement();
            collisionBinder.UpdateElement();

            // カールノイズフィールド
            Binder curlNoiseBinder = tsmiCurlNoise.AddBinding("Enabled", "CanCreateNewCurlNoiseField");
            tsmiCurlNoise.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.CurlNoise);
            };
            this.FieldContextMenuUpdaters += () => curlNoiseBinder.UpdateElement();
            curlNoiseBinder.UpdateElement();

            // カスタムフィールド
            Binder customFieldBinder = tsmiCustomField.AddBinding("Enabled", "CanCreateNewCustomField");
            tsmiCustomField.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.CustomField);
            };
            this.FieldContextMenuUpdaters += () => customFieldBinder.UpdateElement();
            customFieldBinder.UpdateElement();

            base.SetupContextMenu();
        }
    }
}
