﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Input;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// エミッタのツリーノードです。
    /// </summary>
    public class EmitterTreeNode : ProjectTreeNodeBase
    {
        /// <summary>
        /// プロセスタイプです。
        /// </summary>
        private int processType;

        /// <summary>
        /// 実ファイルの存在しないアセットがあるかどうかのフラグです.
        /// 子エミッタの状態も含めます.
        /// </summary>
        private bool hasUnreachableAsset;

        /// <summary>
        /// 不正なアセットがあるかどうかのフラグです.
        /// 子エミッタの状態も含めます.
        /// </summary>
        private bool hasInvalidAsset;

        /// <summary>
        /// 機能の有効 / 無効フラグです。
        /// </summary>
        private bool enableConvert = true;

        /// <summary>
        /// コンテキストメニューに追加したプリセットアイテム.
        /// </summary>
        private List<ToolStripItem> addedItems = new List<ToolStripItem>();

        /// <summary>
        /// コンテキストメニューに追加したチャイルドプリセットアイテム.
        /// </summary>
        private List<ToolStripItem> addedChildItems = new List<ToolStripItem>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EmitterTreeNode()
        {
            this.CanRename = true;

            this.IsDrawModifyMark = true;

            this.AddBinding("Text", "Name");
            this.AddBinding("IsDisplayed", "Displayed");
            this.AddBinding("ProcessType", "ProcessType");
            this.AddBinding("HasUnreachableAsset", "HasUnreachableAsset");
            this.AddBinding("HasInvalidAsset", "HasInvalidAsset");
            this.AddBinding("AvailableReservedShaders", "AvailableReservedShaders");
            this.AddBinding("EnableConvert", "EnableConvert");
        }

        /// <summary>
        /// Enumerates the available reserved shader items.
        /// (each item contains the name, the icon image, and the user data info)
        /// </summary>
        public IEnumerable<Tuple<string, Image, object>> AvailableReservedShaders { get; set; }

        /// <summary>
        /// グレー表示のOn/Offを取得または設定します。
        /// </summary>
        public override bool IsGray
        {
            get
            {
                var parent = this.Parent as EmitterTreeNode;
                if (parent != null)
                {
                    return parent.IsGray || !this.EnableConvert;
                }
                else
                {
                    return !this.EnableConvert;
                }
            }

            set
            {
                throw new NotImplementedException();
            }
        }

        /// <summary>
        /// バイナリコンバートを有効にするかどうかのフラグを取得または設定します。
        /// </summary>
        public bool EnableConvert
        {
            get
            {
                return this.enableConvert;
            }

            set
            {
                this.LogicalTreeElementExtender.SetValue(ref this.enableConvert, value);

                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// 実ファイルの存在しないアセットがあるかどうかのフラグを取得または設定します.
        /// </summary>
        public bool HasUnreachableAsset
        {
            get
            {
                return this.hasUnreachableAsset;
            }

            set
            {
                if (value == this.hasUnreachableAsset)
                {
                    return;
                }

                this.hasUnreachableAsset = value;

                // TreeViewがnullになることがあるのでチェック必須
                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// 不正なアセットがあるかどうかのフラグを取得または設定します.
        /// </summary>
        public bool HasInvalidAsset
        {
            get
            {
                return this.hasInvalidAsset;
            }

            set
            {
                if (value == this.HasInvalidAsset)
                {
                    return;
                }

                this.hasInvalidAsset = value;

                // TreeViewがnullになることがあるのでチェック必須
                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// プロセスタイプを取得または設定します。
        /// </summary>
        public int ProcessType
        {
            get
            {
                return this.processType;
            }

            set
            {
                if (value == this.ProcessType)
                {
                    return;
                }

                this.processType = value;

                // TreeViewがnullになることがあるのでチェック必須
                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// Render the node of viewer.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <returns>Returns true if it self render, or false to tall parent
        /// TreeView to perform default node rendering.</returns>
        protected internal override bool Render(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // The tree node might already be removed between the draw message is dispatched
            // and the actual rendering.
            // (the message dispatches, and the rendering happens on the next application loop)
            if (((TreeNode)this).Parent == null &&
                (this.TreeView == null || this.TreeView.Nodes.Contains(this) == false))
            {
                return true;
            }

            // ノード作成時に無効な領域の描画呼び出しが発生する
            if (bounds.Width == 0 || bounds.Height == 0)
            {
                return true;
            }

            this.DrawBackground(bounds, state, gr);

            this.DrawDisplayedIcon(bounds, state, gr);

            int locationX = ProjectTreeNodeBase.ShowButtonWidth +
                            ProjectTreeNodeBase.ShowButtonMarginRight +
                            bounds.Left;

            locationX = this.DrawHierarchy(bounds, state, gr, locationX);

            Bitmap processIcon = null;

            // プロセスアイコンを取得
            if (this.ProcessType == 0)
            {
                // CPU
                processIcon = Properties.Resources.Icon_CPU;
            }
            else if (this.ProcessType == 1)
            {
                // GPU
                processIcon = Properties.Resources.Icon_GPU;
            }
            else if (this.ProcessType == 2)
            {
                // GPU + SO
                processIcon = Properties.Resources.Icon_SO;
            }

            Bitmap emitterIcon = Properties.Resources.Icon_Emitter_Simple;

            // アセットの状況に対応するノードアイコンを取得
            if (this.HasUnreachableAsset)
            {
                emitterIcon = Properties.Resources.Icon_Emitter_Simple_NoTex;
            }
            else if (this.HasInvalidAsset)
            {
                emitterIcon = Properties.Resources.Icon_Emitter_Simple_Warning;
            }

            // アイコンを描画
            locationX = this.DrawNodeTypeIcon(bounds, state, gr, locationX, emitterIcon, processIcon);

            locationX = this.DrawModifyMark(bounds, state, gr, locationX);

            this.DrawText(bounds, state, gr, locationX, this.Text, this.NodeFont);

            return true;
        }

        /// <summary>
        /// ノードの種類に対応したアイコンを描画。
        /// </summary>
        /// <param name="bounds">描画領域</param>
        /// <param name="state">ステータス</param>
        /// <param name="gr">Graphicsインスタンス</param>
        /// <param name="locationX">X座標</param>
        /// <param name="emitterIcon">エミッタアイコン</param>
        /// <param name="processIcon">プロセスアイコン</param>
        /// <returns>次の描画座標</returns>
        protected virtual int DrawNodeTypeIcon(
            Rectangle bounds,
            TreeNodeStates state,
            Graphics gr,
            int locationX,
            Image emitterIcon,
            Image processIcon)
        {
            System.Diagnostics.Debug.Assert(emitterIcon != null, "エミッタアイコンがnull");

            int locationY = bounds.Top + ((bounds.Height - emitterIcon.Height) / 2);

            if (this.IsGray == false)
            {
                gr.DrawImage(emitterIcon, locationX, locationY, emitterIcon.Width, emitterIcon.Height);

                // プロセスアイコンをエミッタアイコンの上に描画
                if (processIcon != null)
                {
                    gr.DrawImage(processIcon, locationX - 5, locationY - 1, processIcon.Width, processIcon.Height);
                }
            }
            else
            {
                ControlPaint.DrawImageDisabled(gr, emitterIcon, locationX, locationY, this.BackColor);

                // プロセスアイコンをエミッタアイコンの上に描画
                if (processIcon != null)
                {
                    ControlPaint.DrawImageDisabled(gr, processIcon, locationX - 5, locationY - 1, this.BackColor);
                }
            }

            return locationX + ProjectTreeNodeBase.NodeTypeIconMarginRight;
        }

        /// <summary>
        /// コンテキストメニューのセットアップ.
        /// </summary>
        protected override void SetupContextMenu()
        {
            var docMenu = this.ContextMenuStrip as UIContextMenuStrip;
            System.Diagnostics.Debug.Assert(docMenu != null, "ContextMenuStrip must be UIContextMenuStrip.");

            var menuItems = new List<ToolStripItem>();

            // 機能の有効 / 無効
            var tsmiEnableConvert = new UIToolStripMenuItem();
            docMenu.Items.Add(tsmiEnableConvert);
            tsmiEnableConvert.Text = Properties.Resources.MenuItemNameEnableConvert;
            tsmiEnableConvert.ShortcutKeys = Keys.Control | Keys.Shift | Keys.H;
            tsmiEnableConvert.Image = Properties.Resources.Icon_Menu_Enable;
            tsmiEnableConvert.Click += (s, e) => this.EnableConvert = !this.EnableConvert;
            tsmiEnableConvert.Bindings.Add(new UIControls.DataBinding.Binder(
                tsmiEnableConvert,
                "Enabled",
                "CanSwitchConvertFlag"));
            tsmiEnableConvert.DataContext = WorkspaceRootViewModel.Instance;

            // プリセット対応
            var addPresetMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddFromPreset,
                Image = Properties.Resources.Icon_Emitter_Preset,
                ImageTransparentColor = Color.White,
            };
            menuItems.Add(addPresetMenu);
            var dummyItem = new UIToolStripMenuItem();
            this.addedItems.Add(dummyItem);
            addPresetMenu.DropDownItems.Add(dummyItem);
            docMenu.Opening += (s, e) =>
            {
                var hasPreset = PresetUtility.HasPreset;
                if (hasPreset)
                {
                    PresetUtility.SetupPresetMenu(
                        addPresetMenu, this.addedItems, false, PresetMode.Emitter);
                }

                addPresetMenu.Visible = hasPreset;
            };

            // チャイルドプリセット対応
            var addChildPresetMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddChildFromPreset,
                Image = Properties.Resources.Icon_Emitter_Child_Preset,
                ImageTransparentColor = Color.White,
            };
            menuItems.Add(addChildPresetMenu);
            var dummyItem2 = new UIToolStripMenuItem();
            this.addedChildItems.Add(dummyItem2);
            addChildPresetMenu.DropDownItems.Add(dummyItem2);
            docMenu.Opening += (s, e) =>
            {
                var hasPreset = PresetUtility.HasPreset;
                if (hasPreset)
                {
                    PresetUtility.SetupPresetMenu(
                        addChildPresetMenu, this.addedChildItems, false, PresetMode.Emitter, true);
                }

                addChildPresetMenu.Visible = hasPreset;
            };

            // エミッタ
            var tsmiAddEmitter = new UIToolStripMenuItem();
            menuItems.Add(tsmiAddEmitter);
            tsmiAddEmitter.Text = Properties.Resources.MenuItemNameAddEmitter;
            tsmiAddEmitter.Image = Properties.Resources.Icon_Emitter_Simple;
            tsmiAddEmitter.ImageTransparentColor = Color.White;
            tsmiAddEmitter.ShortcutKeys = Keys.Control | Keys.Shift | Keys.N;
            tsmiAddEmitter.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewEmitter();
            };

            // 子エミッタ
            var tsmiAddChildEmitter = new UIToolStripMenuItem();
            menuItems.Add(tsmiAddChildEmitter);
            tsmiAddChildEmitter.Text = Properties.Resources.MenuItemNameAddChildEmitter;
            tsmiAddChildEmitter.Image = Properties.Resources.Icon_Emitter_Child;
            tsmiAddChildEmitter.ImageTransparentColor = Color.White;
            tsmiAddChildEmitter.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewChildEmitter();
            };

            // 階層化の解除
            var tsmiUnparentEmitter = new UIToolStripMenuItem();
            menuItems.Add(tsmiUnparentEmitter);
            tsmiUnparentEmitter.Text = Properties.Resources.MenuItemNameUnparentEmitter;
            tsmiUnparentEmitter.Image = Properties.Resources.Icon_Emitter_LiftingChild;
            tsmiUnparentEmitter.ImageTransparentColor = Color.White;
            tsmiUnparentEmitter.ShortcutKeys = Keys.Control | Keys.Shift | Keys.P;
            tsmiUnparentEmitter.Click += (s, e) =>
            {
                this.GetWorkspaceController().UnparentEmitter();
            };

            // ================= Separator =================
            menuItems.Add(new ToolStripSeparator());

            // フィールド
            var tsmiField = new UIToolStripMenuItem();
            menuItems.Add(tsmiField);
            tsmiField.Text = Properties.Resources.MenuItemAddField;
            tsmiField.Image = Properties.Resources.Icon_Field;
            tsmiField.ImageTransparentColor = Color.White;
            docMenu.Opening += (sender, args) =>
            {
                tsmiField.Enabled = tsmiField.Children.OfType<UIToolStripMenuItem>()
                    .Aggregate(false, (current, item) => current | item.Enabled);
            };

            // ランダム
            var tsmiRandomFe1 = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiRandomFe1);
            tsmiRandomFe1.Text = Properties.Resources.MenuItemNameRandomFe1;
            tsmiRandomFe1.Image = Properties.Resources.Icon_Field_Random;
            tsmiRandomFe1.ImageTransparentColor = Color.White;
            tsmiRandomFe1.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.RandomFe1);
            };

            // 新ランダム
            var tsmiRandom = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiRandom);
            tsmiRandom.Text = Properties.Resources.MenuItemNameRandom;
            tsmiRandom.Image = Properties.Resources.Icon_Field_Wave;
            tsmiRandom.ImageTransparentColor = Color.White;
            tsmiRandom.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Random);
            };

            // 磁力追加
            var tsmiMagnet = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiMagnet);
            tsmiMagnet.Text = Properties.Resources.MenuItemNameMagness;
            tsmiMagnet.Image = Properties.Resources.Icon_Field_Magness;
            tsmiMagnet.ImageTransparentColor = Color.White;
            tsmiMagnet.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Magnet);
            };

            // スピン
            var tsmiSpin = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiSpin);
            tsmiSpin.Text = Properties.Resources.MenuItemNameSpin;
            tsmiSpin.Image = Properties.Resources.Icon_Field_Spin;
            tsmiSpin.ImageTransparentColor = Color.White;
            tsmiSpin.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Spin);
            };

            // 収束
            var tsmiConverge = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiConverge);
            tsmiConverge.Text = Properties.Resources.MenuItemNameConvergence;
            tsmiConverge.Image = Properties.Resources.Icon_Field_Convergence;
            tsmiConverge.ImageTransparentColor = Color.White;
            tsmiConverge.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Converge);
            };

            // 位置に加算
            var tsmiAddLocation = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiAddLocation);
            tsmiAddLocation.Text = Properties.Resources.MenuItemNameAddPos;
            tsmiAddLocation.Image = Properties.Resources.Icon_Field_AddPos;
            tsmiAddLocation.ImageTransparentColor = Color.White;
            tsmiAddLocation.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.AddLocation);
            };

            // シンプルコリジョン
            var tsmiCollision = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiCollision);
            tsmiCollision.Text = Properties.Resources.MenuItemNameCollision;
            tsmiCollision.Image = Properties.Resources.Icon_Field_Collision;
            tsmiCollision.ImageTransparentColor = Color.White;
            tsmiCollision.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.Collision);
            };

            // カールノイズ
            var tsmiCurlNoise = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiCurlNoise);
            tsmiCurlNoise.Text = Properties.Resources.MenuItemNameCurlNoise;
            tsmiCurlNoise.Image = Properties.Resources.Icon_Field_CurlNoise;
            tsmiCurlNoise.ImageTransparentColor = Color.White;
            tsmiCurlNoise.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.CurlNoise);
            };

            // カスタムフィールド
            var tsmiCustomField = new UIToolStripMenuItem();
            tsmiField.Children.Add(tsmiCustomField);
            tsmiCustomField.Text = Properties.Resources.MenuItemNameCustomField;
            tsmiCustomField.Image = Properties.Resources.Icon_Field_Custom;
            tsmiCustomField.ImageTransparentColor = Color.White;
            tsmiCustomField.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.CreateNewField(FieldTypes.CustomField);
            };

            // ストライプ
            var tsmiStripe = new UIToolStripMenuItem();
            menuItems.Add(tsmiStripe);
            tsmiStripe.Text = Properties.Resources.MenuItemAddStripe;
            tsmiStripe.Image = Properties.Resources.Icon_Stripe;
            tsmiStripe.ImageTransparentColor = Color.White;
            docMenu.Opening += (sender, args) =>
            {
                tsmiStripe.Enabled = tsmiStripe.Children.OfType<UIToolStripMenuItem>().
                    Aggregate(false, (current, item) => current | item.Enabled);
            };

            // Reserved shader user data items.
            var reservedShaderMenuItems = new List<UIToolStripMenuItem>();
            if (this.AvailableReservedShaders != null)
            {
                foreach (var item in this.AvailableReservedShaders)
                {
                    var tsmiReservedShader = new UIToolStripMenuItem();
                    if (item.Item1.Contains("Stripe") || item.Item1.Contains("ストライプ"))
                    {
                        tsmiStripe.Children.Add(tsmiReservedShader);
                    }
                    else
                    {
                        menuItems.Add(tsmiReservedShader);
                    }

                    reservedShaderMenuItems.Add(tsmiReservedShader);
                    tsmiReservedShader.Text = string.Format(Properties.Resources.NodeEmitterPluginText, item.Item1);
                    tsmiReservedShader.Image = item.Item2;
                    tsmiReservedShader.ExecutionParameter = item.Item3;
                }
            }

            // ================= Separator =================
            menuItems.Add(new ToolStripSeparator());

            // カスタムアクション
            var tsmiCustomAction = new UIToolStripMenuItem();
            menuItems.Add(tsmiCustomAction);
            tsmiCustomAction.Text = Properties.Resources.MenuItemNameCustomAction;
            tsmiCustomAction.Image = Properties.Resources.Icon_CustomAction;

            // エミッタ拡張パラメータ
            var tsmiEmitterExtParams = new UIToolStripMenuItem();
            menuItems.Add(tsmiEmitterExtParams);
            tsmiEmitterExtParams.Text = Properties.Resources.MenuItemNameEmitterExtParams;
            tsmiEmitterExtParams.Image = Properties.Resources.Icon_EmitterExtParams;

            // ================= Separator =================
            menuItems.Add(new ToolStripSeparator());

            // Add all the menu items to the context menu.
            docMenu.Items.AddRange(menuItems.ToArray());

            // フィールドとストライプのメニューのDataContextを伝搬させる
            foreach (var item in tsmiField.Children.OfType<UIToolStripMenuItem>())
            {
                item.DataContext = tsmiField.DataContext;
            }

            foreach (var item in tsmiStripe.Children.OfType<UIToolStripMenuItem>())
            {
                item.DataContext = tsmiStripe.DataContext;
            }

            // 共通メニュー
            base.SetupContextMenu();

            // 名前を変更
            var tsmiRename = new UIToolStripMenuItem();
            tsmiRename.Text = Properties.Resources.MenuItemNameRename;
            tsmiRename.ShortcutKeys = Keys.F2;
            tsmiRename.Image = Properties.Resources.Icon_ChangeName;
            tsmiRename.Click += (s, e) =>
            {
                var treeView = this.TreeView as WorkspaceTreeView;
                if (treeView != null)
                {
                    treeView.BeginRename();
                }
            };

            // 「名前を変更」を共通メニューの削除の前に挿入
            docMenu.Items.Insert(docMenu.Items.Count - 1, tsmiRename);

            // エミッタを「子」として貼り付け
            var tsmiPasteAsChild = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuPasteNodeAsChild,
                Image = Properties.Resources.Icon_Paste,
            };
            tsmiPasteAsChild.Click += (s, e) =>
            {
                WorkspaceController wc = this.GetWorkspaceController();
                wc.PasteChildOfSelectedNode();
            };

            // 「エミッタを「子」として貼り付け」を共通メニューの「複製」の前に挿入
            docMenu.Items.Insert(docMenu.Items.Count - 3, tsmiPasteAsChild);
            Binder pasteAsChildBinder = tsmiPasteAsChild.AddBinding("Enabled", "CanPasteEmitterAsChild");
            this.EmitterContextMenuUpdaters += pasteAsChildBinder.UpdateElement;
            pasteAsChildBinder.UpdateElement();

            tsmiCustomAction.AddBinding("Executable", "CustomActionMenuExecutable").UpdateElement();
            tsmiEmitterExtParams.AddBinding("Executable", "EmitterExtParamsMenuExecutable").UpdateElement();

            // コンテキストメニューを表示する度に、メニューのOn/Offをチェックするようにする.
            Binder addEmitterBinder = tsmiAddEmitter.AddBinding("Enabled", "CanCreateNewEmitter");
            Binder addChildEmitterBinder = tsmiAddChildEmitter.AddBinding("Enabled", "CanCreateNewChildEmitter");
            Binder addPresetEmitterBinder = addPresetMenu.AddBinding("Enabled", "CanCreateNewEmitter");
            Binder addChildPresetEmitterBinder = addChildPresetMenu.AddBinding("Enabled", "CanCreateNewChildEmitter");
            Binder unparentEmitterBinder = tsmiUnparentEmitter.AddBinding("Enabled", "CanUnparentEmitter");
            this.EmitterContextMenuUpdaters += addEmitterBinder.UpdateElement;
            this.EmitterContextMenuUpdaters += addChildEmitterBinder.UpdateElement;
            this.EmitterContextMenuUpdaters += addPresetEmitterBinder.UpdateElement;
            this.EmitterContextMenuUpdaters += addChildPresetEmitterBinder.UpdateElement;
            this.EmitterContextMenuUpdaters += unparentEmitterBinder.UpdateElement;
            addEmitterBinder.UpdateElement();
            addChildEmitterBinder.UpdateElement();
            addPresetEmitterBinder.UpdateElement();
            addChildPresetEmitterBinder.UpdateElement();
            unparentEmitterBinder.UpdateElement();

            foreach (var item in reservedShaderMenuItems)
            {
                item.AddBinding("Executable", "ReservedShaderMenuExecutable").UpdateElement();
            }

            // コンテキストメニューを表示する度に、メニューのOn/Offをチェックするようにする.
            Binder randomFe1Binder = tsmiRandomFe1.AddBinding("Enabled", "CanCreateNewRandomFe1Field");
            Binder randomBinder = tsmiRandom.AddBinding("Enabled", "CanCreateNewRandomField");
            Binder magnetBinder = tsmiMagnet.AddBinding("Enabled", "CanCreateNewMagnetField");
            Binder spinBinder = tsmiSpin.AddBinding("Enabled", "CanCreateNewSpinField");
            Binder convergeBinder = tsmiConverge.AddBinding("Enabled", "CanCreateNewConvergeField");
            Binder addLocationBinder = tsmiAddLocation.AddBinding("Enabled", "CanCreateNewAddLocationField");
            Binder collisionBinder = tsmiCollision.AddBinding("Enabled", "CanCreateNewCollisionField");
            Binder curlNoiseBinder = tsmiCurlNoise.AddBinding("Enabled", "CanCreateNewCurlNoiseField");
            Binder customFieldBinder = tsmiCustomField.AddBinding("Enabled", "CanCreateNewCustomField");
            this.FieldContextMenuUpdaters += randomFe1Binder.UpdateElement;
            this.FieldContextMenuUpdaters += randomBinder.UpdateElement;
            this.FieldContextMenuUpdaters += magnetBinder.UpdateElement;
            this.FieldContextMenuUpdaters += spinBinder.UpdateElement;
            this.FieldContextMenuUpdaters += convergeBinder.UpdateElement;
            this.FieldContextMenuUpdaters += addLocationBinder.UpdateElement;
            this.FieldContextMenuUpdaters += collisionBinder.UpdateElement;
            this.FieldContextMenuUpdaters += curlNoiseBinder.UpdateElement;
            this.FieldContextMenuUpdaters += customFieldBinder.UpdateElement;
            randomFe1Binder.UpdateElement();
            randomBinder.UpdateElement();
            magnetBinder.UpdateElement();
            spinBinder.UpdateElement();
            convergeBinder.UpdateElement();
            addLocationBinder.UpdateElement();
            collisionBinder.UpdateElement();
            curlNoiseBinder.UpdateElement();
            customFieldBinder.UpdateElement();

            foreach (var item in reservedShaderMenuItems)
            {
                Binder reservedShaderBinder = item.AddBinding("Enabled", "CanCreateNewReservedShader");
                this.ReservedShaderContextMenuUpdaters += reservedShaderBinder.UpdateElement;
                reservedShaderBinder.UpdateElement();
            }

            docMenu.AddBinding("ContextMenuOpeningExecutable", "EvaluateMenuItemsExecutable").UpdateElement();
        }
    }
}
