﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.UIControls.Specifics.TabPages
{
    /// <summary>
    /// ページのタブカラーを編集状態によって変更するためのダミーコントロールです。
    /// </summary>
    public class PropertyTabPageModificationRenderer : Control, IBindable
    {
        /// <summary>
        /// データソースに変更があったかどうか表すフラグです。
        /// </summary>
        bool isDataSourceModified;

        /// <summary>
        /// データに変更があったかどうか表すフラグです。
        /// </summary>
        bool isDataModified;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>
        /// Backing field for Controls property.
        /// (should be empty, just to satisfy ILogicalTreeElement)
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">Parent.</param>
        public PropertyTabPageModificationRenderer(PropertyTabPageBase parent)
        {
            this.Parent = parent;

            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// データソースの変更フラグが変わったときのイベントです。
        /// </summary>
        public event EventHandler DataSourceModifiedChanged;

        /// <summary>
        /// データソースの変更フラグを設定または取得します。
        /// </summary>
        public bool IsDataSourceModified
        {
            get
            {
                return this.isDataSourceModified;
            }

            set
            {
                if (this.isDataSourceModified == value)
                {
                    return;
                }

                this.isDataSourceModified = value;
                this.TriggerDataSourceModifiedChanged();
            }
        }

        /// <summary>
        /// データの変更フラグを設定または取得します。
        /// </summary>
        public bool IsDataModified
        {
            get
            {
                return this.isDataModified;
            }

            set
            {
                if (this.isDataModified == value)
                {
                    return;
                }

                this.isDataModified = value;
                this.TriggerDataSourceModifiedChanged();
            }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get; private set;
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new NullIndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// データソースの変更フラグが変わったときのイベントを発生させます。
        /// </summary>
        private void TriggerDataSourceModifiedChanged()
        {
            if (this.DataSourceModifiedChanged == null)
            {
                return;
            }

            this.DataSourceModifiedChanged(this, EventArgs.Empty);
        }
    }
}
