﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.Manager;
using EffectMaker.Foundation;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Focus;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.SpecificControls;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UIControls.Threading;
using EffectMaker.UIControls.Xaml;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TabPages
{
    /// <summary>
    /// An extended UITabPage class for properties.
    /// </summary>
    public class PropertyTabPageBase : UITabPage, IChildrenFocusable
    {
        /// <summary>
        /// Stores the name of the resource stream.
        /// </summary>
        private ResourceInfo resourceInfo;

        /// <summary>
        /// Stores the flag telling whether the tab has been
        /// selected for the first time or not.
        /// </summary>
        private bool firstTime = true;

        /// <summary>
        /// プリロードしたコントロールです。
        /// </summary>
        private Control preloadedControl = null;

        /// <summary>
        /// A class that processes the tab order of the controls in the property page.
        /// </summary>
        private TabOrderProcessor tabOrderProcessor = null;

        /// <summary>
        /// The data context for the modification flags.
        /// This data context will be set to the caption label.
        /// </summary>
        private string modificationDataContext = "ModificationFlagViewModel";

        /// <summary>
        /// Initializes the PropertyTabPageBase instance.
        /// </summary>
        /// <param name="resourceInfo">リソース情報</param>
        public PropertyTabPageBase(ResourceInfo resourceInfo)
        {
            this.resourceInfo = resourceInfo;

            this.Behaviors.Add(new TabPageCopyPasteContextMenuBehavior());

            this.PropertyTabPageModificationRenderer = new PropertyTabPageModificationRenderer(this);
            this.PropertyTabPageModificationRenderer.DataSourceModifiedChanged += this.OnDataSourceModifiedChanged;
            this.Children.Add(this.PropertyTabPageModificationRenderer);

            this.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// データソースの変更フラグが変わったときのイベントです。
        /// </summary>
        public event EventHandler DataSourceModifiedChanged;

        /// <summary>
        /// 初期状態からの変更点の有無を取得する
        /// </summary>
        public bool IsSomeValuesNotDefault
        {
            get { return this.PropertyTabPageModificationRenderer.IsDataSourceModified; }
        }

        /// <summary>
        /// タブページ内に変更があるか
        /// </summary>
        public bool IsModified
        {
            get { return this.PropertyTabPageModificationRenderer.IsDataModified; }
        }

        /// <summary>
        /// Get or set the data context of the caption label.
        /// </summary>
        public object RendererDataContext
        {
            get { return this.PropertyTabPageModificationRenderer.DataContext; }
            set { this.PropertyTabPageModificationRenderer.DataContext = value; }
        }

        /// <summary>
        /// Get or set the data context for the modification flags.
        /// (the data context for the caption label.)
        /// </summary>
        public string ModificationDataContext
        {
            get
            {
                return this.modificationDataContext;
            }

            set
            {
                if (this.modificationDataContext == value)
                {
                    return;
                }

                this.modificationDataContext = value;
                this.UpdateModificationDataContext();
            }
        }

        /// <summary>
        /// タブの変更状態を管理するViewModelとのバインダー
        /// </summary>
        public BindingContainer ChildBinder
        {
            get { return PropertyTabPageModificationRenderer.Bindings; }
        }

        /// <summary>
        /// 変更フラグ管理クラスです。
        /// </summary>
        protected PropertyTabPageModificationRenderer PropertyTabPageModificationRenderer { get; set; }

        /// <summary>
        /// Select (focus on) the first child control in the property page.
        /// </summary>
        /// <param name="forward">False to select the last child control.</param>
        public void SelectFirstChildControl(bool forward)
        {
            if (this.tabOrderProcessor == null)
            {
                return;
            }

            Control firstControl = this.tabOrderProcessor.GetFirstControl(forward);
            Control nextControl = firstControl;

            while (nextControl != null && nextControl.CanReceiveFocus() == false)
            {
                nextControl = this.tabOrderProcessor.GetNextControl(nextControl, forward);

                if (nextControl == firstControl)
                {
                    nextControl = null;
                    break;
                }
            }

            if (nextControl != null)
            {
                nextControl.Focus();
            }
        }

        /// <summary>
        /// Select (focus on) the next child control in the property page.
        /// </summary>
        /// <param name="forward">False to select the previous child control.</param>
        public void SelectNextChildControl(bool forward)
        {
            if (this.tabOrderProcessor == null)
            {
                return;
            }

            Control focusedControl = ControlFocusManager.FindFocusedControl(this);

            focusedControl = CorrectFocusedControl(focusedControl);

            Control firstControl = this.tabOrderProcessor.GetNextControl(focusedControl, !forward);
            Control nextControl = focusedControl;

            do
            {
                nextControl = this.tabOrderProcessor.GetNextControl(nextControl, forward);

                if (nextControl == firstControl)
                {
                    break;
                }
            }
            while (nextControl != null && nextControl.CanReceiveFocus() == false);

            if (nextControl != null && nextControl.CanReceiveFocus() == true)
            {
                nextControl.Focus();
            }
        }

        /// <summary>
        /// The on preload.
        /// </summary>
        public void OnPreload()
        {
            if (this.preloadedControl != null)
            {
                return;
            }

            using (Stream contentStream = ResourceUtility.Load(this.resourceInfo))
            {
                this.preloadedControl = XamlLoader.Load(contentStream);
                this.Controls.Clear();
                this.AddChildControls((IControl)this.preloadedControl);
            }
        }

        /// <summary>
        /// Called when the tab get selected.
        /// </summary>
        protected internal override void OnSelected()
        {
            // タブの切り替えでスクロール位置がずれるのを抑制します。
            UIPanel.ScrollBlocking = true;

            base.OnSelected();

            ControlFocusManager.GlobalTabFocusControl = this;

            if (this.firstTime)
            {
                LayoutEngineBase.SuspendLayout();
                this.firstTime = false;
                this.UpdatePage();
                this.tabOrderProcessor = new TabOrderProcessor(this);
                this.Select();
                this.Focus();
                LayoutEngineBase.ResumeLayout();
            }

            // Delay layout requests for 0.1 seconds.
            // When updating the property page, all the size, location, visibility, ...etc.
            // changes of the controls trigger layout events, and the events are handled
            // on the next application loop, so we need to suspend layout requests for
            // a short period to prevent unnecessary or redundant layout operations.
            LayoutEngineBase.DelayLayout(1);

            // 抑制解除
            UIPanel.ScrollBlocking = false;
        }

        /// <summary>
        /// コマンド キーを処理します。
        /// </summary>
        /// <returns>
        /// 文字がコントロールによって処理された場合は true。それ以外の場合は false。
        /// </returns>
        /// <param name="msg">処理するウィンドウ メッセージを表す、参照渡しされた <see cref="T:System.Windows.Forms.Message"/>。</param><param name="keyData">処理するキーを表す <see cref="T:System.Windows.Forms.Keys"/> 値の 1 つ。</param>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            ////if (keyData == Keys.F4)
            ////{
            ////    this.UpdateDynamicPage();
            ////    return true;
            ////}

            if (keyData == Keys.Tab)
            {
                this.SelectNextChildControl(true);
                return true;
            }
            else if (keyData == (Keys.Tab | Keys.Shift))
            {
                this.SelectNextChildControl(false);
                return true;
            }
            else
            {
                return base.ProcessCmdKey(ref msg, keyData);
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.ParentChanged"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.EventArgs"/>。</param>
        protected override void OnParentChanged(EventArgs e)
        {
            base.OnParentChanged(e);

            if (this.Parent != null)
            {
                System.Diagnostics.Debug.Assert(Parent is Control, "Parent is Control");

                ////(Parent as Control).PreviewKeyDown += (s, ee) =>
                ////{
                ////    var parentTab = Parent as UITabControl;
                ////    if (parentTab != null && parentTab.SelectedContainer != null)
                ////    {
                ////        if (ee.KeyCode == Keys.F4 && parentTab.SelectedContainer == this)
                ////        {
                ////            this.UpdateDynamicPage();
                ////        }
                ////    }
                ////};
            }
        }

        /// <summary>
        /// フォーカスコントロールの調整
        /// 内部的なコントロールなどがある場合はこの関数で吸収する
        /// </summary>
        /// <param name="focusedControl">フォーカスコントロール</param>
        /// <returns>フォーカス調整済みコントロール</returns>
        private static Control CorrectFocusedControl(Control focusedControl)
        {
            if (focusedControl is UINumericUpDown)
            {
                focusedControl = (focusedControl as UINumericUpDown).InternalTextBox;
            }

            return focusedControl;
        }

        /////// <summary>
        /////// 動的なページを更新する
        /////// </summary>
        ////private void UpdateDynamicPage()
        ////{
        ////    string generalUserDataListFilePath = Path.Combine(
        ////        IOConstants.ExecutableFolderPath,
        ////        "Addins",
        ////        "Addins.xml");

        ////    string reservedShaderListFilePath = Path.Combine(
        ////        IOConstants.ExecutableFolderPath,
        ////        "Addins",
        ////        "ReservedShaders.xml");

        ////    UserDataHelper.Instance.LoadAndSetupAllUserData(
        ////        generalUserDataListFilePath,
        ////        OptionStore.ProjectConfig.CustomShaderPath,
        ////        OptionStore.ProjectConfig.CustomActionPath,
        ////        reservedShaderListFilePath);

        ////    this.UpdatePage();
        ////}

        /// <summary>
        /// ページ更新
        /// </summary>
        private void UpdatePage()
        {
            if (this.preloadedControl == null)
            {
                using (Stream contentStream = ResourceUtility.Load(this.resourceInfo))
                {
                    LayoutEngineBase.SuspendLayout();
                    Control control = XamlLoader.Load(contentStream);
                    LayoutEngineBase.ResumeLayout();

                    this.SetupPage(control);
                }
            }
            else
            {
                this.SetupPage(this.preloadedControl);
            }
        }

        /// <summary>
        /// Setup the tab page according to the given constructed root control.
        /// </summary>
        /// <param name="rootControl">The root control hosted by the current tab page.</param>
        private void SetupPage(Control rootControl)
        {
            if (rootControl == null)
            {
                return;
            }

            if (this.preloadedControl == null)
            {
                this.Controls.Clear();
                LayoutEngineBase.SuspendLayout();
                this.AddChildControls((IControl)rootControl);
                LayoutEngineBase.ResumeLayout();
            }

            // trick to make the content of tab page to be properly layed out.
            SynchronizationContext syncContext = UISynchronizationContextHolder.SynchronizationContext;
            if (syncContext != null)
            {
                syncContext.Post(_ => rootControl.Dock = DockStyle.Fill, null);
            }
            else
            {
                rootControl.Dock = DockStyle.Fill;
            }

            ////rootControl.PreviewKeyDown += (s, e) =>
            ////{
            ////    if (!e.Alt && e.KeyCode == Keys.F4)
            ////    {
            ////        this.UpdateDynamicPage();
            ////    }
            ////};

            this.LogicalTreeElementExtender.NotifyDataContextChanged();
        }

        /// <summary>
        /// Handle PropertyChanged event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "DataContext")
            {
                this.UpdateModificationDataContext();
            }
        }

        /// <summary>
        /// データソースの変更フラグが変わったときのイベントを処理します。
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDataSourceModifiedChanged(object sender, EventArgs e)
        {
            if (this.DataSourceModifiedChanged == null)
            {
                return;
            }

            this.DataSourceModifiedChanged(this, EventArgs.Empty);
        }

        /// <summary>
        /// データコンテキストの変更状態をアップデート
        /// </summary>
        private void UpdateModificationDataContext()
        {
            // First remove all the binders that binds to the label's "DataContext" property.
            DataBinding.Binder[] bindersToRemove = this.Bindings.Where(
                item => item.ElementPropertyName == "RendererDataContext").ToArray();

            // There should be at most one binding for the data context.
            if (bindersToRemove != null && bindersToRemove.Length > 1)
            {
                Logger.Log(
                    LogLevels.Error,
                    "ProjectTreeNodeBase.UpdateModificationDataContext : Data context binds to multiple data sources.");
            }

            foreach (DataBinding.Binder binder in bindersToRemove)
            {
                // If the bound data source is the same, don't replace the binding with the new one.
                if (binder.DataSourcePropertyName == this.modificationDataContext)
                {
                    return;
                }

                this.Bindings.Remove(binder);
            }

            // Also update the modification flag target property bindings.
            this.UpdateModificationFlagTarget();

            // Add new binder to the data context property.
            var rendererBinder = this.AddBinding("RendererDataContext", this.modificationDataContext);
            rendererBinder.UpdateElement();
        }

        /// <summary>
        /// 変更フラグのターゲットをアップデート
        /// </summary>
        private void UpdateModificationFlagTarget()
        {
            // Remove old binders.
            {
                // Remove all the binders that binds to the label's "IsDataSourceModified" property.
                DataBinding.Binder[] bindersToRemove = this.PropertyTabPageModificationRenderer.Bindings.Where(
                    item => item.ElementPropertyName == "IsDataSourceModified").ToArray();
                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    this.PropertyTabPageModificationRenderer.Bindings.Remove(binder);
                }

                // Remove all the binders that binds to the label's "IsDataModified" property.
                bindersToRemove = this.PropertyTabPageModificationRenderer.Bindings.Where(
                    item => item.ElementPropertyName == "IsDataModified").ToArray();
                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    this.PropertyTabPageModificationRenderer.Bindings.Remove(binder);
                }
            }

            // Add the new binders.
            {
                {
                    DataBinding.Binder binder = this.PropertyTabPageModificationRenderer.AddBinding(
                        "IsDataSourceModified",
                        "IsSomeValuesNotDefault");

                    binder.Mode = DataBinding.BindingMode.OneWay;
                }

                {
                    DataBinding.Binder binder = this.PropertyTabPageModificationRenderer.AddBinding(
                        "IsDataModified",
                        "IsAnyValueModified");

                    binder.Mode = DataBinding.BindingMode.OneWay;
                }
            }
        }
    }
}
