﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics.Sliders
{
    /// <summary>
    /// 数値入力テキストボックス
    /// </summary>
    public class NumberTextBox : TextBox
    {
        /// <summary>
        /// 値.
        /// </summary>
        private float value;

        /// <summary>
        /// 最大値.
        /// </summary>
        private float maximum = +float.MaxValue;

        /// <summary>
        /// 最小値.
        /// </summary>
        private float minimum = -float.MaxValue;

        /// <summary>
        /// テキストフォーマット
        /// </summary>
        private string textFormat = string.Empty;

        /// <summary>
        /// フォーマットを外す処理
        /// </summary>
        private Func<string, string> deformat = formated => formated;

        /// <summary>
        /// ツールチップ
        /// </summary>
        private ToolTip toolTip = null;

        /// <summary>
        /// ツールチップ文字列
        /// </summary>
        private string toolTipText;

        /// <summary>
        /// 値変更イベント
        /// </summary>
        public event EventHandler ValueChanged;

        /// <summary>
        /// スライダの値です。
        /// </summary>
        public float Value
        {
            get
            {
                return this.value;
            }

            set
            {
                value = Math.Max(Math.Min(value, this.Maximum), this.Minimum);

                if (this.value != value)
                {
                    this.value = value;

                    if (this.ValueChanged != null)
                    {
                        this.ValueChanged(this, EventArgs.Empty);
                    }
                }

                this.UpdateText();
            }
        }

        /// <summary>
        /// 最大値
        /// </summary>
        public float Maximum
        {
            get
            {
                return this.maximum;
            }

            set
            {
                this.maximum = value;
                if (this.Value > this.maximum)
                {
                    this.Value = this.maximum;
                }
            }
        }

        /// <summary>
        /// 最小値
        /// </summary>
        public float Minimum
        {
            get
            {
                return this.minimum;
            }

            set
            {
                this.minimum = value;
                if (this.Value < this.minimum)
                {
                    this.Value = this.minimum;
                }
            }
        }

        /// <summary>
        /// テキストフォーマット
        /// </summary>
        public string TextFormat
        {
            get
            {
                return this.textFormat;
            }

            set
            {
                if (this.textFormat != value)
                {
                    this.textFormat = value;
                    this.UpdateText();
                }
            }
        }

        /// <summary>
        /// フォーマットを外す処理
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Func<string, string> Deformat
        {
            get
            {
                return this.deformat;
            }

            set
            {
                this.deformat = value;
            }
        }

        /// <summary>
        /// ツールチップテキストの指定
        /// </summary>
        public string ToolTipText
        {
            get
            {
                return this.toolTipText;
            }

            set
            {
                this.toolTipText = value;

                if (this.toolTip == null)
                {
                    this.toolTip = new ToolTip
                    {
                        AutoPopDelay = 1200,
                        InitialDelay = 1,
                        ReshowDelay = 1
                    };

                    this.MouseEnter += (s, e) =>
                        {
                            if (this.ToolTipDuration <= 0)
                            {
                                this.toolTip.Show(this.toolTipText, this, this.ToolTipPoint);
                            }
                            else
                            {
                                this.toolTip.Show(this.toolTipText, this, this.ToolTipPoint, this.ToolTipDuration);
                            }
                        };
                    this.MouseLeave += (s, e) => this.toolTip.Hide(this);
                }
            }
        }

        /// <summary>
        /// ツールチップテキストを表示する座標
        /// </summary>
        public Point ToolTipPoint { get; set; }

        /// <summary>
        /// ツールチップテキストを表示する時間
        /// -1ならマウスを当てている間ずっと表示する。
        /// </summary>
        public int ToolTipDuration { get; set; }

        /// <summary>
        /// キー押下時処理
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnKeyPress(KeyPressEventArgs e)
        {
            // 数字以外入力できないようにします。
            if (!char.IsControl(e.KeyChar)
                && !char.IsDigit(e.KeyChar))
            {
                switch (e.KeyChar)
                {
                    default:
                        e.Handled = true;
                        break;

                    // 正数/負数 ParseTextMath
                    case '+':
                    case '-':
                        break;

                    // float
                    case '.':
                    case 'e':
                    case 'E':
                        break;

                    // ParseTextMath
                    case '*':
                    case ' ':
                    case '/':
                        break;

                    // ()
                    case '(':
                    case ')':
                        break;
                }
            }

            base.OnKeyPress(e);
        }

        /// <summary>
        /// ダイアログ ボックスのキーを処理します。
        /// </summary>
        /// <param name="keyData">キーデータ</param>
        /// <returns>キーストロークがコントロールによって処理および使用された場合は true。キーストロークをさらに処理できるようにする場合は false。</returns>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // リターン入力処理
            if (keyData == Keys.Return)
            {
                this.UpdateValue();
                if (this.ValueChanged != null)
                {
                    this.ValueChanged(this, EventArgs.Empty);
                }

                return true;
            }

            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// フォーカスロスト時処理
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnLostFocus(EventArgs e)
        {
            this.UpdateValue();
            base.OnLostFocus(e);
        }

        /// <summary>
        /// テキスト更新
        /// </summary>
        private void UpdateText()
        {
            if (string.IsNullOrEmpty(this.TextFormat))
            {
                this.Text = this.Value.ToString(CultureInfo.InvariantCulture);
            }
            else
            {
                this.Text = string.Format(this.TextFormat, this.Value);
            }
        }

        /// <summary>
        /// 値更新
        /// </summary>
        private void UpdateValue()
        {
            float parsedValue;

            var deformatedText = this.Deformat(this.Text);

            // 変換できなければ前回の値とする
            if (UtilDialSlider.ParseTextMath(deformatedText, out parsedValue) ||
                float.TryParse(deformatedText, out parsedValue))
            {
                this.Value = MathUtility.RoundToEditableMinimumDigit(parsedValue);
            }
            else
            {
                this.UpdateText();
            }

            this.SelectAll();
        }
    }
}
