﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows.Forms;
using System.Xml;
using EffectMaker.DataModel.Manager;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Xaml;
using EffectMaker.UILogic.Manager;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.ItemContainerSelectors
{
    /// <summary>
    /// An IItemContainerSelector that instances custom user setting panels.
    /// </summary>
    public class CustomUserPageItemContainerSelector : IItemContainerSelector
    {
        /// <summary>Stores the cache of created controls.</summary>
        private static readonly Dictionary<string, CachedXamlControlData> ControlCache =
            new Dictionary<string, CachedXamlControlData>();

        /// <summary>
        /// Instances a UITabPage control.
        /// </summary>
        /// <param name="dataItem">The data item for which to create the page.</param>
        /// <returns>Returns an instance of a UITabPage, or null if data item is null.</returns>
        public IControl SelectItemContainer(object dataItem)
        {
            var viewModel = dataItem as IUserDataInfoExportViewModel;
            if (viewModel == null)
            {
                return null;
            }

            var info = viewModel.UserDataInfo;
            string xamlPath = info.XamlFilePath;
            string xamlContent;

            if (!string.IsNullOrEmpty(info.XamlCode))
            {
                xamlContent = info.XamlCode;
            }
            else
            {
                if (File.Exists(xamlPath) == false)
                {
                    return null;
                }

                xamlContent = File.ReadAllText(xamlPath);
            }

            DateTime lastModifiedTime = File.GetLastWriteTime(xamlPath);

            CachedXamlControlData data;
            if (ControlCache.TryGetValue(xamlPath, out data) == false ||
                DateTime.Equals(data.LastModifiedTime, lastModifiedTime) == false)
            {
                var control = LoadXamlControl(xamlContent, xamlPath);
                if (control == null)
                {
                    return null;
                }

                data = new CachedXamlControlData()
                {
                    XamlPath = xamlPath,
                    LastModifiedTime = lastModifiedTime,
                    LoadedControl = control,
                };

                // Collapse all the controls by default.
                var layoutElement = control as EffectMaker.UIControls.Layout.ILayoutElement;
                if (layoutElement != null)
                {
                    layoutElement.Visibility = EffectMaker.UIControls.Layout.Visibility.Collapsed;
                }

                ControlCache[xamlPath] = data;
            }

            return data.LoadedControl;
        }

        /// <summary>
        /// Helper method for loading Xaml control from the specified path.
        /// </summary>
        /// <param name="xamlContent">The XAML code.</param>
        /// <param name="sourcePath">The source path.</param>
        /// <returns>The loaded control or null if failed.</returns>
        private static IControl LoadXamlControl(string xamlContent, string sourcePath)
        {
            try
            {
                return XamlLoader.Parse(xamlContent) as IControl;
            }
            catch
            {
                Logger.Log(LogLevels.Warning, "CustomUserPageItemContainerSelector.LoadXamlControl : Failed loading Xaml from {0}.", sourcePath);
                return null;
            }
        }

        /// <summary>
        /// Holds data for a cached XAML control.
        /// </summary>
        private class CachedXamlControlData
        {
            /// <summary>
            /// Get or set the path of the XAML file.
            /// </summary>
            public string XamlPath { get; set; }

            /// <summary>
            /// The last modified time of the XAML file.
            /// </summary>
            public DateTime LastModifiedTime { get; set; }

            /// <summary>
            /// Get or set the loaded XAML control.
            /// </summary>
            public IControl LoadedControl { get; set; }
        }
    }
}
