﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using ColorClipboardData = EffectMaker.Foundation.ClipboardDataTypes.ColorClipboardData;

namespace EffectMaker.UIControls.Specifics.GradationEditor
{
    /// <summary>
    /// グラデーションエディタ
    /// </summary>
    public class GradationEditor : UIControl
    {
        /// <summary>
        /// 削除ボタン位置
        /// </summary>
        public const int KeyRemoveButtonPosY = 0;

        /// <summary>
        /// カラーバー水平マージン
        /// </summary>
        private const int ColorBarMarginX = 2;

        /// <summary>
        /// つまみ垂直マージン
        /// </summary>
        private const int KnobMarginY = 4;

        /// <summary>
        /// キーフレーム
        /// </summary>
        private List<GradationKey> keys = new List<GradationKey>();

        /// <summary>
        /// ドラッグ中キー
        /// </summary>
        private int draggingKeyIndex = -1;

        /// <summary>
        /// マウスオーバー中キー
        /// </summary>
        private int mouseoverKeyIndex = -1;

        /// <summary>
        /// アクティブなエレメントのインデックス
        /// </summary>
        private int activeIndex = -1;

        /// <summary>
        /// アクティブか？
        /// </summary>
        private bool isActive;

        /// <summary>
        /// RGBの編集を有効にするか
        /// </summary>
        private bool rgbEditEnabled;

        /// <summary>
        /// アルファの編集を有効にするか
        /// </summary>
        private bool alphaEditEnabled;

        /// <summary>
        /// 時間の編集を有効にするか
        /// </summary>
        private bool timeEditEnabled;

        /// <summary>
        /// ツールチップ
        /// </summary>
        private ToolTip tooltip;

        /// <summary>
        /// ポップアップ元キー
        /// </summary>
        private GradationKey popupContextKey = null;

        /// <summary>
        /// マウスダウンからの経過時間を計測するタイマー
        /// </summary>
        private Stopwatch stopWatch = new Stopwatch();

        /// <summary>
        /// マウスダウン時の座標.
        /// </summary>
        private Point downPos;

        /// <summary>
        /// 履歴記録用の差分データ
        /// </summary>
        private AnimationTableData prevTable = new AnimationTableData();

        /// <summary>
        /// 値変更イベントの実行中か
        /// </summary>
        private bool changing = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GradationEditor()
        {
            this.DoubleBuffered = true;

            this.Height = 36;

            this.InitializeContextMenu();

            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;
        }

        /// <summary>
        /// Get or set the flag indicating whether the keys can be added and removed.
        /// </summary>
        public bool CanAddRemoveKeys { get; protected set; }

        /// <summary>
        /// 削除ホタンが有効か？
        /// </summary>
        public bool IsEnabledRemoveButton
        {
            get
            {
                return this.CanRemoveKey;
            }
        }

        /// <summary>
        /// キーを削除できるか
        /// </summary>
        public bool CanRemoveKey
        {
            get
            {
                return this.keys.Count() > this.MinNumKeys;
            }
        }

        /// <summary>
        /// 最大フレーム
        /// </summary>
        public int MaximumFrame
        {
            get
            {
                return 100;
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether can keys overlap.
        /// (Multiple keys set to the same time in the animation.)
        /// </summary>
        /// <remarks>This flag is always false when CanReorderKeys set to true.</remarks>
        public bool CanKeysOverlap { get; protected set; }

        /// <summary>
        /// Get or set the flag indicating whether the keys can be reordered.
        /// If false, the keys cannot be moved prior to the previous key, nor
        /// exceeding the next.
        /// </summary>
        public bool CanReorderKeys { get; protected set; }

        /// <summary>
        /// 端キーを移動できるか
        /// </summary>
        public bool MovableEdgeKey { get; protected set; }

        /// <summary>
        /// インアウトアニメか
        /// 廃止予定
        /// </summary>
        public bool IsInOutAnim { get; protected set; }

        /// <summary>
        /// アクティブか？
        /// </summary>
        public bool IsActive
        {
            get
            {
                return this.isActive;
            }

            set
            {
                if (this.isActive == value)
                {
                    return;
                }

                this.isActive = value;

                if (this.isActive == false)
                {
                    this.ActiveIndex = -1;
                }

                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");
                this.Invalidate();
            }
        }

        /// <summary>
        /// アクティブなエレメントのインデックス
        /// </summary>
        public int ActiveIndex
        {
            get
            {
                return this.activeIndex;
            }

            set
            {
                if (this.activeIndex == value)
                {
                    return;
                }

                this.activeIndex = value;

                if (this.activeIndex != -1)
                {
                    this.Value[this.ActiveIndex].IsSelected = true;
                }
                else
                {
                    this.Value.SelectedKeyFrames.ForEach(x => x.IsSelected = false);
                }

                // this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");

                this.Invalidate();
            }
        }

        /// <summary>
        /// 値
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public AnimationTableData Value
        {
            get
            {
                var table = new AnimationTableData()
                {
                    MultiSelect = false,
                };

                this.keys.ForEach(key => table.AddKeyFrame(key.Time, key.Color, false));

                if (this.ActiveIndex != -1)
                {
                    table[this.ActiveIndex].IsSelected = true;
                }

                return table;
            }

            set
            {
                this.SetValueInstance(value);

                if (!this.changing)
                {
                    this.prevTable.Set(value);
                }

                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");

                this.Invalidate();
            }
        }

        /// <summary>
        /// 色編集情報
        /// </summary>
        public EditingColorInfo EditingColorInfo
        {
            get
            {
                var time = (this.DraggingKey != null)
                    ? this.DraggingKey.Time
                    : (this.Value.SelectedKeyFrame != null)
                        ? this.Value.SelectedKeyFrame.Frame
                        : 0;

                var color = (this.DraggingKey != null)
                    ? this.DraggingKey.Color
                    : (this.Value.SelectedKeyFrame != null)
                        ? this.Value.SelectedKeyFrame.ValueAsColor
                        : new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);

                return new EditingColorInfo
                {
                    Control =
                        this.IsActive && (this.ActiveIndex != -1) && !this.IsUpdatingDataContext
                            ? this : null,
                    Color = color,
                    RgbEditEnabled = this.RgbEditEnabled,
                    AlphaEditEnabled = this.AlphaEditEnabled,
                    TimeEditEnabled = this.TimeEditEnabled,
                    Time = time
                };
            }

            set
            {
                if (value.RefreshRequest)
                {
                    if (this.TargetName != value.RefreshTargetName)
                    {
                        return;
                    }

                    // コントロールに反映させる
                    if (this.Value.SelectedKeyFrameIndex == -1)
                    {
                        return;
                    }

                    if (this.Focused && !this.IsUpdatingDataContext)
                    {
                        this.keys[this.Value.SelectedKeyFrameIndex].Color = value.Color;
                        this.Invalidate();
                    }

                    // 現在アクティブであればカラーピッカー更新リクエスト
                    value.RefreshColorPickerRequest = this.Focused;
                }
                else
                {
                    if (value.Control == this)
                    {
                        if (this.Value.SelectedKeyFrame == null)
                        {
                            return;
                        }

                        if (this.CanKeysOverlap ||
                            (this.IsAnyKey(key => (key.Index != this.ActiveIndex) && (key.Time == value.Time)) ==
                             false))
                        {
                            this.keys[this.ActiveIndex].Color = value.Color;
                            this.keys[this.ActiveIndex].Time = value.Time;

                            this.Value.SelectedKeyFrame.ValueAsColor = value.Color;
                            this.Value.SelectedKeyFrame.Frame = value.Time;
                            if (!this.IsUpdatingDataContext)
                            {
                                if (value.IsEditing)
                                {
                                    this.LogicalTreeElementExtender.NotifyPropertyChanged(
                                        BindingUpdateType.PropertyChanged, "Value");
                                }
                                else
                                {
                                    this.NotifyValueChanged();
                                }
                            }

                            // アクティブが切り替わる可能性があるので再インデックス
                            {
                                var currentKey = this.keys[this.ActiveIndex];
                                this.SortKeysByKeyTime();
                                this.PutIndex();
                                this.ActiveIndex = currentKey.Index;

                                this.Value.SelectedKeyFrames.ForEach(x => x.IsSelected = false);
                                this.Value[this.ActiveIndex].IsSelected = true;
                            }

                            this.Invalidate();
                        }
                        else
                        {
                            // 元に戻す
                            EventHandler idle = null;

                            idle = (s, e) =>
                            {
                                Application.Idle -= idle;
                                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                                    BindingUpdateType.PropertyChanged, "EditingColorInfo");
                            };

                            Application.Idle += idle;
                        }
                    }
                    else
                    {
                        // アイテム選択中は値更新を行わない(スタックトレースを使わずに分岐)
                        if (value.IsEditing || this.IsUpdatingDataContext)
                        {
                            return;
                        }

                        // 他のコントロールがアクティブなので全て未選択にする
                        this.ActiveIndex = -1;
                        Enumerable.Range(0, this.Value.Count).Select(x => this.Value[x]).ForEach(x => x.IsSelected = false);

                        this.IsActive = false;
                    }
                }
            }
        }

        /// <summary>
        /// RGBの編集を有効にするか
        /// </summary>
        public bool RgbEditEnabled
        {
            get
            {
                return this.rgbEditEnabled;
            }

            set
            {
                this.rgbEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "RgbEditEnabled");
            }
        }

        /// <summary>
        /// アルファの編集を有効にするか
        /// </summary>
        public bool AlphaEditEnabled
        {
            get
            {
                return this.alphaEditEnabled;
            }

            set
            {
                this.alphaEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "AlphaEditEnabled");
            }
        }

        /// <summary>
        /// 時間の編集を有効にするか
        /// </summary>
        public bool TimeEditEnabled
        {
            get
            {
                return this.timeEditEnabled;
            }

            set
            {
                this.timeEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "TimeEditEnabled");
            }
        }

        /// <summary>
        /// 編集対象名
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        /// パーティクルの寿命を取得するExecutableを取得または設定します。
        /// </summary>
        public IExecutable GetParticleLifeExecutable { get; set; }

        /// <summary>
        /// Gets or sets the IExecutable to run when the Value property changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// カラーバー矩形
        /// </summary>
        public RectangleF ColorBarRectangle
        {
            get
            {
                if (this.CanAddRemoveKeys || this.IsInOutAnim)
                {
                    var knobY = GradationKey.RemoveButtonHeight + KnobMarginY;

                    return new RectangleF(
                        ColorBarMarginX,
                        knobY,
                        Width - 1 - (ColorBarMarginX * 2),
                        Height - 1 - knobY);
                }
                else
                {
                    return new RectangleF(
                        ColorBarMarginX,
                        0.0f,
                        Width - 1 - (ColorBarMarginX * 2),
                        Height - 1);
                }
            }
        }

        /// <summary>
        /// Get or set the minimum number of keys.
        /// </summary>
        protected int MinNumKeys { get; set; }

        /// <summary>
        /// Get or set the maximum number of keys.
        /// </summary>
        protected int MaxNumKeys { get; set; }

        /// <summary>
        /// ドラッグ中キー
        /// </summary>
        private GradationKey DraggingKey
        {
            get
            {
                return ((this.draggingKeyIndex != -1) && (this.draggingKeyIndex < this.keys.Count()))
                    ? this.keys[this.draggingKeyIndex]
                    : null;
            }
        }

        /// <summary>
        /// 指定キータイムのキーが存在するか
        /// </summary>
        /// <param name="keyTime">指定キータイム</param>
        /// <returns>存在するか</returns>
        public bool IsAnyKeyTime(int keyTime)
        {
            return this.keys.Any(x => x.Time == keyTime);
        }

        /// <summary>
        /// 指定のキーが存在するか
        /// </summary>
        /// <param name="predicate">判断</param>
        /// <returns>存在するか</returns>
        public bool IsAnyKey(Func<GradationKey, bool> predicate)
        {
            return this.keys.Any(predicate);
        }

        /// <summary>
        /// キータイムから時間を作る
        /// </summary>
        /// <param name="mouseX">マウスX位置</param>
        /// <param name="mouseY">マウス位置</param>
        /// <returns>色</returns>
        public int MakeTime(int mouseX, int mouseY)
        {
            return (int)(this.MaximumFrame * (double)mouseX / this.ColorBarRectangle.Width);
        }

        /// <summary>
        /// キータイムから時間を作る
        /// </summary>
        /// <param name="mouseX">マウスX位置</param>
        /// <param name="mouseY">マウス位置</param>
        /// <returns>色</returns>
        public int MakeUniqueTime(int mouseX, int mouseY)
        {
            var time = this.MakeTime(mouseX, mouseY);

            // 同タイムのキーがあれば操作する
            for (var i = 0; i != this.MaximumFrame; ++i)
            {
                {
                    var nextTime = Math.Min(Math.Max(time + i, 0), this.MaximumFrame);

                    if (this.IsAnyKeyTime(nextTime) == false)
                    {
                        time = nextTime;
                        break;
                    }
                }

                {
                    var nextTime = Math.Min(Math.Max(time - i, 0), this.MaximumFrame);

                    if (this.IsAnyKeyTime(nextTime) == false)
                    {
                        time = nextTime;
                        break;
                    }
                }
            }

            return time;
        }

        /// <summary>
        /// <see cref="T:System.Windows.Forms.Control"/> とその子コントロールが使用しているアンマネージ リソースを解放します。オプションで、マネージ リソースも解放します。
        /// </summary>
        /// <param name="disposing">マネージ リソースとアンマネージ リソースの両方を解放する場合は true。アンマネージ リソースだけを解放する場合は false。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// マウスが離れた時に発生するイベントです。
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            this.mouseoverKeyIndex = -1;
            this.Invalidate();
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.MouseDown"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.MouseEventArgs"/>。</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            this.Focus();

            this.downPos = e.Location;
            this.stopWatch.Restart();

            // コンテキストメニューをポップアップさせるかどうかの判定用キーをクリア
            this.popupContextKey = null;

            // ドラッグ準備
            this.SortKeysByKeyTime();
            this.PutIndex();

            // マウスで新しく選ばれるキーを探す
            GradationKey selectedKey = null;
            {
                this.keys.Reverse();

                foreach (var key in this.keys)
                {
                    if (this.IsInOutAnim)
                    {
                        if ((key == this.keys.First()) || (key == this.keys.Last()))
                        {
                            continue;
                        }
                    }

                    var handled = false;
                    var isSelected = false;

                    key.OnMouseDown(e, ref handled, ref isSelected);

                    if (isSelected)
                    {
                        selectedKey = key;
                    }

                    if (handled)
                    {
                        break;
                    }
                }

                this.keys.Reverse();
            }

            var isPushedAlt = (Control.ModifierKeys & Keys.Alt) == Keys.Alt;

            GradationKey copySourceKey = null;
            if (isPushedAlt)
            {
                copySourceKey = selectedKey;
                selectedKey = null;
            }

            // キーを新規作成する
            var isNewKey = false;
            if ((e.Button == MouseButtons.Left) && (selectedKey == null))
            {
                if (this.CanAddRemoveKeys)
                {
                    if ((e.Y >= this.ColorBarRectangle.Top) &&
                        (e.Y <= this.ColorBarRectangle.Bottom))
                    {
                        if (this.keys.Count() < this.MaxNumKeys)
                        {
                            // 追加するキーの初期色をつくる
                            ColorRgba newColor = null;
                            {
                                if (copySourceKey != null)
                                {
                                    newColor = copySourceKey.Color.Clone() as ColorRgba;
                                }
                                else
                                {
                                    if (isPushedAlt && (this.ActiveIndex != -1))
                                    {
                                        // 現在アクティブなキーの色で新規キーを作る
                                        newColor = this.keys[this.ActiveIndex].Color.Clone() as ColorRgba;
                                    }
                                    else
                                    {
                                        // マウスキー位置の色で新規ーを作る
                                        newColor = this.MakeColorFrom(e.X, e.Y);
                                    }
                                }
                            }

                            selectedKey = this.AddKey(e.X, e.Y, newColor);
                            selectedKey.OnMouseDown(e);
                            isNewKey = true;
                        }
                        else
                        {
                            if (this.tooltip == null)
                            {
                                this.PrepareToolTip();
                            }

                            this.tooltip.Show(
                                Properties.Resources.GradationEditorWarningMsgKeyframeExceedsMaxNum,
                                this,
                                (int)this.ColorBarRectangle.Right + 2,
                                (int)this.ColorBarRectangle.Top,
                                1000);
                        }
                    }
                }
            }

            if ((e.Button == MouseButtons.Right) && (isNewKey == false) && (selectedKey != null))
            {
                // コンテキストメニュー用キーを保存
                this.popupContextKey = selectedKey;
            }
            else
            {
                // ドラッグ準備
                this.SortKeysByKeyTime();
                this.PutIndex();

                if (selectedKey != null)
                {
                    this.draggingKeyIndex = selectedKey.Index;
                    if (this.ActiveIndex != selectedKey.Index)
                    {
                        // キー列が入れ替わるため、キーのマウス押下処理をやり直す
                        this.ActiveIndex = selectedKey.Index;
                        this.DraggingKey.OnMouseDown(e);
                    }

                    this.IsActive = true;
                    this.ActiveIndex = selectedKey.Index;
                    this.prevTable.Set(this.Value);
                }
                else
                {
                    this.draggingKeyIndex = -1;
                }

                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");

                this.Invalidate();
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.MouseMove"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.MouseEventArgs"/>。</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if (this.DraggingKey != null)
            {
                var isEdgeKey = (this.draggingKeyIndex == this.keys.First().Index) || (this.draggingKeyIndex == this.keys.Last().Index);
                var isDragEditing = this.MovableEdgeKey || (isEdgeKey == false);
                if (isDragEditing &&
                    (Math.Abs(this.downPos.X - e.X) > 2 || Math.Abs(this.downPos.Y - e.Y) > 2))
                {
                    this.DraggingKey.OnMouseDrag(e);
                    this.downPos.X = 65535;
                    this.downPos.Y = 65535;
                }

                if (this.CanReorderKeys == false)
                {
                    this.AlignmentClampKeysByKeyTime();
                }

                this.Invalidate();
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");
            }
            else
            {
                var oldMouseoverKeyIndex = this.mouseoverKeyIndex;
                this.mouseoverKeyIndex = -1;

                var isMouseOver = false;

                this.SortKeysByKeyTime();
                this.PutIndex();

                foreach (var key in this.keys)
                {
                    key.OnMouseMove(e, ref isMouseOver);
                    if (isMouseOver)
                    {
                        this.mouseoverKeyIndex = key.Index;
                        break;
                    }
                }

                if (this.mouseoverKeyIndex != oldMouseoverKeyIndex)
                {
                    if (this.IsActive)
                    {
                        this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");
                    }

                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.MouseUp"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.MouseEventArgs"/>。</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            if (this.DraggingKey != null)
            {
                var dragedKey = this.DraggingKey;
                var isReqRemove = false;

                dragedKey.OnMouseUp(e, ref isReqRemove);

                if (isReqRemove)
                {
                    this.keys.Remove(dragedKey);
                    this.IsActive = false;
                }

                if (dragedKey.IsDraged || isReqRemove)
                {
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
                    this.NotifyValueChanged();
                }

                // インデックスを振り直す
                this.SortKeysByKeyTime();
                this.PutIndex();

                // EditingColorInfoに反映するため、先にdraggingKeyIndexを更新してからActiveIndexを更新する
                this.draggingKeyIndex = isReqRemove ? -1 : dragedKey.Index;
                this.ActiveIndex = this.draggingKeyIndex;

                this.draggingKeyIndex = -1;
                this.Invalidate();
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.Paint"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.PaintEventArgs"/>。</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            var smoothingMode = e.Graphics.SmoothingMode;

            try
            {
                e.Graphics.SmoothingMode = SmoothingMode.HighQuality;

                this.DrawColorBar(e.Graphics);

                // キーの描画。ドラッグ中は最手前で描画する
                {
                    this.keys.Where(x => x != this.DraggingKey).ForEach(
                        x => x.Draw(
                            e.Graphics,
                            new GradationKey.DrawInfo
                            {
                                State =
                                    (x.Index == this.mouseoverKeyIndex)             ? GradationKey.DrawInfo.StateKind.MouseOver :
                                    (x.Index == this.ActiveIndex && this.IsActive)  ? GradationKey.DrawInfo.StateKind.Selected  :
                                                                                      GradationKey.DrawInfo.StateKind.Normal
                            }));

                    if (this.DraggingKey != null)
                    {
                        this.DraggingKey.Draw(e.Graphics, new GradationKey.DrawInfo { State = GradationKey.DrawInfo.StateKind.Dragging });
                    }

                    if (this.DraggingKey != null || this.mouseoverKeyIndex != -1)
                    {
                        this.DrawFramePopup(e.Graphics);
                    }
                }
            }
            finally
            {
                e.Graphics.SmoothingMode = smoothingMode;
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.KeyDown"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.KeyEventArgs"/>。</param>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            base.OnKeyDown(e);

            if (e.KeyCode == Keys.Delete)
            {
                if (this.CanAddRemoveKeys && this.CanRemoveKey)
                {
                    this.RemoveSelectedKey();
                }
            }
        }

        /// <summary>
        /// Called when the Value property changes.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected virtual void OnValueChanged(ValueChangedExEventArgs e)
        {
            IExecutable executable = this.ValueChangedExecutable;
            if (executable != null && executable.CanExecute(e))
            {
                executable.Execute(e);
            }
        }

        /// <summary>
        /// 選択キーを削除する
        /// </summary>
        private void RemoveSelectedKey()
        {
            if (this.ActiveIndex == -1)
            {
                return;
            }

            var oldActiveIndex = this.ActiveIndex;
            var selectedKey = this.keys[this.ActiveIndex];

            this.keys.Remove(selectedKey);

            // アクティブインデックスを振り直す
            this.SortKeysByKeyTime();
            this.PutIndex();

            this.ActiveIndex = -1;
            this.IsActive = false;

            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");
            this.NotifyValueChanged();

            this.Invalidate();
        }

        /// <summary>
        /// フレーム数ポップアップを描画します。
        /// </summary>
        /// <param name="g">グラフィクスコンテキスト</param>
        private void DrawFramePopup(Graphics g)
        {
            var target = this.DraggingKey ?? this.keys.FirstOrDefault(k => k.Index == this.mouseoverKeyIndex);
            if (target == null)
            {
                return;
            }

            string text;
            if (this.GetParticleLifeExecutable != null)
            {
                var reciever = new float[1];
                this.GetParticleLifeExecutable.Execute(reciever);
                text = string.Format("{0}f", (int)Math.Round(reciever[0] * 0.01f * target.Time));
            }
            else
            {
                text = string.Format("{0}%", target.Time);
            }

            var textSize = TextRenderer.MeasureText(text, this.Font);
            var x = this.ColorBarRectangle.X + target.DrawBasePosX + 12;
            var y = this.ColorBarRectangle.Y + 2;
            var w = textSize.Width + 4;
            var h = this.ColorBarRectangle.Height - 4;

            if (x + w > this.ColorBarRectangle.X + this.ColorBarRectangle.Width)
            {
                x = this.ColorBarRectangle.X + target.DrawBasePosX - 2 - w;
            }

            using (var backBrush = new SolidBrush(Color.FromArgb(224, 87, 105, 138)))
            {
                g.FillRectangle(backBrush, x, y, w, h);
            }

            using (var foreBrush = new SolidBrush(Color.White))
            {
                g.DrawString(text, this.Font, foreBrush, x + 2, y + 2);
            }
        }

        /// <summary>
        /// カラーバー描画
        /// </summary>
        /// <param name="g">グラフィックオブジェクト</param>
        private void DrawColorBar(Graphics g)
        {
            var rect = new Rectangle((int)this.ColorBarRectangle.X, (int)this.ColorBarRectangle.Y, (int)this.ColorBarRectangle.Width, (int)this.ColorBarRectangle.Height);

            // 背景
            {
                g.FillRectangle(Brushes.Black, rect);
            }

            // カラーバー
            {
                using (var brush = this.MakeColorBarBrush())
                {
                    if (ColorUtility.IsGammaCorrectionEnabled && ColorUtility.Gamma != 1.0)
                    {
                        using (var tempImage = new Bitmap(rect.Width, rect.Height))
                        using (var tempGr = Graphics.FromImage(tempImage))
                        {
                            tempGr.FillRectangle(brush, new Rectangle(0, 0, rect.Width, Height));

                            var attr = new ImageAttributes();
                            attr.SetGamma((float)(1.0 / ColorUtility.Gamma));

                            g.DrawImage(
                                    tempImage,
                                    rect,
                                    0.0f,
                                    0.0f,
                                    rect.Width,
                                    rect.Height,
                                    GraphicsUnit.Pixel,
                                    attr);
                        }
                    }
                    else
                    {
                        g.FillRectangle(brush, rect);
                    }
                }

                // 縁
                g.DrawRectangle(Pens.Black, rect);
            }
        }

        /// <summary>
        /// カラーバーブラシ作成
        /// </summary>
        /// <returns>カラーバーブラシ</returns>
        private Brush MakeColorBarBrush()
        {
            if (this.keys.Any() == false)
            {
                return new SolidBrush(Color.White);
            }

            var brush = new LinearGradientBrush(this.ColorBarRectangle, Color.Black, Color.Black, LinearGradientMode.Horizontal);
            {
                var colors = new List<Color>();
                var positions = new List<float>();
                {
                    var sortedKeys = this.keys.OrderBy(x => x.Time).ToArray();

                    foreach (var key in sortedKeys)
                    {
                        colors.Add(ColorUtility.NormalizeColor(key.Color).ToWinColor());
                        positions.Add((float)key.Time / this.MaximumFrame);
                    }

                    if (positions.First() != 0.0f)
                    {
                        colors.Insert(
                            0,
                            ColorUtility.NormalizeColor(sortedKeys.First().Color).ToWinColor());
                        positions.Insert(0, 0.0f);
                    }

                    if (positions.Last() != 1.0f)
                    {
                        colors.Add(ColorUtility.NormalizeColor(sortedKeys.Last().Color).ToWinColor());
                        positions.Add(1.0f);
                    }
                }

                brush.InterpolationColors = new ColorBlend
                {
                    Colors = colors.ToArray(),
                    Positions = positions.ToArray()
                };
            }

            return brush;
        }

        /// <summary>
        /// キータイムでソート
        /// </summary>
        private void SortKeysByKeyTime()
        {
            this.keys.Sort((x, y) => x.Time - y.Time);
        }

        /// <summary>
        /// インデックスをふる
        /// </summary>
        private void PutIndex()
        {
            var index = 0;

            this.SortKeysByKeyTime();
            this.keys.ForEach(x => x.Index = index++);
        }

        /// <summary>
        /// キータイムで整列させる
        /// CanReorderKeys == false 時の操作
        /// </summary>
        private void AlignmentClampKeysByKeyTime()
        {
            System.Diagnostics.Debug.Assert(this.CanReorderKeys == false, "this.CanReorderKeys == false");
            System.Diagnostics.Debug.Assert(this.DraggingKey != null, "this.DraggingKey != null");

            var draggingKeyKeyTime = this.DraggingKey.Time;
            var draggingKeyIndex = this.DraggingKey.Index;

            // ドラッグ中キーより前
            for (var i = 0; i != draggingKeyIndex; ++i)
            {
                if (this.keys[i].Time > draggingKeyKeyTime)
                {
                    this.keys[i].Time = draggingKeyKeyTime;
                }
            }

            // ドラッグ中キーより後
            for (var i = draggingKeyIndex; i != this.keys.Count(); ++i)
            {
                if (this.keys[i].Time < draggingKeyKeyTime)
                {
                    this.keys[i].Time = draggingKeyKeyTime;
                }
            }
        }

        /// <summary>
        /// キー追加
        /// </summary>
        /// <param name="mouseX">マウスX位置</param>
        /// <param name="mouseY">マウスY位置</param>
        /// <param name="color">色</param>
        /// <returns>追加いたキー</returns>
        private GradationKey AddKey(int mouseX, int mouseY, ColorRgba color)
        {
            System.Diagnostics.Debug.Assert(color != null, "color != null");

            var keyTime = this.MakeUniqueTime(mouseX, mouseY);
            var key = new GradationKey(this) { Color = color, Time = keyTime };

            this.keys.Add(key);
            this.ActiveIndex = this.keys.Count - 1;
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
            this.NotifyValueChanged();

            return key;
        }

        /// <summary>
        /// キータイムから色を作る
        /// </summary>
        /// <param name="mouseX">マウスX位置</param>
        /// <param name="mouseY">マウス位置</param>
        /// <returns>色</returns>
        private ColorRgba MakeColorFrom(int mouseX, int mouseY)
        {
            var sortedKeys = this.keys.OrderBy(x => x.Time).ToArray();
            if (sortedKeys.Any() == false)
            {
                return new ColorRgba { R = 1.0f, G = 1.0f, B = 1.0f, A = 1.0f };
            }

            var keyTime = this.MakeTime(mouseX, mouseY);

            if (keyTime <= sortedKeys.First().Time)
            {
                return sortedKeys.First().Color;
            }

            if (keyTime >= sortedKeys.Last().Time)
            {
                return sortedKeys.Last().Color;
            }

            for (var i = 0; i != sortedKeys.Count() - 1; ++i)
            {
                var leftKey  = sortedKeys[i + 0];
                var rightKey = sortedKeys[i + 1];

                if ((keyTime >= leftKey.Time) &&
                    (keyTime <= rightKey.Time))
                {
                    System.Diagnostics.Debug.Assert(leftKey.Time <= rightKey.Time, "leftKey.Time <= rightKey.Time");

                    var length = rightKey.Time - leftKey.Time;
                    var ratio  = (float)(keyTime - leftKey.Time) / length;

                    return new ColorRgba
                    {
                        R = ((rightKey.Color.R - leftKey.Color.R) * ratio) + leftKey.Color.R,
                        G = ((rightKey.Color.G - leftKey.Color.G) * ratio) + leftKey.Color.G,
                        B = ((rightKey.Color.B - leftKey.Color.B) * ratio) + leftKey.Color.B,
                        A = ((rightKey.Color.A - leftKey.Color.A) * ratio) + leftKey.Color.A
                    };
                }
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// ツールチップの準備
        /// </summary>
        private void PrepareToolTip()
        {
            this.tooltip = new ToolTip
            {
                BackColor = Color.WhiteSmoke,
                OwnerDraw = true
            };

            this.tooltip.Draw  += (s, e) =>
            {
                e.DrawBackground();
                e.DrawBorder();

                e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.AntiAlias;

                Rectangle rectIcon;
                Rectangle rectTitle;
                Rectangle rectText;

                var text = this.tooltip.GetToolTip(e.AssociatedControl);
                var icon = Properties.Resources.GradationEditor_Warning;

                MeasureToolTipSize(
                    icon,
                    Properties.Resources.GradationEditorWarningCaption,
                    text,
                    out rectIcon,
                    out rectTitle,
                    out rectText);

                using (var titleFont = new Font("Arial", 8, FontStyle.Bold))
                using (var textFont = new Font("Arial", 8, FontStyle.Regular))
                {
                    // Render the icon.
                    e.Graphics.DrawImage(
                        icon,
                        rectIcon.Location);

                    // Draw tooltip title string.
                    e.Graphics.DrawString(
                        Properties.Resources.GradationEditorWarningCaption,
                        titleFont,
                        Brushes.Black,
                        rectTitle.Location);

                    // Draw tooltip text.
                    e.Graphics.DrawString(
                        text,
                        textFont,
                        Brushes.Black,
                        rectText.Location);
                }
            };

            this.tooltip.Popup += (s, e) =>
            {
                Rectangle rectIcon;
                Rectangle rectTitle;
                Rectangle rectText;

                e.ToolTipSize =
                    MeasureToolTipSize(
                        Properties.Resources.GradationEditor_Warning,
                        Properties.Resources.GradationEditorWarningCaption,
                        this.tooltip.GetToolTip(e.AssociatedControl),
                        out rectIcon,
                        out rectTitle,
                        out rectText);
            };
        }

        /// <summary>
        /// Measure the size of the tooltip, along with the rectangles
        /// of the icon, title and the text.
        /// </summary>
        /// <param name="toolTipIcon">The icon for the tooltip.</param>
        /// <param name="toolTipTitle">The title for the tooltip.</param>
        /// <param name="toolTipText">The text for the tooltip.</param>
        /// <param name="rectIcon">The rectangle for the icon.</param>
        /// <param name="rectTitle">The rectangle for the title.</param>
        /// <param name="rectText">The rectangle for the text.</param>
        /// <returns>The size of the tooltip.</returns>
        private Size MeasureToolTipSize(Image toolTipIcon, string toolTipTitle, string toolTipText, out Rectangle rectIcon, out Rectangle rectTitle, out Rectangle rectText)
        {
            // 3 pixels of padding on each side.
            var size = new Size(6, 6);

            using (var titleFont = new Font("Arial", 8, FontStyle.Bold))
            using (var textFont = new Font("Arial", 8, FontStyle.Regular))
            {
                // Icon
                size.Width += toolTipIcon.Width;
                size.Height += toolTipIcon.Height;

                // Tooltip title
                var titleSize =
                    TextRenderer.MeasureText(toolTipTitle, titleFont);
                size.Width += titleSize.Width + 3;
                size.Height += Math.Max(0, titleSize.Height - size.Height);

                // Tooltip text
                var textSize =
                    TextRenderer.MeasureText(toolTipText, textFont);
                size.Width += textSize.Width + 3;
                size.Height += Math.Max(0, textSize.Height - size.Height);

                rectIcon = new Rectangle(
                    3,
                    (size.Height - toolTipIcon.Height) / 2,
                    toolTipIcon.Width,
                    toolTipIcon.Height);

                rectTitle = new Rectangle(
                    rectIcon.Right + 3,
                    ((size.Height - titleSize.Height) / 2) + 2,
                    titleSize.Width,
                    titleSize.Height);

                rectText = new Rectangle(
                    rectTitle.Right + 3,
                    ((size.Height - textSize.Height) / 2) + 2,
                    textSize.Width,
                    textSize.Height);
            }

            return size;
        }

        /// <summary>
        /// コンテキストメニュー初期化
        /// </summary>
        private void InitializeContextMenu()
        {
            this.ContextMenuStrip = ContextMenuUtility.CreateCopyPaste(
                () =>
                {
                    Clipboard.SetData(
                        ColorClipboardData.ClipboardFormat,
                        new ColorClipboardData
                        {
                            R = this.popupContextKey.Color.R,
                            G = this.popupContextKey.Color.G,
                            B = this.popupContextKey.Color.B,
                            A = this.popupContextKey.Color.A,
                            EnabledRgb = this.RgbEditEnabled,
                            EnabledAlpha = this.AlphaEditEnabled
                        });
                },
                () =>
                {
                    if (Clipboard.ContainsData(ColorClipboardData.ClipboardFormat))
                    {
                        var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                        var newValue = this.Value;

                        newValue[this.popupContextKey.Index].ValueAsColor = new ColorRgba(
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.R : 1.0f,
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.G : 1.0f,
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.B : 1.0f,
                            (this.AlphaEditEnabled && data.EnabledAlpha) ? data.A : 1.0f);

                        // 差分値を保持したまま値を更新し
                        this.changing = true;
                        this.Value = newValue;
                        this.changing = false;

                        // ここで履歴に積んで、差分値も更新する
                        this.NotifyValueChanged();
                    }
                });

            this.ContextMenuStrip.Opening += (s, e) =>
            {
                e.Cancel = this.popupContextKey == null;
                if (e.Cancel == false)
                {
                    var enabledPaste = Clipboard.ContainsData(ColorClipboardData.ClipboardFormat);

                    if (enabledPaste)
                    {
                        var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                        enabledPaste = (data.EnabledRgb == this.RgbEditEnabled) &&
                                       (data.EnabledAlpha == this.AlphaEditEnabled);
                    }

                    this.ContextMenuStrip.Items[ContextMenuUtility.CopyPasteMenuPasteIndex].Enabled = enabledPaste;
                }
            };
        }

        /// <summary>
        /// Handle GammaCorrectionChanged event from application.
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.Invalidate();
        }

        /// <summary>
        /// テーブルをコントロールにセットします。
        /// </summary>
        /// <param name="table">アニメーションテーブル</param>
        private void SetValueInstance(AnimationTableData table)
        {
            if (table == null)
            {
                this.keys.Clear();
            }
            else
            {
                this.keys = Enumerable.Range(0, table.Count).Select(x => table[x]).Select(
                    x => new GradationKey(this)
                    {
                        Color = x.ValueAsColor.Clone() as ColorRgba,
                        Time = x.Frame
                    }).ToList();

                this.PutIndex();
                this.ActiveIndex = table.SelectedKeyFrameIndex;
            }
        }

        /// <summary>
        /// OnValueChangedを呼び出して、履歴に積む値を確定します。
        /// </summary>
        private void NotifyValueChanged()
        {
            this.changing = true;

            var nowValue = this.Value;
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged, propertyName: "Value");
            this.OnValueChanged(new ValueChangedExEventArgs(
                this.prevTable,
                nowValue,
                false,
                this.TargetName));

            this.prevTable.Set(nowValue);

            this.changing = false;
        }
    }
}
