﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// キーの最大数に達したことを警告するツールチップです。
    /// </summary>
    internal class WarningToolTip
    {
        /// <summary>
        /// 警告ツールチップ
        /// </summary>
        private readonly ToolTip warningToolTip;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public WarningToolTip()
        {
            this.warningToolTip = new ToolTip
            {
                BackColor = Color.WhiteSmoke,
                OwnerDraw = true
            };
            this.PrepareToolTip();
        }

        /// <summary>
        /// 表示します。1秒後に自動的に消えます。
        /// </summary>
        /// <param name="control"></param>
        /// <param name="argLoc"></param>
        public void Show(IWin32Window control, PointF argLoc)
        {
            this.warningToolTip.Show(
                Properties.Resources.GradationEditorWarningMsgKeyframeExceedsMaxNum,
                control,
                (int)argLoc.X,
                (int)argLoc.Y - 16,
                1000);
        }

        /// <summary>
        /// ツールチップの準備
        /// </summary>
        private void PrepareToolTip()
        {
            this.warningToolTip.Draw += (s, e) =>
            {
                e.DrawBackground();
                e.DrawBorder();

                e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.AntiAlias;

                Rectangle rectIcon;
                Rectangle rectTitle;
                Rectangle rectText;

                var text = this.warningToolTip.GetToolTip(e.AssociatedControl);
                var icon = Properties.Resources.GradationEditor_Warning;

                MeasureToolTipSize(
                    icon,
                    Properties.Resources.GradationEditorWarningCaption,
                    text,
                    out rectIcon,
                    out rectTitle,
                    out rectText);

                using (var titleFont = new Font("Arial", 8, FontStyle.Bold))
                using (var textFont = new Font("Arial", 8, FontStyle.Regular))
                {
                    // Render the icon.
                    e.Graphics.DrawImage(
                        icon,
                        rectIcon.Location);

                    // Draw tooltip title string.
                    e.Graphics.DrawString(
                        Properties.Resources.GradationEditorWarningCaption,
                        titleFont,
                        Brushes.Black,
                        rectTitle.Location);

                    // Draw tooltip text.
                    e.Graphics.DrawString(
                        text,
                        textFont,
                        Brushes.Black,
                        rectText.Location);
                }
            };

            this.warningToolTip.Popup += (s, e) =>
            {
                Rectangle rectIcon;
                Rectangle rectTitle;
                Rectangle rectText;

                e.ToolTipSize =
                    MeasureToolTipSize(
                        Properties.Resources.GradationEditor_Warning,
                        Properties.Resources.GradationEditorWarningCaption,
                        this.warningToolTip.GetToolTip(e.AssociatedControl),
                        out rectIcon,
                        out rectTitle,
                        out rectText);
            };
        }

        /// <summary>
        /// ツールチップのサイズを計算します。
        /// </summary>
        /// <param name="toolTipIcon">アイコン</param>
        /// <param name="toolTipTitle">タイトル</param>
        /// <param name="toolTipText">テキスト</param>
        /// <param name="rectIcon">[OUT]アイコン描画領域</param>
        /// <param name="rectTitle">[OUT]タイトル描画領域</param>
        /// <param name="rectText">[OUT]テキスト描画領域</param>
        /// <returns>ツールチップ全体サイズ</returns>
        private static Size MeasureToolTipSize(
            Image toolTipIcon,
            string toolTipTitle,
            string toolTipText,
            out Rectangle rectIcon,
            out Rectangle rectTitle,
            out Rectangle rectText)
        {
            // 3 pixels of padding on each side.
            var size = new Size(6, 6);

            using (var titleFont = new Font("Arial", 8, FontStyle.Bold))
            using (var textFont = new Font("Arial", 8, FontStyle.Regular))
            {
                // Icon
                size.Width += toolTipIcon.Width;
                size.Height += toolTipIcon.Height;

                // Tooltip title
                var titleSize =
                    TextRenderer.MeasureText(toolTipTitle, titleFont);
                size.Width += titleSize.Width + 3;
                size.Height += Math.Max(0, titleSize.Height - size.Height);

                // Tooltip text
                var textSize =
                    TextRenderer.MeasureText(toolTipText, textFont);
                size.Width += textSize.Width + 3;
                size.Height += Math.Max(0, textSize.Height - size.Height);

                rectIcon = new Rectangle(
                    3,
                    (size.Height - toolTipIcon.Height) / 2,
                    toolTipIcon.Width,
                    toolTipIcon.Height);

                rectTitle = new Rectangle(
                    rectIcon.Right + 3,
                    ((size.Height - titleSize.Height) / 2) + 2,
                    titleSize.Width,
                    titleSize.Height);

                rectText = new Rectangle(
                    rectTitle.Right + 3,
                    ((size.Height - textSize.Height) / 2) + 2,
                    textSize.Width,
                    textSize.Height);
            }

            return size;
        }
    }
}
