﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Render.Renderable;

namespace EffectMaker.UIControls.Specifics.CurveEditor.RenderingElements
{
    /// <summary>
    /// マウスカーソルに追従する描画要素です。
    /// </summary>
    internal class VisualCursor : IRenderingElement
    {
        /// <summary>
        /// カーソル画像
        /// </summary>
        private readonly ImageRenderable cursorImage;

        /// <summary>
        /// カーソルに追従する軸ラベルX
        /// </summary>
        private readonly TextBlock cursorLabelX;

        /// <summary>
        /// カーソルに追従する軸ラベルY
        /// </summary>
        private readonly TextBlock cursorLabelY;

        /// <summary>
        /// 補助軸X
        /// </summary>
        private readonly LineSegment cursorX;

        /// <summary>
        /// 補助軸Y
        /// </summary>
        private readonly LineSegment cursorY;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public VisualCursor()
        {
            this.cursorImage = new ImageRenderable
            {
                Image = Properties.Resources.CurveEditor_Icon_Curve_Cursor_Normal,
                HAlignment = HAlignment.Center,
                VAlignment = VAlignment.Center,
                Padding = new Padding(1, 1, 0, 0)
            };
            this.cursorX = new LineSegment
            {
                BorderColor = Color.LightPink
            };
            this.cursorY = new LineSegment
            {
                BorderColor = Color.LightPink
            };
            this.cursorLabelX = new TextBlock
            {
                Font = new Font(FontFamily.GenericSansSerif, 9.0f),
                Text = "X",
                FillColor = Color.Transparent,
                BorderColor = Color.Transparent,
                TextColor = Color.Violet,
                AutoSize = true,
                HAlignment = HAlignment.Right
            };
            this.cursorLabelY = new TextBlock
            {
                Font = new Font(FontFamily.GenericSansSerif, 9.0f),
                Text = "Y",
                FillColor = Color.Transparent,
                BorderColor = Color.Transparent,
                TextColor = Color.Violet,
                AutoSize = true,
                HAlignment = HAlignment.Right
            };
        }

        /// <summary>
        /// カーソル位置
        /// </summary>
        public PointF Location
        {
            get { return this.cursorImage.Location; }
            set { this.cursorImage.Location = value; }
        }

        /// <summary>
        /// アップデート
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        /// <param name="nowMouse">マウス座標</param>
        /// <param name="normalizeAt">1.0をいくつの値とするか</param>
        /// <param name="withinInvalidArea">マウス座標が無効領域かどうか</param>
        /// <param name="preFix">値文字列のプレフィックス</param>
        /// <param name="postFix">値文字列のポストフィックス</param>
        public void Update(
            LayeredViewports viewports,
            PointF nowMouse,
            float normalizeAt,
            bool withinInvalidArea,
            string preFix,
            string postFix)
        {
            this.cursorImage.Image = withinInvalidArea
                ? Properties.Resources.CurveEditor_Icon_Curve_Cursor_NoAdd
                : Properties.Resources.CurveEditor_Icon_Curve_Cursor_Normal;

            PointF viewportPoint = viewports.Main.TransformPoint(nowMouse);
            PointF overlayPoint = viewports.Overlay.TransformPoint(nowMouse);
            PointF underlayPoint = viewports.Underlay.TransformPoint(nowMouse);

            float left = Constants.CalcTimeFromPos(viewportPoint.X);
            float top = Constants.CalcValueFromPos(viewportPoint.Y, normalizeAt);

            this.cursorX.Vertex1 = new PointF(-10000, overlayPoint.Y);
            this.cursorX.Vertex2 = new PointF(1000000.0f, overlayPoint.Y);
            this.cursorY.Vertex1 = new PointF(overlayPoint.X, 1000000.0f);
            this.cursorY.Vertex2 = new PointF(overlayPoint.X, -10000);

            this.cursorLabelX.Location = new PointF(
                underlayPoint.X + 19.0f,
                viewports.Overlay.Y + viewports.Overlay.Bounds.Height);
            this.cursorLabelX.Text = string.Format(
                "{0:d}{1}",
                (int)Math.Round(left),
                postFix);

            this.cursorLabelY.Location = new PointF(
                viewports.Main.X + 2.0f,
                underlayPoint.Y - 12.0f);
            this.cursorLabelY.Text = string.Format(preFix + "{0:f2}", top);
        }

        /// <summary>
        /// 表示します。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Show(LayeredViewports viewports)
        {
            this.Hide(viewports);
            viewports.Overlay.AddRenderable(this.cursorX);
            viewports.Overlay.AddRenderable(this.cursorY);
            viewports.Overlay.AddRenderable(this.cursorImage);
            viewports.Underlay.AddRenderable(this.cursorLabelX);
            viewports.Underlay.AddRenderable(this.cursorLabelY);
        }

        /// <summary>
        /// 非表示にします。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Hide(LayeredViewports viewports)
        {
            viewports.Overlay.RemoveRenderable(this.cursorX);
            viewports.Overlay.RemoveRenderable(this.cursorY);
            viewports.Overlay.RemoveRenderable(this.cursorImage);
            viewports.Underlay.RemoveRenderable(this.cursorLabelX);
            viewports.Underlay.RemoveRenderable(this.cursorLabelY);
        }
    }
}
