﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics.CurveEditor.RenderingElements
{
    /// <summary>
    /// ビューポート上の無効領域を表す描画要素です。
    /// </summary>
    internal class InvalidArea : IRenderingElement
    {
        /// <summary>
        /// 無効エリア・左
        /// </summary>
        private readonly RectangleShape left;

        /// <summary>
        /// 無効エリア・上
        /// </summary>
        private readonly RectangleShape top;

        /// <summary>
        /// 無効エリア・右
        /// </summary>
        private readonly RectangleShape right;

        /// <summary>
        /// 無効エリア・下
        /// </summary>
        private readonly RectangleShape bottom;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InvalidArea()
        {
            // 非使用領域の色塗り
            this.left = new RectangleShape
            {
                FillColor = Color.Lavender,
                BorderThickness = 0
            };
            this.right = new RectangleShape
            {
                FillColor = Color.Lavender,
                BorderThickness = 0
            };
            this.bottom = new RectangleShape
            {
                FillColor = Color.Lavender,
                BorderThickness = 0
            };
            this.top = new RectangleShape
            {
                FillColor = Color.Lavender,
                BorderThickness = 0
            };
        }

        /// <summary>
        /// 下側の領域が表示されているか否かを取得します。
        /// </summary>
        public bool BottomVisible
        {
            get
            {
                return MathUtility.IsZero(this.bottom.Size.Height, 0.00000001f);
            }
        }

        /// <summary>
        /// 表示します。
        /// 既に表示済みにビューポートならば一旦削除して描画順を最後にします。
        /// </summary>
        /// <param name="viewports">表示するビューポート</param>
        public void Show(LayeredViewports viewports)
        {
            this.Hide(viewports);
            viewports.Main.AddRenderable(this.left);
            viewports.Main.AddRenderable(this.top);
            viewports.Main.AddRenderable(this.right);
            viewports.Main.AddRenderable(this.bottom);
        }

        /// <summary>
        /// 非表示にします。
        /// </summary>
        /// <param name="viewports">非表示にするビューポート</param>
        public void Hide(LayeredViewports viewports)
        {
            viewports.Main.RemoveRenderable(this.left);
            viewports.Main.RemoveRenderable(this.top);
            viewports.Main.RemoveRenderable(this.right);
            viewports.Main.RemoveRenderable(this.bottom);
        }

        /// <summary>
        /// 引数に渡した座標が無効領域内かどうかを判定します。
        /// </summary>
        /// <param name="argPos">判定したい座標</param>
        /// <returns>領域内ならtrue,領域外ならfalse.</returns>
        public bool Pick(PointF argPos)
        {
            return this.left.Pick(argPos) ||
                this.top.Pick(argPos) ||
                this.right.Pick(argPos) ||
                this.bottom.Pick(argPos);
        }

        /// <summary>
        /// 領域を更新します。
        /// </summary>
        /// <param name="viewport">ビューポート</param>
        /// <param name="enabled">エディタの有効状態</param>
        /// <param name="maxTime">時間の上限座標</param>
        /// <param name="maxLimit">値の上限座標</param>
        /// <param name="minLimit">値の下限座標</param>
        public void Update(Viewport viewport, bool enabled, float maxTime, float maxLimit, float minLimit)
        {
            if (!enabled)
            {
                this.left.Size = new SizeF(0.0f, 0.0f);
                this.right.Size = new SizeF(0.0f, 0.0f);
                this.top.Size = new SizeF(0.0f, 0.0f);
                this.bottom.Size = new SizeF(0.0f, 0.0f);

                return;
            }

            var posLT = viewport.TransformPoint(viewport.Location);
            var posLB = viewport.TransformPoint(new PointF(viewport.X, viewport.Y + viewport.Height));
            var posRT = viewport.TransformPoint(new PointF(viewport.X + viewport.Width, viewport.Y));

            this.left.Location = posLT;
            this.left.Size = new SizeF(
                Math.Max(0, Constants.ViewportOriginX - posLT.X),
                Math.Max(0, posLB.Y - posLT.Y));

            this.right.Location = new PointF(maxTime, posLT.Y);
            this.right.Size = new SizeF(
                Math.Max(0, posRT.X - Constants.ViewportOriginX - maxTime),
                Math.Max(0, posLB.Y - posLT.Y));

            this.top.Location = posLT;
            this.top.Size = new SizeF(
                Math.Max(0, posRT.X - posLT.X),
                Math.Max(0, maxLimit - posLT.Y));

            this.bottom.Location = new PointF(posLB.X, minLimit);
            this.bottom.Size = new SizeF(
                Math.Max(0, posRT.X - posLT.X),
                Math.Max(0, posLB.Y - minLimit));
        }
    }
}
