﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// 階層化されたビューポートを管理するクラスです。
    /// </summary>
    internal class LayeredViewports
    {
        /// <summary>
        /// リセット時のビューポート位置
        /// </summary>
        private PointF initialTrans;

        /// <summary>
        /// 拡縮の中心座標
        /// </summary>
        private PointF zoomBasePoint;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="control"></param>
        public LayeredViewports(Control control)
        {
            this.Main = new Viewport(control)
            {
                FillColor = Color.White,
                BorderThickness = 0
            };
            this.Overlay = new Viewport(control)
            {
                FillColor = Color.Transparent,
                BorderColor = Color.Black,
                BorderThickness = 1
            };
            this.Underlay = new Viewport(control)
            {
                FillColor = Color.Transparent,
                BorderThickness = 0
            };
        }

        /// <summary>
        /// メインビューポート
        /// </summary>
        public Viewport Main { get; private set; }

        /// <summary>
        /// オーバーレイビューポート
        /// </summary>
        public Viewport Overlay { get; private set; }

        /// <summary>
        /// アンダーレイビューポート
        /// </summary>
        public Viewport Underlay { get; private set; }

        /// <summary>
        /// レンダー部描画時の更新領域
        /// </summary>
        public Rectangle RenderRect { get; private set; }

        /// <summary>
        /// スクロール開始座標
        /// </summary>
        public PointF PrevScrollingPosition { get; set; }

        /// <summary>
        /// ドラッグ開始座標
        /// </summary>
        public PointF DragStartPosition { get; set; }

        /// <summary>
        /// ホイールズームが開始しているか否か
        /// </summary>
        public bool IsZooming { get; set; }

        /// <summary>
        /// ビューポートの平行移動量が初期状態か否か
        /// </summary>
        public bool IsInitialTranslation
        {
            get
            {
                return Math.Abs(this.Main.Translation.X - this.initialTrans.X) > 0.01f
                    || Math.Abs(this.Main.Translation.Y - this.initialTrans.Y) > 0.01f;
            }
        }

        /// <summary>
        /// ビューポートを描画します。
        /// </summary>
        /// <param name="g">グラフィクスコンテキスト</param>
        public void Draw(Graphics g)
        {
            this.Underlay.ResumeUpdateAndRendering();
            this.Underlay.Update(g);
            this.Underlay.Draw(g);
            this.Underlay.SuspendUpdateAndRendering();

            this.Main.ResumeUpdateAndRendering();
            this.Main.Update(g);
            this.Main.Draw(g);
            this.Main.SuspendUpdateAndRendering();

            this.Overlay.ResumeUpdateAndRendering();
            this.Overlay.Update(g);
            this.Overlay.Draw(g);
            this.Overlay.SuspendUpdateAndRendering();
        }

        /// <summary>
        /// ビューポートをリサイズします。
        /// </summary>
        /// <param name="control">コントロール</param>
        public void Resize(Control control)
        {
            const int x = 75;
            const int y = 65;
            const int rightMargin = 20;
            const int bottomMargin = 25;

            // メインとオーバーレイはカーブ編集領域をカバー
            this.Main.Bounds = new RectangleF(
                x,
                y,
                Math.Max(200.0f, control.Width - (x + rightMargin)),
                Math.Max(150.0f, control.Height - (y + bottomMargin)));
            this.Overlay.Bounds = new RectangleF(
                x,
                y,
                this.Main.Width + 1,
                this.Main.Height + 1);

            // アンダーレイは全領域をカバー
            this.Underlay.Bounds = new RectangleF(
                0.0f,
                0.0f,
                control.ClientRectangle.Width,
                control.ClientRectangle.Height);

            // 描画領域を更新
            this.RenderRect = new Rectangle(
                0,
                (int)this.Main.Y - 9,
                control.ClientRectangle.Width,
                control.ClientRectangle.Height - (int)this.Main.Y + 9);
        }

        /// <summary>
        /// スクロールを行います。
        /// </summary>
        /// <param name="argLoc">スクロール後の基準位置</param>
        /// <param name="scale">スケールの参照</param>
        /// <param name="validScale">有効スケールの取得処理</param>
        public void Scroll(PointF argLoc, ref PointF scale, Func<PointF> validScale)
        {
            if ((Control.ModifierKeys & Keys.Alt) == Keys.Alt)
            {
                PointF beforeTrans = this.Main.TransformPoint(argLoc);

                float moveX = argLoc.X - this.PrevScrollingPosition.X;
                float moveY = argLoc.Y - this.PrevScrollingPosition.Y;

                // ALT+ドラッグで軸ごとにスケール変更する時点で記憶しているスケールを破棄する
                if (Math.Abs(moveX) < 0.0001f || Math.Abs(moveY) < 0.0001f)
                {
                    scale = validScale();
                }

                scale.X *= 1.0f + (moveX / 100.0f);
                scale.Y *= 1.0f - (moveY / 100.0f);

                this.Main.Scale = validScale();
                this.Main.Update(null);

                var afterTrans = this.Main.TransformPoint(argLoc);

                var diffX = beforeTrans.X - afterTrans.X;
                var diffY = beforeTrans.Y - afterTrans.Y;

                var nowTrans = this.Main.Translation;
                this.Main.Translation = new PointF(nowTrans.X - diffX, nowTrans.Y - diffY);
            }
            else
            {
                var nowTrans = this.Main.Translation;
                var tmpScale = validScale();
                nowTrans.X += (argLoc.X - this.PrevScrollingPosition.X) / tmpScale.X;
                nowTrans.Y += (argLoc.Y - this.PrevScrollingPosition.Y) / tmpScale.Y;
                this.Main.Translation = nowTrans;
            }

            this.PrevScrollingPosition = argLoc;
        }

        /// <summary>
        /// ホイールズームを行います。
        /// </summary>
        /// <param name="delta">ホイールの回転量</param>
        /// <param name="originPos">拡縮基準位置</param>
        /// <param name="zeroPinMode">0固定モード</param>
        /// <param name="scale">スケールの参照</param>
        /// <param name="validScale">有効スケールの取得処理</param>
        public void Zoom(int delta, PointF originPos, int zeroPinMode, ref PointF scale, Func<PointF> validScale)
        {
            float zoomCenterY = this.Main.Y + (this.Main.Height * 0.5f);
            switch (zeroPinMode)
            {
                case 0:
                case -1:
                case 2:
                    zoomCenterY = originPos.Y;
                    break;
            }

            PointF basePos = new PointF(originPos.X, zoomCenterY);
            PointF beforeTrans = this.Main.TransformPoint(basePos);

            if (!this.IsZooming)
            {
                this.zoomBasePoint = beforeTrans;
                this.IsZooming = true;
            }

            if (delta > 0)
            {
                scale.X *= 1.1f;
                scale.Y *= 1.1f;
            }
            else
            {
                scale.X *= 0.9f;
                scale.Y *= 0.9f;
            }

            this.Main.Scale = validScale();
            this.Main.Update(null);

            var afterTrans = this.Main.TransformPoint(basePos);

            var diffX = this.zoomBasePoint.X - afterTrans.X;
            var diffY = this.zoomBasePoint.Y - afterTrans.Y;

            var nowTrans = this.Main.Translation;
            this.Main.Translation = new PointF(nowTrans.X - diffX, nowTrans.Y - diffY);
        }

        /// <summary>
        /// ビューポートの平行移動量を現状で初期状態とします。
        /// </summary>
        public void ResetInitialTranslation()
        {
            this.initialTrans = this.Main.Translation;
        }

        /// <summary>
        /// メインビューポートの座標系からオーバーレイビューポートの座標系へ
        /// </summary>
        /// <param name="point">メインビューポートの座標系の座標値</param>
        /// <returns>オーバーレイビューポートの座標系の座標値</returns>
        public PointF TransformPointMainToOverlay(PointF point)
        {
            return this.Overlay.TransformPoint(this.Main.InverseTransformPoint(point));
        }

        /// <summary>
        /// メインビューポート内にクランプしたマウス座標を得ます。
        /// </summary>
        /// <returns>クランプしたマウス座標</returns>
        public PointF GetClampedMousePosition()
        {
            var pos = this.Main.RenderingControl.PointToClient(Cursor.Position);
            pos.X = (int)MathUtility.Clamp(pos.X, this.Main.X, this.Main.X + this.Main.Width);
            pos.Y = (int)MathUtility.Clamp(pos.Y, this.Main.Y, this.Main.Y + this.Main.Height);

            return pos;
        }

        /// <summary>
        /// マウスがビューポート内にいるかどうかを判定します。
        /// </summary>
        /// <param name="argPos">マウス座標</param>
        /// <returns>ビューポート内にいるならtrue,そうでなければfalse.</returns>
        public bool IsInnerMouseSpace(PointF argPos)
        {
            var viewportPos = new PointF(
                argPos.X - this.Main.Location.X,
                argPos.Y - this.Main.Location.Y);

            return this.Overlay.Pick(viewportPos);
        }

        /// <summary>
        /// ビューポートのギリギリ端っこにいるノードの座標を画面中心方向に少しずらします。
        /// </summary>
        /// <param name="input">ノードの座標</param>
        /// <returns>補正後の座標</returns>
        public PointF AdjustBorderVertex(PointF input)
        {
            var tmpV = this.Main.InverseTransformPoint(input);
            if (tmpV.X <= this.Main.Location.X)
            {
                tmpV.X += 1.5f;
            }
            else if (tmpV.X >= this.Main.Location.X + this.Main.Width)
            {
                tmpV.X -= 1.5f;
            }

            if (tmpV.Y <= this.Main.Location.Y)
            {
                tmpV.Y += 1.5f;
            }
            else if (tmpV.Y >= this.Main.Location.Y + this.Main.Height)
            {
                tmpV.Y -= 1.5f;
            }

            return this.Main.TransformPoint(tmpV);
        }
    }
}
