﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Linq;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Renderable;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// カーブの形状1本分を表現するクラス.
    /// </summary>
    internal class CurveShape
    {
        /// <summary>
        /// エディタへの参照.
        /// </summary>
        private readonly CurveEditor editor;

        /// <summary>
        /// ビューポート座標系での頂点配列.
        /// </summary>
        private readonly List<PointF> vertices;

        /// <summary>
        /// ノードの形状コレクション
        /// </summary>
        private readonly List<RectangleShape> nodeShapes;

        /// <summary>
        /// コネクターの形状コレクション
        /// </summary>
        private readonly List<LineSegment> connectors;

        /// <summary>
        /// スプラインカーブ表現
        /// </summary>
        private readonly CurveRenderable sprineCurve;

        /// <summary>
        /// ノードの塗りつぶしカラー.
        /// </summary>
        private readonly Color fillColor;

        /// <summary>
        /// ノードの縁取りカラー.
        /// </summary>
        private readonly Color borderColor;

        /// <summary>
        /// 編集対象外の時のカラー.
        /// </summary>
        private readonly Color deactiveColor;

        /// <summary>
        /// カーブ1本分を表すインスタンスのコンストラクタ.
        /// </summary>
        /// <param name="argEditor">親エディタ</param>
        /// <param name="argFill">塗りつぶしカラー</param>
        /// <param name="argBorder">縁取りカラー</param>
        /// <param name="argDeactive">編集対象外カラー</param>
        public CurveShape(CurveEditor argEditor, Color argFill, Color argBorder, Color argDeactive)
        {
            this.editor = argEditor;
            this.vertices = new List<PointF>();
            this.nodeShapes = new List<RectangleShape>();
            this.connectors = new List<LineSegment>();
            if (this.editor.EditorMode == CurveEditor.EditorModes.EmitterTimeAnim)
            {
                this.sprineCurve = new CurveRenderable()
                {
                    BorderColor = argFill
                };
            }

            this.fillColor = argFill;
            this.borderColor = argBorder;
            this.deactiveColor = argDeactive;
        }

        /// <summary>
        /// ビューポートにノード形状を登録
        /// </summary>
        /// <param name="argVP">描画するビューポート</param>
        public void ShowNode(LayeredViewports argVP)
        {
            argVP.Overlay.RemoveRenderableRange(this.nodeShapes);
            if (this.editor.Enabled)
            {
                argVP.Overlay.AddRenderableRange(this.nodeShapes);
            }
        }

        /// <summary>
        /// ビューポートにライン形状を登録
        /// 描画順制御のため、ノードはいったん削除されます。
        /// ノードも描画する場合はShowNodeをこれの後に呼んでください。
        /// </summary>
        /// <param name="argVP">描画するビューポート</param>
        /// <param name="argList">重複線分を検出するためのリスト</param>
        /// <param name="argActive">編集対象か否か</param>
        public void ShowLines(LayeredViewports argVP, List<LineSegment> argList, bool argActive)
        {
            argVP.Main.RemoveRenderableRange(this.connectors);
            argVP.Overlay.RemoveRenderableRange(this.nodeShapes);
            if (this.sprineCurve != null)
            {
                argVP.Main.RemoveRenderable(this.sprineCurve);
            }

            if (this.editor.InterMode == 0)
            {
                foreach (var mine in this.connectors)
                {
                    bool discovered = argList.Any(line => line.Vertex1 == mine.Vertex1 && line.Vertex2 == mine.Vertex2);

                    if (!discovered)
                    {
                        mine.BorderColor = argActive ? this.fillColor : this.deactiveColor;
                        if (this.editor.Enabled)
                        {
                            argVP.Main.AddRenderable(mine);
                        }

                        argList.Add(mine);
                    }
                }
            }
            else
            {
                if (this.vertices.Count > 0 && this.sprineCurve != null)
                {
                    this.sprineCurve.BorderColor = argActive ? this.fillColor : this.deactiveColor;
                    this.sprineCurve.Vertices = this.vertices;
                    if (this.editor.Enabled)
                    {
                        argVP.Main.AddRenderable(this.sprineCurve);
                    }
                }
            }
        }

        /// <summary>
        /// ビューポートからノードとラインを削除する
        /// </summary>
        /// <param name="argVP">削除するビューポート</param>
        public void HideAll(LayeredViewports argVP)
        {
            argVP.Main.RemoveRenderableRange(this.connectors);
            argVP.Overlay.RemoveRenderableRange(this.nodeShapes);
            if (this.sprineCurve != null)
            {
                argVP.Main.RemoveRenderable(this.sprineCurve);
            }
        }

        /// <summary>
        /// キーの追加
        /// </summary>
        /// <param name="viewports">対象ビューポート</param>
        /// <param name="argLoc">座標</param>
        /// <param name="active">編集対象か否か</param>
        /// <param name="noAdjust">先頭、末尾にキーを追加した際の補正処理を行うか否か</param>
        /// <returns>追加したノードの形状</returns>
        public RectangleShape AddKey(LayeredViewports viewports, PointF argLoc, bool active, bool noAdjust = false)
        {
            // 最大個数より多くなるようだったらnullを返して何もしない。
            if (this.nodeShapes.Count >= this.editor.MaxNodeNum)
            {
                return null;
            }

            var nodePos = viewports.Main.TransformPoint(argLoc);
            bool eval = (this.editor.EditorMode == 0) &&
                        (Math.Round(nodePos.X) > Constants.ViewportOriginX + Constants.ViewportFullWidth);
            if ((Math.Round(nodePos.X) < Constants.ViewportOriginX ||
                eval ||
                this.editor.CalcValueFromPos(nodePos.Y) > this.editor.MaxLimit ||
                this.editor.CalcValueFromPos(nodePos.Y) < this.editor.MinLimit) &&
                !noAdjust)
            {
                return null;
            }

            nodePos.X = this.editor.SnapTime(nodePos.X);
            nodePos.Y = this.editor.SnapValue(nodePos.Y);

            var nodeShape = new RectangleShape
            {
                BorderColor = this.borderColor,
                FillColor = this.fillColor,
                Location = viewports.TransformPointMainToOverlay(nodePos) - Constants.NodeCenterOffset,
                Size = Constants.NodeSize
            };

            if (active)
            {
                if (this.editor.Enabled)
                {
                    viewports.Overlay.AddRenderable(nodeShape);
                }

                viewports.Main.Update(null);
                viewports.Overlay.Update(null);
            }

            // 重複キー対策
            if (!noAdjust)
            {
                nodePos = this.HandleOverlapKeys(nodePos);
            }

            bool inserted = this.AddNodeCore(viewports, nodeShape, nodePos, true, active);

            // 非編集対象の軸に関しては、前後のキーの値をそのまま継承する
            // データを直接ビューに反映する際はnoAdjustをtrueにして無効化する
            if (!active && !noAdjust && !inserted)
            {
                int adjustIndex;
                int index = this.nodeShapes.IndexOf(nodeShape);
                if (index - 1 < 0)
                {
                    adjustIndex = index + 1;
                }
                else
                {
                    adjustIndex = index - 1;
                }

                this.vertices[index] = new PointF(
                    this.vertices[index].X,
                    this.vertices[adjustIndex].Y);
                this.nodeShapes[index].Location = new PointF(
                    this.nodeShapes[index].Location.X,
                    this.nodeShapes[adjustIndex].Location.Y);
            }

            this.Reconnect(viewports);

            return nodeShape;
        }

        /// <summary>
        /// キーの削除
        /// </summary>
        /// <param name="viewports">対象ビューポート</param>
        /// <param name="argIndex">対象インデックス</param>
        /// <returns>削除できたらtrue,できなかったらfalse.</returns>
        public bool RemoveKey(LayeredViewports viewports, int argIndex)
        {
            if (this.vertices.Count <= 1)
            {
                return false;
            }

            var node = this.nodeShapes[argIndex];
            viewports.Overlay.RemoveRenderable(node);
            node.Dispose();

            this.vertices.RemoveAt(argIndex);
            this.nodeShapes.RemoveAt(argIndex);
            this.Reconnect(viewports);

            return true;
        }

        /// <summary>
        /// キーの移動
        /// </summary>
        /// <param name="viewports">描画ビューポート</param>
        /// <param name="argNode">対象ノード</param>
        /// <param name="argLoc">移動先座標</param>
        /// <param name="noAdjust">補正処理を行うか否か</param>
        /// <returns>ノード移動後の中心座標</returns>
        public PointF MoveKey(LayeredViewports viewports, RectangleShape argNode, PointF argLoc, bool noAdjust = false)
        {
            var index = this.nodeShapes.IndexOf(argNode);
            if (index == -1)
            {
                throw new InvalidDataException();
            }

            var nodePos = this.ClampNodePos(viewports.Main.TransformPoint(argLoc));

            nodePos.X = this.editor.SnapTime(nodePos.X);
            nodePos.Y = this.editor.SnapValue(nodePos.Y, index);

            this.vertices.RemoveAt(index);
            this.nodeShapes.RemoveAt(index);

            // 重複キー対策
            if (!noAdjust)
            {
                nodePos = this.HandleOverlapKeys(nodePos);
            }

            this.AddNodeCore(viewports, argNode, nodePos, false);

            this.Reconnect(viewports);

            return argNode.Location + Constants.NodeCenterOffset;
        }

        /// <summary>
        /// 非編集対象カーブのキーを移動
        /// ToDo:ここで非編集対象カーブのキーを滑らせる処理を実装できそう
        /// </summary>
        /// <param name="viewports">描画ビューポート</param>
        /// <param name="argIndex">対象インデックス</param>
        /// <param name="argLoc">移動先座標</param>
        public void MoveKeyFrameCombined(LayeredViewports viewports, int argIndex, PointF argLoc)
        {
            if (argIndex >= this.nodeShapes.Count)
            {
                throw new InvalidDataException();
            }

            var node = this.nodeShapes[argIndex];
            var nodePos = this.ClampNodePos(viewports.Main.TransformPoint(argLoc));

            nodePos.X = this.editor.SnapTime(nodePos.X);
            nodePos.Y = this.vertices[argIndex].Y;

            this.vertices.RemoveAt(argIndex);
            this.nodeShapes.RemoveAt(argIndex);
            this.AddNodeCore(viewports, node, this.HandleOverlapKeys(nodePos), false);
            this.Reconnect(viewports);
        }

        /// <summary>
        /// リセットする
        /// </summary>
        /// <param name="argVP">対象ビューポート</param>
        public void Reset(LayeredViewports argVP)
        {
            bool result = true;
            do
            {
                result = this.RemoveKey(argVP, 0);
            }
            while (result);

            if (this.nodeShapes.Count < 1)
            {
                return;
            }

            this.MoveKey(
                argVP,
                this.nodeShapes[0],
                argVP.Main.InverseTransformPoint(new PointF(
                    this.editor.CalcPosFromTime(0.0f),
                    this.editor.CalcPosFromValue(this.editor.DefaultValue))),
                    true);
        }

        /// <summary>
        /// ノードが指す座標値を取得
        /// </summary>
        /// <param name="argNode">ノード</param>
        /// <returns>座標値</returns>
        public PointF GetKeyPos(RectangleShape argNode)
        {
            int index = this.nodeShapes.IndexOf(argNode);
            if (index == -1)
            {
                return new PointF(-1, -1);
            }

            return this.vertices[this.nodeShapes.IndexOf(argNode)];
        }

        /// <summary>
        /// ノードが指すキーインデックスを取得
        /// </summary>
        /// <param name="argNode">ノード</param>
        /// <returns>キーインデックス</returns>
        public int GetIndex(RectangleShape argNode)
        {
            return this.nodeShapes.IndexOf(argNode);
        }

        /// <summary>
        /// ノードの中心座標と大きさをエディタのスケールに併せて調整します。
        /// </summary>
        public void Rescale(LayeredViewports argVP)
        {
            foreach (var nodeShape in this.nodeShapes)
            {
                nodeShape.Location =
                    argVP.TransformPointMainToOverlay(this.vertices[this.nodeShapes.IndexOf(nodeShape)]) - Constants.NodeCenterOffset;
            }
        }

        /// <summary>
        /// ノードの選択
        /// </summary>
        /// <param name="argVP">対象ビューポート</param>
        /// <param name="argLoc">ピックされたマウス座標</param>
        /// <returns>選択されたノードがあればそのノードを、無ければnullを返す。</returns>
        public RectangleShape Selection(LayeredViewports argVP, PointF argLoc)
        {
            foreach (var node in this.nodeShapes)
            {
                // ノードの座標をスクリーン座標変換
                var nodeToScreen = argVP.Main.InverseTransformPoint(this.vertices[this.nodeShapes.IndexOf(node)]);

                // スクリーン座標同士で判定
                if (Math.Abs(argLoc.X - nodeToScreen.X) <= Constants.PickRange &&
                    Math.Abs(argLoc.Y - nodeToScreen.Y) <= Constants.PickRange)
                {
                    return node;
                }
            }

            return null;
        }

        /// <summary>
        /// キー座標配列をアニメーションデータ空間に変換して取得します。
        /// </summary>
        /// <returns>アニメーションデータ配列</returns>
        public PointF[] GetValues()
        {
            return this.vertices.Select(point => new PointF(
                    this.editor.CalcTimeFromPos(point.X),
                    this.editor.CalcValueFromPos(point.Y))).ToArray();
        }

        /// <summary>
        /// ノード配列を取得します。
        /// </summary>
        /// <returns>ノード配列</returns>
        public List<RectangleShape> GetNodes()
        {
            return this.nodeShapes;
        }

        /// <summary>
        /// ノード間を接続する線分を更新します。
        /// </summary>
        /// <param name="viewports">階層化されたビューポート</param>
        public void Reconnect(LayeredViewports viewports)
        {
            if (this.sprineCurve != null)
            {
                viewports.Main.RemoveRenderable(this.sprineCurve);
            }

            viewports.Main.RemoveRenderableRange(this.connectors);
            this.connectors.Clear();

            if (this.vertices.Count == 0)
            {
                Logger.Log(LogLevels.Warning, "Invalid animation state: Count = 0");
                return;
            }

            if (this.editor.InterMode == 0)
            {
                for (int i = 1; i < this.nodeShapes.Count; ++i)
                {
                    var connector = new LineSegment
                    {
                        Vertex1 = viewports.AdjustBorderVertex(this.vertices[i - 1]),
                        Vertex2 = viewports.AdjustBorderVertex(this.vertices[i]),
                        BorderColor = this.fillColor
                    };
                    this.connectors.Add(connector);
                }

                // 先頭キーから時間マイナス軸方向への破線
                this.connectors.Add(new LineSegment
                {
                    Vertex1 = this.vertices[0],
                    Vertex2 = new PointF(-10000.0f, this.vertices[0].Y),
                    BorderColor = this.fillColor,
                    DashStyle = DashStyle.Dash
                });

                // 終端キーから時間プラス軸方向への破線
                this.connectors.Add(new LineSegment
                {
                    Vertex1 = this.vertices.Last(),
                    Vertex2 = new PointF(this.vertices.Last().X + 1000000.0f, this.vertices.Last().Y),
                    BorderColor = this.fillColor,
                    DashStyle = DashStyle.Dash
                });

                if (this.editor.Enabled)
                {
                    viewports.Main.AddRenderableRange(this.connectors);
                }
            }
        }

        /// <summary>
        /// 等しいカーブかどうかを評価します.
        /// </summary>
        /// <param name="other">他のカーブ</param>
        /// <returns>等しかったらtrue,等しくなかったらfalse.</returns>
        public bool Equals(CurveShape other)
        {
            if (this.vertices.Count != other.vertices.Count)
            {
                return false;
            }

            for (int i = 0; i < this.vertices.Count; ++i)
            {
                if (Math.Abs(this.vertices[i].X - other.vertices[i].X) >= 0.001
                    || Math.Abs(this.vertices[i].Y - other.vertices[i].Y) >= 0.001)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 重複するキーを前後にずらす調整を行います。
        /// </summary>
        /// <param name="nodePos">変更のあったノードの座標値</param>
        /// <returns>調整後の座標値</returns>
        private PointF HandleOverlapKeys(PointF nodePos)
        {
            for (int i = 0; i < this.nodeShapes.Count; ++i)
            {
                var v = (float)Math.Round(this.editor.CalcTimeFromPos(this.vertices[i].X));
                var n = (float)Math.Round(this.editor.CalcTimeFromPos(nodePos.X));

                int maxFrame = this.editor.EditorMode == 0 ? Constants.ParticleAnimMax : Constants.EmitterAnimMax;

                if ((int)v != (int)n)
                {
                    continue;
                }

                int adjust = (int)n + 1;
                bool subtract = (adjust > maxFrame) || (i >= this.nodeShapes.Count);
                for (int j = i + 1; j < this.nodeShapes.Count; ++j)
                {
                    var vv = (int)Math.Round(this.editor.CalcTimeFromPos(this.vertices[j].X));
                    if (adjust >= 0 && adjust <= maxFrame && vv != adjust)
                    {
                        break;
                    }

                    ++adjust;

                    if (adjust > maxFrame)
                    {
                        subtract = true;
                        break;
                    }
                }

                if (subtract)
                {
                    adjust = (int)n - 1;
                    for (int j = i - 1; j >= 0; --j)
                    {
                        var vv = (int)Math.Round(this.editor.CalcTimeFromPos(this.vertices[j].X));
                        if (adjust >= 0 && adjust <= maxFrame && vv != adjust)
                        {
                            break;
                        }

                        --adjust;

                        if (adjust < 0)
                        {
                            throw new InvalidOperationException();
                        }
                    }
                }

                nodePos.X = (float)Math.Round(this.editor.CalcPosFromTime(adjust));
                break;
            }

            return nodePos;
        }

        /// <summary>
        /// 挿入を考慮しつつキーを追加します。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        /// <param name="nodeShape">矩形の左上座標</param>
        /// <param name="nodePos">ノードの実際の位置</param>
        /// <param name="insertMode">キー挿入時の分割処理を有効にするフラグ</param>
        /// <param name="active">キーが編集対象か否か</param>
        /// <returns>キーの挿入が発生したらtrue、しなかったらfalse.</returns>
        private bool AddNodeCore(LayeredViewports viewports, RectangleShape nodeShape, PointF nodePos, bool insertMode, bool active = true)
        {
            bool inserted = false;

            for (int i = 0; i < this.nodeShapes.Count; ++i)
            {
                if (this.vertices[i].X > nodePos.X)
                {
                    // 編集対象でない場合は前後のキーの値の補間値を採用しない
                    if (!active)
                    {
                        // キーの挿入時にカーブの最近点を求める
                        if (i != 0 && insertMode)
                        {
                            var len = this.vertices[i].X - this.vertices[i - 1].X;
                            var t = len > 0.0f ? (nodePos.X - this.vertices[i - 1].X) / len : 0.0f;

                            nodePos.Y = ((1.0f - t) * this.vertices[i - 1].Y) + (t * this.vertices[i].Y);
                        }
                    }

                    nodeShape.Location = viewports.TransformPointMainToOverlay(nodePos) - Constants.NodeCenterOffset;
                    this.vertices.Insert(i, nodePos);
                    this.nodeShapes.Insert(i, nodeShape);
                    inserted = true;

                    break;
                }
            }

            if (!inserted)
            {
                nodeShape.Location = viewports.TransformPointMainToOverlay(nodePos) - Constants.NodeCenterOffset;
                this.vertices.Add(nodePos);
                this.nodeShapes.Add(nodeShape);
            }

            return inserted;
        }

        /// <summary>
        /// ノードの座標をクランプします。
        /// </summary>
        /// <param name="nodePos">ノード座標</param>
        /// <returns>クランプした座標</returns>
        private PointF ClampNodePos(PointF nodePos)
        {
            // 左端クランプ
            if (nodePos.X < Constants.ViewportOriginX)
            {
                nodePos.X = Constants.ViewportOriginX;
            }

            if (this.editor.EditorMode == 0)
            {
                // パーティクル時間アニメなら右端は100%でクランプ
                if (nodePos.X > Constants.ViewportFullWidth + Constants.ViewportOriginX)
                {
                    nodePos.X = Constants.ViewportFullWidth + Constants.ViewportOriginX;
                }
            }
            else
            {
                // エミッタ時間アニメなら右端は65535フレームでクランプ
                const float MaxF = (float)Constants.EmitterAnimMax / (float)Constants.ParticleAnimMax;
                if (nodePos.X > (MaxF * Constants.ViewportFullWidth) + Constants.ViewportOriginX)
                {
                    nodePos.X = (MaxF * Constants.ViewportFullWidth) + Constants.ViewportOriginX;
                }
            }

            // 値軸上限クランプ
            if (this.editor.CalcValueFromPos(nodePos.Y) > this.editor.MaxLimit)
            {
                nodePos.Y = this.editor.CalcPosFromValue(this.editor.MaxLimit);
            }

            // 値軸下限クランプ
            if (this.editor.CalcValueFromPos(nodePos.Y) < this.editor.MinLimit)
            {
                nodePos.Y = this.editor.CalcPosFromValue(this.editor.MinLimit);
            }

            return nodePos;
        }
    }
}
