﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.CodeDom;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Specifics.CurveEditor.RenderingElements;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// カーブエディタのメインコントロールです。
    /// </summary>
    public partial class CurveEditor : UIUserControl
    {
        /// <summary>
        /// 編集モードです。
        /// </summary>
        public enum EditorModes
        {
            /// <summary>
            /// パーティクル時間アニメーション
            /// </summary>
            ParticleTimeAnim,

            /// <summary>
            /// エミッタ時間アニメーション
            /// </summary>
            EmitterTimeAnim,
        }

        #region バッキングフィールド

        /// <summary>
        /// ループモード
        /// </summary>
        private int loopMode;

        /// <summary>
        /// ループ終端フレーム
        /// </summary>
        private int loopEnd;

        /// <summary>
        /// バインディング用アニメーションテーブル
        /// </summary>
        private AnimationTableData values = new AnimationTableData();

        /// <summary>
        /// 開始位置ランダム
        /// </summary>
        private int randomize;

        /// <summary>
        /// 補間タイプ
        /// </summary>
        private int interMode;

        /// <summary>
        /// チェックボックスリスト
        /// </summary>
        private readonly List<UICheckBox> checkboxList;

        /// <summary>
        /// チャンネルリスト
        /// </summary>
        private List<CurveChannel> channels;

        /// <summary>
        /// 他のキーに値をスナップボタンの状態
        /// </summary>
        private bool otherKeySnapping = true;

        /// <summary>
        /// デフォルトの0固定モード
        /// </summary>
        private int defaultZeroPin = 3;

        /// <summary>
        /// ループモードの表示/非表示を管理するフラグ
        /// </summary>
        private bool disableLoopMode = false;

        /// <summary>
        /// ランダムスタートのUIの表示/非表示を管理するフラグ
        /// </summary>
        private bool disableRandomStart = false;

        /// <summary>
        /// エディターモード.
        /// </summary>
        private EditorModes editorMode = EditorModes.ParticleTimeAnim;

        #endregion

        #region 選択状態フィールド

        /// <summary>
        /// 選択候補になりうるノードリスト(マウスオーバーによるピック)
        /// </summary>
        private readonly List<RectangleShape> selectedNodes = new List<RectangleShape>();

        /// <summary>
        /// マウスオーバーしているノードに重ねる円形状
        /// </summary>
        private readonly CircleShape selectedMarker;

        /// <summary>
        /// テキストボックスでの編集対象になるノード(nullは未選択を表す)
        /// </summary>
        private RectangleShape targetNode = null;

        /// <summary>
        /// 選択しているノードに重ねるマーカー画像
        /// </summary>
        private readonly ImageRenderable targetMarker;

        #endregion

        #region 差分取得用フィールド

        /// <summary>
        /// 履歴スタック用差分テーブル
        /// </summary>
        private readonly AnimationTableData oldValue = new AnimationTableData();

        /// <summary>
        /// ループ終端フレームの前回値
        /// </summary>
        private int prevEnd;

        /// <summary>
        /// 0固定モードの前回値
        /// </summary>
        private int prevZeroPin;

        /// <summary>
        /// フレーム数ボックス入力値
        /// </summary>
        private string textFrameBackup;

        /// <summary>
        /// 値ボックス入力値
        /// </summary>
        private string textValueBackup;

        /// <summary>
        /// ループ終端ボックス入力値
        /// </summary>
        private string textLoopEndBackup;

        #endregion

        #region 状態フィールド

        /// <summary>
        /// カーソルがエディタ内にあるかどうか
        /// </summary>
        private bool entering = false;

        /// <summary>
        /// データ更新中の逆方向同期を抑制
        /// </summary>
        private bool updating = false;

        /// <summary>
        /// テキスト入力時のスナップ無効化フラグ
        /// </summary>
        private bool noSnapping = false;

        /// <summary>
        /// ドラッグ中フラグ
        /// </summary>
        private bool dragging = false;

        /// <summary>
        /// マージンを超えてドラッグが本当に開始されたかフラグ
        /// </summary>
        private bool dragStarted = false;

        /// <summary>
        /// 軸ロックの方向.
        /// </summary>
        private int lockDirection = 0;

        /// <summary>
        /// 拡縮スケール
        /// </summary>
        private PointF scale = new PointF(1.0f, 1.0f);

        #endregion

        #region 複合オブジェクト

        /// <summary>
        /// 複数の重なったビューポートを管理するオブジェクト
        /// </summary>
        private readonly LayeredViewports viewports;

        /// <summary>
        /// 複数チャンネル分のカーブとその操作を管理するオブジェクト
        /// </summary>
        private readonly CurveManupirator manipurator;

        /// <summary>
        /// 座標系に関する要素を描画するオブジェクト
        /// </summary>
        private readonly CoordinateSystem coordinateSystem = new CoordinateSystem();

        /// <summary>
        /// マウスカーソルに追従する要素を描画するオブジェクト
        /// </summary>
        private readonly VisualCursor visualCursor = new VisualCursor();

        /// <summary>
        /// フレーム数ポップアップを描画するオブジェクト
        /// </summary>
        private readonly FramePopup framePopup = new FramePopup();

        /// <summary>
        /// 無効領域を描画するオブジェクト
        /// </summary>
        private readonly InvalidArea invalidArea = new InvalidArea();

        /// <summary>
        /// 警告ツールチップ
        /// </summary>
        private readonly WarningToolTip warningToolTip = new WarningToolTip();

        /// <summary>
        /// ツールチップ拡張
        /// </summary>
        private ExtendedToolTip extendedToolTip = null;

        #endregion

        #region コンストラクタ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CurveEditor()
        {
            this.InitializeComponent();
            this.checkboxList = new List<UICheckBox>
            {
                this.checkbox_X,
                this.checkbox_Y,
                this.checkbox_Z,
                this.checkbox_W
            };
            this.checkbox_W.Visibility = Visibility.Hidden;
            foreach (var t in this.checkboxList)
            {
                t.Checked = true;
            }

            this.MinLimit = -1000000.0f;
            this.MaxLimit = 1000000.0f;
            this.NormalizeAt = 1.0f;
            this.IsBlockingToggleProperty = false;
            this.DoubleBuffering = true;
            this.EditorMode = EditorModes.ParticleTimeAnim;

            this.viewports = new LayeredViewports(this);
            this.manipurator = new CurveManupirator(this);
            this.selectedMarker = new CircleShape
            {
                Radius = Constants.NodeSize,
                FillColor = Color.Transparent,
                BorderColor = Color.BlueViolet
            };
            this.targetMarker = new ImageRenderable
            {
                Image = Properties.Resources.CurveEditor_selection,
                HAlignment = HAlignment.Center,
                VAlignment = VAlignment.Center,
                Padding = new Padding(1, 1, 0, 0)
            };

            this.invalidArea.Show(this.viewports);
            this.ResizeViewport();

            this.UnderlayCurveDrawer = new UnderlayCurveDrawer(
                this.viewports,
                this.CalcPosFromTime,
                this.CalcPosFromValue,
                () =>
                {
                    this.Rescale();
                    this.UpdateTarget();
                    this.RequestDraw();
                });

            // 補間タイプはビューアがリニアにしか対応していないため UI を非表示にする
            // 今後復帰する可能性がある…かも？
            this.label_InterMode.Visibility = Visibility.Collapsed;
            this.combobox_InterMode.Visibility = Visibility.Collapsed;

            this.AdditionalInitialize();
        }

        #endregion

        #region 公開プロパティ

        /// <summary>
        /// trueにすると、ループモードのUIを非表示にします.
        /// </summary>
        public bool DisableLoopMode
        {
            get
            {
                return this.disableLoopMode;
            }

            set
            {
                this.disableLoopMode = value;
                if (value == true)
                {
                    this.button_LoopMode.Enabled = false;
                    this.button_LoopMode.Visible = false;
                    this.text_LoopEnd.Enabled = false;
                    this.text_LoopEnd.Visible = false;
                }
                else
                {
                    this.button_LoopMode.Enabled = true;
                    this.button_LoopMode.Visible = true;
                    this.text_LoopEnd.Enabled = true;
                    this.text_LoopEnd.Visible = true;
                }
            }
        }

        /// <summary>
        /// trueにすると、ランダムスタートのUIを非表示にします.
        /// </summary>
        public bool DisableRandomStart
        {
            get
            {
                return this.disableRandomStart;
            }

            set
            {
                this.disableRandomStart = value;
                if (value == true)
                {
                    this.checkbox_Random.Enabled = false;
                    this.checkbox_Random.Visible = false;
                    this.label_SepLoopRandom.Enabled = false;
                    this.label_SepLoopRandom.Visible = false;
                }
                else
                {
                    this.checkbox_Random.Enabled = true;
                    this.checkbox_Random.Visible = true;
                    this.label_SepLoopRandom.Enabled = true;
                    this.label_SepLoopRandom.Visible = true;
                }
            }
        }

        /// <summary>
        /// 0ならパーティクル時間アニメモード、1ならエミッタ時間アニメモード.
        /// </summary>
        public EditorModes EditorMode
        {
            get
            {
                return this.editorMode;
            }

            set
            {
                this.editorMode = value;
                if (this.editorMode == EditorModes.ParticleTimeAnim)
                {
                    this.text_LoopEnd.Visibility = Visibility.Visible;
                    this.label_SepLoopRandom.Visibility = Visibility.Visible;
                    this.checkbox_Random.Visibility = Visibility.Visible;
                    // this.label_InterMode.Visibility = Visibility.Collapsed;
                    // this.combobox_InterMode.Visibility = Visibility.Collapsed;

                    this.combobox_SnapFrame.Items.Clear();
                    this.combobox_SnapFrame.Items.Add("1%");
                    this.combobox_SnapFrame.Items.Add("2%");
                    this.combobox_SnapFrame.Items.Add("5%");
                    this.combobox_SnapFrame.Items.Add("10%");
                    this.combobox_SnapFrame.SelectedIndex = 0;
                }
                else if (this.editorMode == EditorModes.EmitterTimeAnim)
                {
                    this.text_LoopEnd.Visibility = Visibility.Collapsed;
                    this.label_SepLoopRandom.Visibility = Visibility.Collapsed;
                    this.checkbox_Random.Visibility = Visibility.Collapsed;
                    // this.label_InterMode.Visibility = Visibility.Visible;
                    // this.combobox_InterMode.Visibility = Visibility.Visible;

                    this.combobox_SnapFrame.Items.Clear();
                    this.combobox_SnapFrame.Items.Add("1");
                    this.combobox_SnapFrame.Items.Add("2");
                    this.combobox_SnapFrame.Items.Add("5");
                    this.combobox_SnapFrame.Items.Add("10");
                    this.combobox_SnapFrame.SelectedIndex = 0;
                }
            }
        }

        /// <summary>
        /// 整数値のみならtrue,そうじゃなかったらfalse.
        /// </summary>
        public bool IntegerOnly { get; set; }

        /// <summary>
        /// 値変更時に実行するExecutableを設定・取得します。
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// リセット処理を行うExecutableを設定・取得します.
        /// </summary>
        public IExecutable ResetExecutable { get; set; }

        /// <summary>
        /// パーティクルの寿命を取得するExectuableを設定・取得します。
        /// </summary>
        public IExecutable GetParticleLifeExecutable { get; set; }

        /// <summary>
        /// アニメーションデータにバインドされたプロパティ名.
        /// </summary>
        public string ValuesTargetPropertyName { get; set; }

        /// <summary>
        /// 再生モードボタンにバインドされたプロパティ名.
        /// </summary>
        public string LoopModeTargetPropertyName { get; set; }

        /// <summary>
        /// ループフレーム数にバインドされたプロパティ名.
        /// </summary>
        public string LoopEndTargetPropertyName { get; set; }

        /// <summary>
        /// 開始位置ランダムチェックボックスにバインドされたプロパティ名.
        /// </summary>
        public string RandomizeTargetPropertyName { get; set; }

        /// <summary>
        /// 補間タイプチェックボックスにバインドされたプロパティ名.
        /// </summary>
        public string InterModeTargetPropertyName { get; set; }

        /// <summary>
        /// X軸のチェックボックス情報
        /// </summary>
        public bool EnableX
        {
            get { return this.checkbox_X.Checked; }
            set { this.checkbox_X.Checked = value; }
        }

        /// <summary>
        /// Y軸のチェックボックス情報
        /// </summary>
        public bool EnableY
        {
            get { return this.checkbox_Y.Checked; }
            set { this.checkbox_Y.Checked = value; }
        }

        /// <summary>
        /// Z軸のチェックボックス情報
        /// </summary>
        public bool EnableZ
        {
            get { return this.checkbox_Z.Checked; }
            set { this.checkbox_Z.Checked = value; }
        }

        /// <summary>
        /// W軸のチェックボックス情報
        /// </summary>
        public bool EnableW
        {
            get { return this.checkbox_W.Checked; }
            set { this.checkbox_W.Checked = value; }
        }

        /// <summary>
        /// 他キースナップモード情報
        /// </summary>
        public bool EnableOtherKeySnap
        {
            get
            {
                return this.otherKeySnapping;
            }

            set
            {
                this.otherKeySnapping = value;
                this.button_SnapKeyMode.BackgroundImage = this.otherKeySnapping ?
                    Properties.Resources.Icon_Curve__KeySnap_On :
                    Properties.Resources.Icon_Curve__KeySnap_Off;
            }
        }

        /// <summary>
        /// 時間スナップ情報
        /// </summary>
        public int SnapTimeLevel
        {
            get { return this.combobox_SnapFrame.SelectedIndex; }
            set { this.combobox_SnapFrame.SelectedIndex = value; }
        }

        /// <summary>
        /// 値スナップ情報
        /// </summary>
        public int SnapValueLevel
        {
            get { return this.combobox_SnapValue.SelectedIndex; }
            set { this.combobox_SnapValue.SelectedIndex = value; }
        }

        /// <summary>
        /// 0固定モード
        /// -1は非表示、-2の場合はDefaultZeroPinの設定を適用する
        /// </summary>
        public int ZeroPinMode
        {
            get
            {
                return this.combobox_ZeroPin.SelectedIndex;
            }

            set
            {
                if (value == -1 || (value == -2 && this.defaultZeroPin == -1))
                {
                    this.combobox_ZeroPin.SelectedIndex = -1;
                }
                else
                {
                    this.combobox_ZeroPin.SelectedIndex =
                        value == -2 ? this.DefaultZeroPin : value;
                }
            }
        }

        /// <summary>
        /// XAMLからの0固定モード指定用(-1でコンボボックス自体を非表示)
        /// </summary>
        public int DefaultZeroPin
        {
            private get
            {
                return this.defaultZeroPin;
            }

            set
            {
                if (value < -1 || value > 3)
                {
                    return;
                }

                this.defaultZeroPin = value;
                if (value == -1)
                {
                    // 0固定モードのUIを非表示にする
                    this.label_SepZeroSnap.Visibility = Visibility.Hidden;
                    this.combobox_ZeroPin.Visibility = Visibility.Hidden;
                    this.panel_ZeroPin.Visibility = Visibility.Hidden;

                    this.ZeroPinMode = -1;
                }
                else
                {
                    // 0固定モードのUIを表示する
                    this.label_SepZeroSnap.Visibility = Visibility.Visible;
                    this.combobox_ZeroPin.Visibility = Visibility.Visible;
                    this.panel_ZeroPin.Visibility = Visibility.Visible;

                    if (this.ZeroPinMode == -1)
                    {
                        this.ZeroPinMode = this.DefaultZeroPin;
                    }
                }
            }
        }

        /// <summary>
        /// アニメーションテーブル
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public AnimationTableData Values
        {
            get
            {
                return this.values;
            }

            set
            {
                if (value == null || this.values.Equals(value))
                {
                    return;
                }

                var clonedValues = (AnimationTableData)value.Clone();
                this.LogicalTreeElementExtender.SetValue(ref this.values, clonedValues);
                this.SyncView();
                if (!this.updating)
                {
                    this.oldValue.Set(this.values);
                }
            }
        }

        /// <summary>
        /// ループモード
        /// </summary>
        public int LoopMode
        {
            get
            {
                return this.loopMode;
            }

            set
            {
                var oldLoopMode = this.loopMode;
                this.loopMode = value;
                if (this.loopMode == 0)
                {
                    this.button_LoopMode.BackgroundImage =
                        Properties.Resources.CurveEditor_Icon_AnimTable_Loop_Off;
                    this.text_LoopEnd.Enabled = false;
                    if (this.EditorMode == EditorModes.ParticleTimeAnim)
                    {
                        this.checkbox_Random.Visibility = Visibility.Collapsed;
                        this.label_SepLoopRandom.Visibility = Visibility.Collapsed;
                    }
                }
                else
                {
                    this.button_LoopMode.BackgroundImage =
                        Properties.Resources.CurveEditor_Icon_AnimTable_Loop_On;
                    this.text_LoopEnd.Enabled = true;
                    if (this.EditorMode == EditorModes.ParticleTimeAnim)
                    {
                        this.checkbox_Random.Visibility = Visibility.Visible;
                        this.label_SepLoopRandom.Visibility = Visibility.Visible;
                    }
                }

                if (this.IsBlockingToggleProperty || this.IsUpdatingDataContext)
                {
                    return;
                }

                this.LogicalTreeElementExtender.NotifyPropertyChanged();
                this.OnValueChanged(new ValueChangedExEventArgs(
                    oldLoopMode,
                    value,
                    false,
                    this.LoopModeTargetPropertyName));
            }
        }

        /// <summary>
        /// ループ終端フレーム
        /// </summary>
        public int LoopEnd
        {
            get
            {
                return this.loopEnd;
            }

            set
            {
                this.loopEnd = value;
                this.text_LoopEnd.Text = value.ToString();
                if (this.IsBlockingToggleProperty || this.IsUpdatingDataContext)
                {
                }
                else if (!this.noSnapping)
                {
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.Validation);
                    this.OnValueChanged(new ValueChangedExEventArgs(
                        this.prevEnd,
                        this.loopEnd,
                        false,
                        this.LoopEndTargetPropertyName));
                }

                this.prevEnd = this.loopEnd;
            }
        }

        /// <summary>
        /// 開始位置ランダム
        /// </summary>
        public int Randomize
        {
            get
            {
                return this.randomize;
            }

            set
            {
                var oldRandomize = this.randomize;
                this.randomize = value;
                this.checkbox_Random.Checked = value == 1;
                if (this.IsBlockingToggleProperty || this.IsUpdatingDataContext)
                {
                    return;
                }

                this.LogicalTreeElementExtender.NotifyPropertyChanged();
                this.OnValueChanged(new ValueChangedExEventArgs(
                    oldRandomize,
                    value,
                    false,
                    this.RandomizeTargetPropertyName));
            }
        }

        /// <summary>
        /// 補間タイプ
        /// </summary>
        public int InterMode
        {
            get
            {
                return this.interMode;
            }

            set
            {
                var oldInterMode = this.interMode;
                this.interMode = value;
                this.combobox_InterMode.SelectedIndex = value;
                if (this.IsBlockingToggleProperty || this.IsUpdatingDataContext)
                {
                    return;
                }

                this.LogicalTreeElementExtender.NotifyPropertyChanged();
                this.OnValueChanged(new ValueChangedExEventArgs(
                    oldInterMode,
                    value,
                    false,
                    this.InterModeTargetPropertyName));
            }
        }

        /// <summary>
        /// チャンネルリスト
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public List<CurveChannel> Channels
        {
            get { return this.channels ?? (this.channels = new List<CurveChannel>()); }
            set { this.channels = value; }
        }

        /// <summary>
        /// 最大値
        /// </summary>
        public float MaxLimit { get; set; }

        /// <summary>
        /// 最小値
        /// </summary>
        public float MinLimit { get; set; }

        /// <summary>
        /// 基本スケールで1.0の値をいくつとして扱うかの値
        /// </summary>
        public float NormalizeAt { get; set; }

        /// <summary>
        /// キーの初期値
        /// </summary>
        public float DefaultValue { get; set; }

        /// <summary>
        /// 値ラベルで表示する小数点以下桁数
        /// </summary>
        public int LabelDigit { get; set; }

        /// <summary>
        /// 値ラベルの先頭に付加する文字列
        /// </summary>
        public string LabelPrefix { get; set; }

        /// <summary>
        /// ビューポートの調整処理をトリガーします。
        /// </summary>
        public bool TriggerAdjustViewport
        {
            get
            {
                return false;
            }

            set
            {
                if (value)
                {
                    this.AfterAdaptPreset();
                }
            }
        }

        /// <summary>
        /// エミッタ時間アニメの切り換え時に再生タイプと補間タイプが上書きされるのを抑制
        /// </summary>
        public bool IsBlockingToggleProperty { get; set; }

        /// <summary>
        /// エミッタ時間アニメ用の非アクティブカーブ描画管理オブジェクトを取得します。
        /// </summary>
        public UnderlayCurveDrawer UnderlayCurveDrawer { get; private set; }

        #endregion
    }
}
