﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Specifics.Sliders;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Control class for the alpha slider panel.
    /// </summary>
    [DefaultEvent("ValueChanged")]
    public partial class SliderPanelAlpha : UserControl
    {
        /// <summary>
        /// カラープレビューの矩形の横幅。
        /// </summary>
        private const int ColorPreviewRectWidth = 36;

        /// <summary>
        /// カラープレビューの矩形の縦幅。
        /// </summary>
        private const int ColorPreviewRectHeight = 16;

        /// <summary>
        /// コンテキストメニュー。
        /// </summary>
        private UIContextMenuStrip contextMenu = new UIContextMenuStrip();

        /// <summary>
        /// イベント抑制深さ
        /// </summary>
        private int suppressEventDepth;

#if false // TODO:コード復帰
        /// private UIToolStripMenuItem m_cmiCopy = null;
        /// private UIToolStripMenuItem m_cmiPaste = null;
#endif

        /// <summary>
        /// ColorSliderControllerAlpha.
        /// </summary>
        private ColorSliderControllerAlpha controllerAlpha = new ColorSliderControllerAlpha();

        /// <summary>
        /// isMouseOverSwatch.
        /// </summary>
        private bool isMouseOverSwatch = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public SliderPanelAlpha()
        {
            this.DoubleBuffered = true;

            this.InitializeComponent();

            this.CreateContextMenu();

            this.controllerAlpha.SetColorElementSliders(this.sliderElement1);

            this.controllerAlpha.ValueChanged += this.OnControllerValueChanged;
            this.textElement1.ValueChanged += this.OnDialSliderValueChanged;
        }

        /// <summary>
        /// Event triggered when the hue value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// Get or set the color value.
        /// </summary>
        public ColorRgba Value
        {
            get
            {
                return this.controllerAlpha.Value;
            }

            set
            {
#if false
                if (this.Enabled == false)
                {
                    return;
                }
#endif

                this.BeginSuppressEvent();

                this.textElement1.Value  = value.A;
                this.controllerAlpha.Value = value;

                this.Invalidate();

                this.EndSuppressEvent();
            }
        }

        /// <summary>
        /// イベントを抑制するかどうか。
        /// </summary>
        private bool IsSuppressEvent
        {
            get { return this.suppressEventDepth > 0; }
        }

        /// <summary>
        /// イベント抑制開始
        /// </summary>
        public void BeginSuppressEvent()
        {
            ++this.suppressEventDepth;
        }

        /// <summary>
        /// イベント抑制終了
        /// </summary>
        public void EndSuppressEvent()
        {
            --this.suppressEventDepth;
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            var rect = new Rectangle(
                0,
                (this.DisplayRectangle.Height - ColorPreviewRectHeight) / 2,
                ColorPreviewRectWidth,
                ColorPreviewRectHeight);

            // Check if the mouse cursor is over the color swatch.
            this.isMouseOverSwatch = rect.Contains(e.Location);

            base.OnMouseMove(e);
        }

        /// <summary>
        /// スライダーテキストが変更されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnDialSliderValueChanged(object sender, EventArgs e)
        {
            if (this.IsSuppressEvent)
            {
                return;
            }

            this.BeginSuppressEvent();

            this.sliderElement1.Value = this.textElement1.Value;

            this.Invalidate();

            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, new SequentialValueChangedEventArgs(false));
            }

            this.EndSuppressEvent();
        }

        /// <summary>
        /// Handle ValueChanged event from the controller.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnControllerValueChanged(
                     object sender,
                     SequentialValueChangedEventArgs e)
        {
            if (this.IsSuppressEvent)
            {
                return;
            }

            this.BeginSuppressEvent();

            if (sender == this.textElement1)
            {
                this.sliderElement1.Value = this.textElement1.Value;
            }
            else
            {
                this.textElement1.Value = this.sliderElement1.Value;
            }

            this.Invalidate();

            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, e);
            }

            this.EndSuppressEvent();
        }

        /// <summary>
        /// Handle Opening event from the color swatch context menu.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnContextMenuOpening(object sender, CancelEventArgs e)
        {
#if false // TODO:コード復帰
            e.Cancel = !(this.m_bMouseOverSwatch);

            bool isPasteEnabled = true;

            var data =
                Clipboard.GetData(UAColorValueEdit.CLIPBOARD_DATA_FORMAT) as UAColorValueEdit.ColorEditClipboardData;
            if (data == null)
            {
                isPasteEnabled = false;
            }
            else if (data.HasColorA == false)
            {
                isPasteEnabled = false;
            }

            this.m_cmiPaste.Enabled = isPasteEnabled;
#endif
        }

        /// <summary>
        /// Helper method for creating the context menu for the color swatch.
        /// </summary>
        private void CreateContextMenu()
        {
#if false // TODO:コード復帰
            // "Copy" menu item.
            this.m_cmiCopy = new UIToolStripMenuItem()
            {
                Text = res.Strings.COPY,
                Image = Properties.Resources.TB_CopyHS
            };

            // Copy handler.
            this.m_cmiCopy.Click += (s, e) =>
            {
                ColorRgba color = this.Value;

                var data = new UAColorValueEdit.ColorEditClipboardData();

                data.HasColorR = false;
                data.HasColorG = false;
                data.HasColorB = false;
                data.HasColorA = true;
                data.ColorR = 1.0f;
                data.ColorG = 1.0f;
                data.ColorB = 1.0f;
                data.ColorA = this.Value.A;

                Clipboard.SetData(UAColorValueEdit.CLIPBOARD_DATA_FORMAT, data);
            };

            this.m_contextMenu.Items.Add(this.m_cmiCopy);

            // "Paste" menu item.
            this.m_cmiPaste = new UIToolStripMenuItem()
            {
                Text = res.Strings.PASTE,
                Image = Properties.Resources.TB_PasteHS
            };

            // Paste handler.
            this.m_cmiPaste.Click += (s, e) =>
            {
                var data =
                    Clipboard.GetData(UAColorValueEdit.CLIPBOARD_DATA_FORMAT) as UAColorValueEdit.ColorEditClipboardData;
                if (data == null)
                {
                    return;
                }

                if (data.HasColorA == false)
                {
                    return;
                }

                this.Value = new ColorRgba(1.0f, 1.0f, 1.0f, data.ColorA);

                if (this.ValueChanged != null)
                {
                    this.ValueChanged(this, new SequentialValueChangedEventArgs(false));
                }
            };

            this.m_contextMenu.Items.Add(this.m_cmiPaste);

            this.m_contextMenu.Opening += this.OnContextMenuOpening;

            this.ContextMenuStrip = this.m_contextMenu;
#endif
        }
    }
}
