﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Class for alpha color element slider controller.
    /// </summary>
    public class ColorSliderControllerAlpha
    {
        /// <summary>
        /// スライダ.
        /// </summary>
        private ColorElementSlider sliderA = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ColorSliderControllerAlpha()
        {
        }

        /// <summary>
        /// Event triggered when the color value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// Get or set the color value.
        /// </summary>
        public ColorRgba Value
        {
            get
            {
                if (this.sliderA == null)
                {
                    return new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);
                }

                return new ColorRgba(1.0f, 1.0f, 1.0f, this.sliderA.Value);
            }

            set
            {
                if (this.sliderA == null)
                {
                    return;
                }

                this.sliderA.Value = value.A;
            }
        }

        /// <summary>
        /// Set color element sliders.
        /// </summary>
        /// <param name="sliderA">Alpha color slider.</param>
        public void SetColorElementSliders(ColorElementSlider sliderA)
        {
            this.sliderA = sliderA;

            this.sliderA.IgnoreGammaCorrection = true;

            this.sliderA.RequestUpdateBitmap += this.OnSliderRequestUpdateBitmap;
            this.sliderA.ValueChanged += this.OnSliderValueChanged;

            this.UpdateSliderAImage();
        }

        /// <summary>
        /// Clear the controlling color element sliders.
        /// </summary>
        public void ClearColorElementSliders()
        {
            if (this.sliderA != null)
            {
                this.sliderA.RequestUpdateBitmap -= this.OnSliderRequestUpdateBitmap;
                this.sliderA.ValueChanged -= this.OnSliderValueChanged;

                this.sliderA = null;
            }
        }

        /// <summary>
        /// Update the alpha color slider background image.
        /// </summary>
        private void UpdateSliderAImage()
        {
            if (this.sliderA == null)
            {
                return;
            }

            Bitmap img = this.sliderA.ColorBarBitmap;
            if (img == null)
            {
                return;
            }

            if (img.Width <= 0)
            {
                return;
            }

            using (var g = Graphics.FromImage(img))
            using (var brush = new LinearGradientBrush(
                                                   g.VisibleClipBounds,
                                                   Color.Black,
                                                   Color.White,
                                                   0.0f,
                                                   false))
            {
                g.FillRectangle(brush, g.VisibleClipBounds);
            }

            this.sliderA.Invalidate();
        }

        /// <summary>
        /// Handle RequestUpdateBitmap event for the sliders.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSliderRequestUpdateBitmap(object sender, EventArgs e)
        {
            if (sender == this.sliderA)
            {
                this.UpdateSliderAImage();
            }
        }

        /// <summary>
        /// Handle ValueChanged event for the sliders.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSliderValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // Fire the event.
            if (sender != null && sender == this.sliderA && this.ValueChanged != null)
            {
                this.ValueChanged(this, e);
            }
        }
    }
}
