﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Slider control class for color element sliders.
    /// </summary>
    public class ColorElementSlider : UIUserControl, ISupportInitialize
    {
        /// <summary>
        /// ハンドルの横幅。
        /// </summary>
        private const int HandleWidth = 11;

        /// <summary>
        /// ハンドルの縦幅。
        /// </summary>
        private const int HandleHeight = 5;

        /// <summary>
        /// どのくらい時間が経過したらエベントを送るかの値。
        /// </summary>
        private const double ValueChangeEventInternal = 1.0f;       // 100.0f

        /// <summary>
        /// 最後に値の変更が起きた時刻。
        /// </summary>
        private DateTime lastValueChangeEventTime = DateTime.MinValue;

        /// <summary>
        /// 初期化中かどうか。
        /// </summary>
        private bool isInitializing = false;

        /// <summary>
        /// カラーバービットマップ。
        /// </summary>
        private Bitmap imgColorBar = null;

        /// <summary>
        /// スライダの値。
        /// </summary>
        private float sliderValue = 1.0f;

        /// <summary>
        /// 色を示す矩形。
        /// </summary>
        private Rectangle rectColorBar = new Rectangle();

        /// <summary>
        /// GammaEnabled プロパティのバッキングフィールドです。
        /// </summary>
        private bool gammaEnabled;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ColorElementSlider()
        {
            this.DoubleBuffered        = true;
            this.TabStop               = false;
            this.IgnoreGammaCorrection = false;

            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;
        }

        /// <summary>
        /// Event triggered when the hue value is changed.
        /// </summary>
#pragma warning disable 67
        public event SequentialValueChangedEventHandler ValueChanged = null;
#pragma warning restore 67

        /// <summary>
        /// Event triggered when the background bitmap needs to be updated.
        /// </summary>
        public event EventHandler RequestUpdateBitmap = null;

        /// <summary>
        /// Get or set the value.
        /// </summary>
        public float Value
        {
            get
            {
                return this.sliderValue;
            }

            set
            {
                this.sliderValue = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get the bitmap for the color bar.
        /// </summary>
        public Bitmap ColorBarBitmap
        {
            get { return this.imgColorBar; }
        }

        /// <summary>
        /// ガンマ補正を有効にするかどうか取得または設定します。
        /// </summary>
        public bool GammaEnabled
        {
            get
            {
                return this.gammaEnabled;
            }

            set
            {
                if (value != this.gammaEnabled)
                {
                    this.gammaEnabled = value;

                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// ガンマ補正を強制的にオフにするかどうか取得または設定します。
        /// アルファ値のスライダーはこの値を true にします。
        /// </summary>
        public bool IgnoreGammaCorrection { get; set; }

        /// <summary>
        /// Begin initialization.
        /// </summary>
        public void BeginInit()
        {
            this.isInitializing = true;
        }

        /// <summary>
        /// End initialization.
        /// </summary>
        public void EndInit()
        {
            this.isInitializing = false;

            // First update the bitmap size.
            this.UpdateColorBarBitmapSize();

            // Render the bitmap.
            if (this.RequestUpdateBitmap != null)
            {
                this.RequestUpdateBitmap(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Clean up the used resources.
        /// </summary>
        /// <param name="disposing">True on disposing managed resources.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.imgColorBar != null)
                {
                    this.imgColorBar.Dispose();
                }

                OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Set value at the specified location.
        /// </summary>
        /// <param name="x">The location x.</param>
        /// <param name="y">The location y.</param>
        /// <param name="isChanging">変更中か</param>
        protected void SetValueAt(int x, int y, bool isChanging)
        {
            int location = x - this.rectColorBar.X;

            float value;
            if (location < 0)
            {
                value = 0.0f;
            }
            else if (location > (this.rectColorBar.Width - 1))
            {
                value = 1.0f;
            }
            else
            {
                value = (float)location / (float)(this.rectColorBar.Width - 1);
            }

            if (value != this.sliderValue)
            {
                this.sliderValue = value;
                this.Invalidate();
            }

            // Notify value changed.
            if (true)
            {
                double lastEventTimeSpan =
                    (DateTime.Now - this.lastValueChangeEventTime).TotalMilliseconds;

                if (isChanging == false ||
                     lastEventTimeSpan >= ValueChangeEventInternal)
                {
                    this.lastValueChangeEventTime = DateTime.Now;

                    if (this.ValueChanged != null)
                    {
                        this.ValueChanged(this, new SequentialValueChangedEventArgs(isChanging));
                    }

                    // マウスアップをロストすることがあるため無効化
                    ////Application.DoEvents();
                }
            }
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            Graphics g = e.Graphics;

            int handleHalfWidth = HandleWidth / 2;

            Rectangle rectColorBarOutline = new Rectangle(
                                                this.rectColorBar.X - 1,
                                                this.rectColorBar.Y - 1,
                                                this.rectColorBar.Width + 1,
                                                this.rectColorBar.Height + 1);

            if (this.imgColorBar != null && this.Enabled == true)
            {
                ImageAttributes attr = new ImageAttributes();

                // Render the background color.
                if (this.IgnoreGammaCorrection == false && this.GammaEnabled && ColorUtility.Gamma != 1.0)
                {
                    attr.SetGamma((float)(1.0 / ColorUtility.Gamma));
                }

                attr.SetWrapMode(WrapMode.Tile);

                g.DrawImage(
                  this.imgColorBar,
                  this.rectColorBar,
                  0.0f,
                  0.0f,
                  this.imgColorBar.Width,
                  this.imgColorBar.Height,
                  GraphicsUnit.Pixel,
                  attr);

                // Render the handle.
                int pos = (int)(Math.Max(Math.Min(this.sliderValue, 1.0f), 0.0f) * (float)(this.rectColorBar.Width - 1)) + this.rectColorBar.X;

                // Render the border.
                g.DrawRectangle(Pens.Black, rectColorBarOutline);

                // アイコンイメージを描画
                g.DrawImage(
                    Properties.Resources.Icon_ColorPicker_Cursor,
                    pos - handleHalfWidth - 2,
                    rectColorBarOutline.Bottom + 1);
            }
            else
            {
                // Render the gray-out color and border.
                g.FillRectangle(Brushes.LightGray, this.rectColorBar);
                g.DrawRectangle(Pens.DarkGray, rectColorBarOutline);
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X, e.Y, true);
            }

            base.OnMouseDown(e);
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X, e.Y, false);
            }

            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X, e.Y, true);
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);

            this.rectColorBar.X = (HandleWidth / 2) + 1;
            this.rectColorBar.Y = HandleHeight - 1;
            this.rectColorBar.Width = this.DisplayRectangle.Width - HandleWidth - 3;
            this.rectColorBar.Height = this.DisplayRectangle.Height - (HandleHeight * 2) - 5;

            // First update the bitmap size.
            this.UpdateColorBarBitmapSize();

            // Redraw the bitmap.
            if (this.RequestUpdateBitmap != null)
            {
                this.RequestUpdateBitmap(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Update the size of the slider color bar bitmap.
        /// </summary>
        private void UpdateColorBarBitmapSize()
        {
            if (this.isInitializing == true)
            {
                return;
            }

            int width = this.rectColorBar.Width;
            int height = 1;
            if (width <= 0)
            {
                return;
            }

            // Dispose the previous bitmap and create a new one if necessary.
            if (this.imgColorBar == null ||
                 this.imgColorBar.Width != width ||
                 this.imgColorBar.Height != height)
            {
                if (this.imgColorBar != null)
                {
                    this.imgColorBar.Dispose();
                }

                this.imgColorBar = new Bitmap(width, height);
            }
        }

        /// <summary>
        /// Handle GammaCorrectionChanged event from application.
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.Invalidate();
        }
    }
}
