﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Control class for adjusting the saturation and brightness value in HSB color space.
    /// </summary>
    [DefaultEvent("ValueChanged")]
    public class SaturationValueBox : UIUserControl, ISupportInitialize
    {
        /// <summary>
        /// 初期化フラグ.
        /// </summary>
        private bool initializing = false;

        /// <summary>
        /// 背景イメージ.
        /// </summary>
        private Bitmap imgBG = null;

        /// <summary>
        /// 色相.
        /// </summary>
        private float hueValue = 0.0f;

        /// <summary>
        /// 彩度.
        /// </summary>
        private float fvalueS = 0.0f;

        /// <summary>
        /// 明度.
        /// </summary>
        private float fvalueV = 1.0f;

        /// <summary>
        /// GammaEnabled プロパティのバッキングフィールドです。
        /// </summary>
        private bool gammaEnabled = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public SaturationValueBox()
        {
            this.DoubleBuffered = true;
            this.TabStop = false;
        }

        /// <summary>
        /// Event triggered when the saturation/brightness value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// Get or set the S element (saturation) in HSV color space.
        /// </summary>
        public float ValueS
        {
            get
            {
                return this.fvalueS;
            }

            set
            {
                this.fvalueS = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the V element (value) in HSV color space.
        /// </summary>
        public float ValueV
        {
            get
            {
                return this.fvalueV;
            }

            set
            {
                this.fvalueV = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// ガンマ補正を有効にするかどうか取得または設定します。
        /// </summary>
        public bool GammaEnabled
        {
            get
            {
                return this.gammaEnabled;
            }

            set
            {
                if (value != this.gammaEnabled)
                {
                    this.gammaEnabled = value;

                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// Begin initialization.
        /// </summary>
        public void BeginInit()
        {
            this.initializing = true;
        }

        /// <summary>
        /// End initialization.
        /// </summary>
        public void EndInit()
        {
            this.initializing = false;

            this.UpdateBackgroundBitmap();
        }

        /// <summary>
        /// Set hue value.
        /// </summary>
        /// <param name="hue">The value.</param>
        public void SetHue(float hue)
        {
            if (this.hueValue == hue)
            {
                return;
            }

            this.hueValue = hue;
            this.UpdateBackgroundBitmap();

            this.Invalidate();
        }

        /// <summary>
        /// Clean up the used resources.
        /// </summary>
        /// <param name="disposing">True on disposing managed resources.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.imgBG != null)
                {
                    this.imgBG.Dispose();
                }
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Set S (saturation) and V (value) of HSV color at the specified location.
        /// </summary>
        /// <param name="x">The location x.</param>
        /// <param name="y">The location y.</param>
        /// <param name="changing">
        /// Flag indicating whether the value is still being modified.
        /// </param>
        protected void SetValueAt(
            int x,
            int y,
            bool changing)
        {
            int posX = x - 1;
            int posY = y - 1;

            int width = this.DisplayRectangle.Width - 2;
            int height = this.DisplayRectangle.Height - 2;

            float valueX;
            if (posX < 0)
            {
                valueX = 0.0f;
            }
            else if (posX > height)
            {
                valueX = 1.0f;
            }
            else
            {
                valueX = (float)posX / (float)width;
            }

            float valueY;
            if (posY < 0)
            {
                valueY = 1.0f;
            }
            else if (posY > height)
            {
                valueY = 0.0f;
            }
            else
            {
                valueY = 1.0f - ((float)posY / (float)height);
            }

            if (valueX != this.fvalueS || valueY != this.fvalueV)
            {
                this.fvalueS = valueX;
                this.fvalueV = valueY;

                this.Invalidate();
            }

            // Notify value changed.
            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, new SequentialValueChangedEventArgs(changing));
            }
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            Graphics g = e.Graphics;

            Rectangle rectBGImg = new Rectangle(
                1, 1, this.DisplayRectangle.Width - 2, this.DisplayRectangle.Height - 2);
            Rectangle rectBorder = new Rectangle(
                0, 0, this.DisplayRectangle.Width - 1, this.DisplayRectangle.Height - 1);

            if (this.imgBG != null && this.Enabled == true)
            {
                // Render the background color.
                if (this.GammaEnabled && ColorUtility.Gamma != 1.0)
                {
                    var attr = new ImageAttributes();
                    attr.SetGamma((float)(1.0 / ColorUtility.Gamma));

                    g.DrawImage(
                        this.imgBG,
                        rectBGImg,
                        0.0f,
                        0.0f,
                        this.imgBG.Width,
                        this.imgBG.Height,
                        GraphicsUnit.Pixel,
                        attr);
                }
                else
                {
                    g.DrawImage(
                        this.imgBG,
                        rectBGImg,
                        0.0f,
                        0.0f,
                        this.imgBG.Width,
                        this.imgBG.Height,
                        GraphicsUnit.Pixel);
                }

                // Render the selection.
                int iposX = (int)(this.fvalueS * (float)(this.DisplayRectangle.Width - 3)) + 1;
                int iposY = (int)((1.0f - this.fvalueV) * (float)(this.DisplayRectangle.Height - 3)) + 1;

                // Inner black rectangle for selection.
                g.DrawRectangle(Pens.Black, iposX - 1, iposY - 1, 2, 2);

                // Middle white rectangle for selection.
                g.DrawRectangle(Pens.White, iposX - 2, iposY - 2, 4, 4);

                // Outer black rectangle for selection.
                g.DrawRectangle(Pens.Black, iposX - 3, iposY - 3, 6, 6);

                // Render the border.
                g.DrawRectangle(Pens.Black, rectBorder);
            }
            else
            {
                // Render the gray-out color and border.
                g.FillRectangle(Brushes.LightGray, rectBGImg);
                g.DrawRectangle(Pens.DarkGray, rectBorder);
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X, e.Y, true);
            }

            base.OnMouseDown(e);
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X, e.Y, false);
            }

            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X, e.Y, true);
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);

            this.UpdateBackgroundBitmap();
        }

        /// <summary>
        /// Update the slider background color bitmap.
        /// </summary>
        private void UpdateBackgroundBitmap()
        {
            if (this.initializing == true)
            {
                return;
            }

            int width = this.DisplayRectangle.Width - 2;
            int height = this.DisplayRectangle.Height - 2;

            if (width <= 0 || height <= 0)
            {
                return;
            }

            // Dispose the previous bitmap and create a new one if necessary.
            if (this.imgBG == null ||
                 this.imgBG.Width != width ||
                 this.imgBG.Height != height)
            {
                if (this.imgBG != null)
                {
                    this.imgBG.Dispose();
                }

                this.imgBG = new Bitmap(width, height);
            }

            {
                var rgb = ColorUtility.ToColorRgba(this.hueValue, 1.0f, 1.0f, 1.0f);
                var rect = new Rectangle(0, 0, width, height);

                using (var g = Graphics.FromImage(this.imgBG))
                using (var h = new LinearGradientBrush(rect, Color.White, rgb.ToWinColor(), LinearGradientMode.Horizontal))
                using (var v = new LinearGradientBrush(rect, Color.FromArgb(0, 0, 0, 0), Color.Black, LinearGradientMode.Vertical))
                {
                    g.FillRectangle(h, rect);
                    g.FillRectangle(v, rect);
                }
            }
        }
    }
}
