﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Slider control class for adjusting the hue value in HSB color space.
    /// </summary>
    [DefaultEvent("ValueChanged")]
    public class HueSlider : UIUserControl, ISupportInitialize
    {
        /// <summary>
        /// どのくらい時間が経過したらエベントを送るかの値。
        /// </summary>
        private const double ValueChangeEventInternal = 1.0f;         // 100.0f;

        /// <summary>
        /// 最後に値の変更が起きた時刻。
        /// </summary>
        private DateTime lastValueChangeEventTime = DateTime.MinValue;

        /// <summary>
        /// 初期化中かどうか。
        /// </summary>
        private bool initializing = false;

        /// <summary>
        /// 背景ビットマップ。
        /// </summary>
        private Bitmap imgBG = null;

        /// <summary>
        /// Hue の値。
        /// </summary>
        private float hueValue = 0.0f;

        /// <summary>
        /// マウスダウン時の座標.
        /// </summary>
        private Point downPos;

        /// <summary>
        /// GammaEnabled プロパティのバッキングフィールドです。
        /// </summary>
        private bool gammaEnabled = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public HueSlider()
        {
            this.DoubleBuffered = true;
            this.TabStop        = false;
        }

        /// <summary>
        /// Event triggered when the hue value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// Get or set the hue value.
        /// </summary>
        public float ValueH
        {
            get
            {
                return this.hueValue;
            }

            set
            {
                this.hueValue = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// ガンマ補正を有効にするかどうか取得または設定します。
        /// </summary>
        public bool GammaEnabled
        {
            get
            {
                return this.gammaEnabled;
            }

            set
            {
                if (value != this.gammaEnabled)
                {
                    this.gammaEnabled = value;

                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// Begin initialization.
        /// </summary>
        public void BeginInit()
        {
            this.initializing = true;
        }

        /// <summary>
        /// End initialization.
        /// </summary>
        public void EndInit()
        {
            this.initializing = false;

            this.UpdateBackgroundBitmap();
        }

        /// <summary>
        /// Clean up the used resources.
        /// </summary>
        /// <param name="disposing">True on disposing managed resources.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.imgBG != null)
                {
                    this.imgBG.Dispose();
                }
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Set hue at the specified location.
        /// </summary>
        /// <param name="x">The location x.</param>
        /// <param name="y">The location y.</param>
        /// <param name="bchanging">Flag indicating whether the value is still being modified.</param>
        protected void SetValueAt(int x, int y, bool bchanging)
        {
            int location = x - 1;
            int width = this.DisplayRectangle.Width - 11;

            float tempValue;
            if (location < 0)
            {
                tempValue = 0.0f;
            }
            else if (location > width)
            {
                tempValue = 1.0f;
            }
            else
            {
                tempValue = (float)location / (float)width;
            }

            if (tempValue != this.hueValue)
            {
                Func<float, bool> isZeroOrOne = v => (v == 0.0f) || (v == 1.0f);

                if (isZeroOrOne(this.hueValue) && isZeroOrOne(tempValue))
                {
                }
                else
                {
                    this.hueValue = tempValue;
                    this.Invalidate();
                }
            }

            // Notify value changed.
            if (this.ValueChanged != null)
            {
                double lastEventTimeSpan =
                    (DateTime.Now - this.lastValueChangeEventTime).TotalMilliseconds;

                if (bchanging == false ||
                     lastEventTimeSpan >= ValueChangeEventInternal)
                {
                    this.lastValueChangeEventTime = DateTime.Now;
                    this.ValueChanged(this, new SequentialValueChangedEventArgs(bchanging));
                }

                // マウスアップをロストすることがあるため無効化
                ////Application.DoEvents();
            }
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            Graphics g = e.Graphics;

            Rectangle rectBGImg = new Rectangle(
                                      6,
                                      1,
                                      this.DisplayRectangle.Width - 12,
                                      this.DisplayRectangle.Height - 12);
            Rectangle rectBorder = new Rectangle(
                                       5,
                                       0,
                                       this.DisplayRectangle.Width - 11,
                                       this.DisplayRectangle.Height - 11);

            if (this.imgBG != null && this.Enabled == true)
            {
                // Render the background color.
                if (this.GammaEnabled && ColorUtility.Gamma != 1.0)
                {
                    ImageAttributes attr = new ImageAttributes();
                    attr.SetGamma((float)(1.0 / ColorUtility.Gamma));

                    g.DrawImage(
                      this.imgBG,
                      rectBGImg,
                      0.0f,
                      0.0f,
                      this.imgBG.Width,
                      this.imgBG.Height,
                      GraphicsUnit.Pixel,
                      attr);
                }
                else
                {
                    g.DrawImage(
                      this.imgBG,
                      rectBGImg,
                      0.0f,
                      0.0f,
                      this.imgBG.Width,
                      this.imgBG.Height,
                      GraphicsUnit.Pixel);
                }

                // Render the handle.
                int pos = (int)(this.hueValue * (float)(this.DisplayRectangle.Width - 11)) + 1;

                // Render the border.
                g.DrawRectangle(Pens.Black, rectBorder);

                // Render the bar cursor.
                g.DrawImage(
                    Properties.Resources.Icon_ColorBar_Cursor,
                    pos,
                    this.DisplayRectangle.Height - 10);
            }
            else
            {
                // Render the gray-out color and border.
                g.FillRectangle(Brushes.LightGray, rectBGImg);
                g.DrawRectangle(Pens.DarkGray, rectBorder);
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// 色相バーのカーソルのサイズ分、通知する座標値をずらしています。
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X - 5, e.Y, true);
                this.downPos = e.Location;
            }

            base.OnMouseDown(e);
        }

        /// <summary>
        /// Handle MouseUp event.
        /// 色相バーのカーソルのサイズ分、通知する座標値をずらしています。
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.SetValueAt(e.X - 5, e.Y, false);
            }

            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle MouseMove event.
        /// 色相バーのカーソルのサイズ分、通知する座標値をずらしています。
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                if (Math.Abs(this.downPos.X - e.X) > 2 || Math.Abs(this.downPos.Y - e.Y) > 2)
                {
                    this.SetValueAt(e.X - 5, e.Y, true);
                    this.downPos.X = 65535;
                    this.downPos.Y = 65535;
                }
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);

            this.UpdateBackgroundBitmap();
        }

        /// <summary>
        /// Update the slider background color bitmap.
        /// </summary>
        private void UpdateBackgroundBitmap()
        {
            if (this.initializing == true)
            {
                return;
            }

            int width = this.DisplayRectangle.Width - 2;
            int height = this.DisplayRectangle.Width - 12;
            if (width <= 0 || height <= 0)
            {
                return;
            }

            // Dispose the previous bitmap and create a new one if necessary.
            if (this.imgBG == null ||
                 this.imgBG.Width != width ||
                 this.imgBG.Height != height)
            {
                if (this.imgBG != null)
                {
                    this.imgBG.Dispose();
                }

                this.imgBG = new Bitmap(width, height);
            }

            using (var g = Graphics.FromImage(this.imgBG))
            using (var brush = new LinearGradientBrush(
                                                   g.VisibleClipBounds,
                                                   Color.Blue,
                                                   Color.Red,
                                                   0.0f,
                                                   false))
            {
                Color[] colorArray =
                {
                                        Color.Red,
                                        Color.Yellow,
                                        Color.FromArgb(0, 255, 0),
                                        Color.Cyan,
                                        Color.Blue,
                                        Color.Magenta,
                                        Color.Red
                                     };
                float[] posArray =
                {
                                        0.0f,
                                        0.1687f,
                                        0.3314f,
                                        0.498f,
                                        0.6628f,
                                        0.8333f,
                                        1.0f
                                   };

                ColorBlend colorBlend = new ColorBlend();
                colorBlend.Colors = colorArray;
                colorBlend.Positions = posArray;

                brush.InterpolationColors = colorBlend;

                g.FillRectangle(brush, g.VisibleClipBounds);
            }
        }
    }
}
