﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using ColorClipboardData = EffectMaker.Foundation.ClipboardDataTypes.ColorClipboardData;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// カラーボックス
    /// </summary>
    public class ColorBox : UIPanel
    {
        /// <summary>
        /// 色
        /// </summary>
        private ColorRgba value = new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);

        /// <summary>
        /// プレビュー色
        /// </summary>
        private ColorRgba editingValue = new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);

        /// <summary>
        /// アクティブか？
        /// </summary>
        private bool isActive;

        /// <summary>
        /// RGBの編集を有効にするか
        /// </summary>
        private bool rgbEditEnabled;

        /// <summary>
        /// アルファの編集を有効にするか
        /// </summary>
        private bool alphaEditEnabled;

        /// <summary>
        /// 時間の編集を有効にするか
        /// </summary>
        private bool timeEditEnabled;

        /// <summary>
        /// 値変更イベントの実行中か
        /// </summary>
        private bool changing = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorBox()
        {
            this.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;

            this.ContextMenuStrip = ContextMenuUtility.CreateCopyPaste(
                () =>
                {
                    Clipboard.SetData(
                        ColorClipboardData.ClipboardFormat,
                        new ColorClipboardData
                        {
                            R = this.Value.R,
                            G = this.Value.G,
                            B = this.Value.B,
                            A = this.Value.A,
                            EnabledRgb = this.RgbEditEnabled,
                            EnabledAlpha = this.AlphaEditEnabled
                        });
                },
                () =>
                {
                    if (Clipboard.ContainsData(ColorClipboardData.ClipboardFormat))
                    {
                        var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                        // ペースト前の値を保持
                        var tmpValue = this.Value.Clone() as ColorRgba;

                        // ペーストを反映
                        this.Value = new ColorRgba(
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.R : 1.0f,
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.G : 1.0f,
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.B : 1.0f,
                            (this.AlphaEditEnabled && data.EnabledAlpha) ? data.A : 1.0f);

                        // ペースト前の差分値を復帰
                        this.editingValue.Set(tmpValue);

                        // ペースト時に実行するイベントを処理
                        if (this.ColorPasted != null)
                        {
                            this.ColorPasted();
                        }

                        // 履歴を積む
                        this.NotifyValueChanged();
                    }
                });

            this.ContextMenuStrip.Opening += (s, e) =>
            {
                var enabledPaste = Clipboard.ContainsData(ColorClipboardData.ClipboardFormat);

                if (enabledPaste)
                {
                    var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                    enabledPaste = (data.EnabledRgb == this.RgbEditEnabled) &&
                                   (data.EnabledAlpha == this.AlphaEditEnabled);
                }

                this.ContextMenuStrip.Items[ContextMenuUtility.CopyPasteMenuPasteIndex].Enabled = enabledPaste;
            };

            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;
        }

        /// <summary>
        /// Event triggered when the Value value is changed.
        /// </summary>
        public event EventHandler ColorChanged;

        /// <summary>
        /// Gets or sets the color pasted.
        /// </summary>
        public Action ColorPasted { get; set; }

        /// <summary>
        /// 色
        /// </summary>
        public ColorRgba Value
        {
            get
            {
                return this.value.Clone() as ColorRgba;
            }

            set
            {
                this.SetColorPrimitive(value);

                if (!this.changing)
                {
                    this.editingValue.Set(this.Value);
                }

                if (this.IsActive)
                {
                    this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");
                }
            }
        }

        /// <summary>
        /// 色
        /// </summary>
        public ColorRgba EditingValue
        {
            get
            {
                return this.editingValue.Clone() as ColorRgba;
            }

            set
            {
                this.SetColorPrimitive(value);
            }
        }

        /// <summary>
        /// アクティブか？
        /// </summary>
        public bool IsActive
        {
            get
            {
                return this.isActive;
            }

            set
            {
                this.isActive = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "IsActive");

                this.Invalidate();
            }
        }

        /// <summary>
        /// 色編集情報
        /// </summary>
        public EditingColorInfo EditingColorInfo
        {
            get
            {
                return new EditingColorInfo
                {
                    Control = this.IsUpdatingDataContext ? null : this,
                    Color = this.Value,
                    RgbEditEnabled = this.RgbEditEnabled,
                    AlphaEditEnabled = this.AlphaEditEnabled,
                    TimeEditEnabled = this.TimeEditEnabled
                };
            }

            set
            {
                if (value.RefreshRequest)
                {
                    if (this.TargetName == value.RefreshTargetName)
                    {
                        if (!this.IsUpdatingDataContext && this.IsActive && value.Control == this)
                        {
                            this.EditingValue = value.Color;
                        }

                        // 現在アクティブであればカラーピッカー更新リクエスト
                        value.RefreshColorPickerRequest = this.IsActive;
                    }
                }
                else
                {
                    if (value.Control == this)
                    {
                        if (!this.IsUpdatingDataContext)
                        {
                            this.SetColorPrimitive(value.Color);

                            if (!value.IsEditing)
                            {
                                this.NotifyValueChanged();
                            }
                        }
                    }
                    else
                    {
                        // 他のコントロールがアクティブなので未選択にする
                        this.IsActive = false;
                    }
                }
            }
        }

        /// <summary>
        /// RGBの編集を有効にするか
        /// </summary>
        public bool RgbEditEnabled
        {
            get
            {
                return this.rgbEditEnabled;
            }

            set
            {
                this.rgbEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "RgbEditEnabled");
            }
        }

        /// <summary>
        /// アルファの編集を有効にするか
        /// </summary>
        public bool AlphaEditEnabled
        {
            get
            {
                return this.alphaEditEnabled;
            }

            set
            {
                this.alphaEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "AlphaEditEnabled");
            }
        }

        /// <summary>
        /// 時間の編集を有効にするか
        /// </summary>
        public bool TimeEditEnabled
        {
            get
            {
                return this.timeEditEnabled;
            }

            set
            {
                this.timeEditEnabled = value;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "TimeEditEnabled");
            }
        }

        /// <summary>
        /// データコンテキスト切り替え中かどうかを取得するプロパティ
        /// </summary>
        public bool IsUpdatingDataContext
        {
            get { return Extenders.LogicalTreeElementExtender.IsUpdatingDataContext; }
        }

        /// <summary>
        /// 編集対象名
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        /// Gets or sets the IExecutable to run when the Value property changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// OnValueChangedでRGBとAの値を分離するかどうかを指定します。
        /// </summary>
        public bool DisableAlphaSplit { get; set; }

        /// <summary>
        /// <see cref="T:System.Windows.Forms.Control"/> とその子コントロールが使用しているアンマネージ リソースを解放します。オプションで、マネージ リソースも解放します。
        /// </summary>
        /// <param name="disposing">マネージ リソースとアンマネージ リソースの両方を解放する場合は true。アンマネージ リソースだけを解放する場合は false。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.Paint"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.PaintEventArgs"/>。</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            const float ActiveEdgeWidth = 2.0f;

            base.OnPaint(e);

            var rect = this.ClientRectangle;

            var innerRect = rect;
            innerRect.X += (int)ActiveEdgeWidth;
            innerRect.Y += (int)ActiveEdgeWidth;
            innerRect.Width  -= (int)ActiveEdgeWidth * 2;
            innerRect.Height -= (int)ActiveEdgeWidth * 2;

            var drawingColor = (ColorRgba)this.Value.Clone();
            if (this.AlphaEditEnabled)
            {
                drawingColor.R = 1.0f;
                drawingColor.G = 1.0f;
                drawingColor.B = 1.0f;
                e.Graphics.FillRectangle(Brushes.Black, innerRect);
            }
            else
            {
                drawingColor.A = 1.0f;
            }

            using (var brush = new SolidBrush(
                ColorUtility.ApplyColorGamma(ColorUtility.NormalizeColor(drawingColor)).ToWinColor()))
            {
                e.Graphics.FillRectangle(brush, innerRect);
            }

            if (this.IsActive)
            {
                using (var pen = new Pen(System.Drawing.Brushes.DarkBlue, ActiveEdgeWidth))
                {
                    e.Graphics.DrawRectangle(pen, rect);
                }
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.MouseDown"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.MouseEventArgs"/>。</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            this.Focus();

            this.IsActive = true;
            this.EditingValue = this.Value;
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingColorInfo");
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");
            this.Invalidate();
        }

        /// <summary>
        /// Called when the Value property changes.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected virtual void OnValueChanged(ValueChangedExEventArgs e)
        {
            IExecutable executable = this.ValueChangedExecutable;
            if (executable != null && executable.CanExecute(e))
            {
                executable.Execute(e);
            }
        }

        /// <summary>
        /// Handle GammaCorrectionChanged event from application.
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.Invalidate();
        }

        /// <summary>
        /// カラーをコントロールにセットします。
        /// </summary>
        /// <param name="color">カラー</param>
        private void SetColorPrimitive(ColorRgba color)
        {
            if (this.value != color)
            {
                this.value.Set(color);
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Value");

                if (this.ColorChanged != null)
                {
                    this.ColorChanged(this, EventArgs.Empty);
                }

                this.Invalidate();
            }
        }

        /// <summary>
        /// OnValueChangedを呼び出して、履歴に積む値を確定します。
        /// </summary>
        private void NotifyValueChanged()
        {
            this.changing = true;

            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.Validation, propertyName: "Value");
            if (!this.RgbEditEnabled && this.AlphaEditEnabled && !this.DisableAlphaSplit)
            {
                // OnValueChanged使うとコンバータが通せないので分岐で何とかする
                this.OnValueChanged(new ValueChangedExEventArgs(
                    this.EditingValue.A,
                    this.Value.A,
                    false,
                    this.TargetName));
            }
            else
            {
                this.OnValueChanged(new ValueChangedExEventArgs(
                    this.EditingValue,
                    this.Value,
                    false,
                    this.TargetName));
            }

            this.editingValue.Set(this.value);

            this.changing = false;
        }
    }
}
