﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UILogic.Commands;
using System;
using System.ComponentModel;
using System.IO;
using System.Text;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// ファイル選択ダイアログを表示するビヘイビアです。
    /// </summary>
    public class ShowOpenFileDialogBehavior : Behavior<UIButton>, INotifyPropertyChanged
    {
        /// <summary>
        /// ファイルパスです。
        /// </summary>
        private string filePath;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShowOpenFileDialogBehavior()
        {
            this.FileType           = LastAccessDirectoryTypes.Any;
            this.AllowMultiByteName = true;
            this.BasePath           = string.Empty;
            this.MaxByteCount       = 0;
            this.FilePath           = string.Empty;
        }

        /// <summary>
        /// ファイルタイプを取得または設定します。
        /// </summary>
        public LastAccessDirectoryTypes FileType { get; set; }

        /// <summary>
        /// マルチバイト文字を含んだ名前を許可するかどうか取得または設定します。
        /// </summary>
        public bool AllowMultiByteName { get; set; }

        /// <summary>
        /// ファイルパスを相対パスに変換するときのベースパスを取得または設定します。
        /// </summary>
        public string BasePath { get; set; }

        /// <summary>
        /// パスの最大バイト数を取得または設定します。
        /// 制限しないときは 0 を指定します。
        /// </summary>
        public int MaxByteCount { get; set; }

        /// <summary>
        /// ファイルパスを取得または設定します。
        /// </summary>
        public string FilePath
        {
            get { return this.filePath; }

            set
            {
                this.filePath = value;

                using (new CommandCombiner())
                {
                    this.PropertyChanged?.Invoke(this, new PropertyChangedEventArgs("FilePath"));
                }
            }
        }

        /// <summary>
        /// ボタンにアタッチするときの処理を行います。
        /// </summary>
        protected override void OnAttached()
        {
            this.AssociatedObject.Click += this.OnClicked;
        }

        /// <summary>
        /// ボタンからデタッチするときの処理を行います。
        /// </summary>
        protected override void OnDetaching()
        {
            this.AssociatedObject.Click -= this.OnClicked;
        }

        /// <summary>
        /// ボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnClicked(object sender, EventArgs e)
        {
            // ファイル選択ダイアログを作成
            OpenFileDialog dialog = new OpenFileDialog();

            dialog.AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle;
            dialog.Multiselect = false;
            dialog.Filter = IOConstants.GetFilter(this.FileType);

            // 初期ディレクトリを設定
            if (string.IsNullOrEmpty(this.FilePath) == false)
            {
                dialog.InitialDirectory = PathUtility.GetDirectoryName(this.FilePath);
            }
            else
            {
                dialog.InitialDirectory = ApplicationIOManager.GetLastAccessedDir(this.FileType);
            }

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(this.AssociatedObject);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(this.FileType, dialog.FileName);

            // 選択されたファイルを取得
            string filePath = dialog.FileName;

            // パスを相対パスに変換する
            if (string.IsNullOrEmpty(this.BasePath) == false)
            {
                filePath = PathUtility.GetRelativePath(this.BasePath, filePath);
            }

            // ファイル名にマルチバイト文字が使われていないかチェック
            if (this.AllowMultiByteName == false)
            {
                string name = Path.GetFileName(filePath);
                byte[] nameUtf8 = Encoding.UTF8.GetBytes(name);

                if (name.Length != nameUtf8.Length)
                {
                    MessageBox.Show(Properties.Resources.FileDragDropBehaviorWarningMultiByteName, Properties.Resources.FileDragDropBehaviorWarningCaption);
                    return;
                }
            }

            // ファイルパスの文字列が最大バイト数を超えないかチェック
            if (this.MaxByteCount > 0)
            {
                byte[] pathUtf8 = Encoding.UTF8.GetBytes(filePath);

                if (pathUtf8.Length > this.MaxByteCount)
                {
                    MessageBox.Show(Properties.Resources.FileDragDropBehaviorWarningByteCount, Properties.Resources.FileDragDropBehaviorWarningCaption);
                    return;
                }
            }

            // FilePathを更新
            this.FilePath = filePath;
        }

        #region INotifyPropertyChanged members

        /// <summary>
        /// プロパティの値が変更されたときに発生します。
        /// </summary>
        public new event PropertyChangedEventHandler PropertyChanged;

        #endregion
    }
}
