﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Windows.Forms;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UILogic.Manager;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// テクスチャビューアを開くBehaviorです。
    /// </summary>
    public class OpenTextureViewerBehavior : Behavior<UIPanel>, INotifyPropertyChanged
    {
        /// <summary>
        /// ファイルパスです。
        /// </summary>
        private string filePath = string.Empty;

        /// <summary>
        /// カラーチャンネル
        /// </summary>
        private ColorChannelFlags colorChannel = ColorChannelFlags.Rgb;

        /// <summary>
        /// 元のカーソルです。
        /// </summary>
        private Cursor oldCursor;

        /// <summary>
        /// Raised when the value of a property changes.
        /// </summary>
        public new event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// ファイルパスを取得または設定します。
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }

            set
            {
                this.filePath = value;

                if (this.AssociatedObject != null)
                {
                    if (EffectMaker.BusinessLogic.Manager.TextureManager.Instance.CanLoad(this.filePath))
                    {
                        this.AssociatedObject.Cursor = Cursors.Hand;
                    }
                    else
                    {
                        this.AssociatedObject.Cursor = this.oldCursor ?? Cursors.Default;
                    }
                }

                var handler = this.PropertyChanged;
                if (handler != null)
                {
                    handler(this, new PropertyChangedEventArgs("FilePath"));
                }
            }
        }

        /// <summary>
        /// カラーチャンネル
        /// </summary>
        public ColorChannelFlags ColorChannel
        {
            get
            {
                return this.colorChannel;
            }

            set
            {
                this.colorChannel = value;

                var handler = this.PropertyChanged;
                if (handler != null)
                {
                    handler(this, new PropertyChangedEventArgs("ColorChannel"));
                }
            }
        }

        /// <summary>
        /// UIButtonにアタッチするときの処理を行います。
        /// </summary>
        protected override void OnAttached()
        {
            this.AssociatedObject.Click += this.OnClick;
            this.AssociatedObject.DoubleClick += this.OnDoubleClick;

            this.oldCursor = this.AssociatedObject.Cursor;
            this.AssociatedObject.Cursor = Cursors.Hand;
        }

        /// <summary>
        /// UIButtonからデタッチするときの処理を行います。
        /// </summary>
        protected override void OnDetaching()
        {
            this.AssociatedObject.Cursor = this.oldCursor;

            this.AssociatedObject.DoubleClick -= this.OnDoubleClick;
            this.AssociatedObject.Click -= this.OnClick;
        }

        /// <summary>
        /// クリックされたときの処理を行います。
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnClick(object sender, EventArgs e)
        {
            if (TextureViewerDialogManager.IsVisibleDialog)
            {
                if (EffectMaker.BusinessLogic.Manager.TextureManager.Instance.CanLoad(this.FilePath))
                {
                    TextureViewerDialogManager.TargetColorChannel = this.ColorChannel;
                    TextureViewerDialogManager.TargetTexturePath = this.FilePath;
                }
            }
        }

        /// <summary>
        /// ダブルクリックされたときの処理を行います。
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnDoubleClick(object sender, EventArgs e)
        {
            if (EffectMaker.BusinessLogic.Manager.TextureManager.Instance.CanLoad(this.FilePath))
            {
                TextureViewerDialogManager.TargetColorChannel = this.ColorChannel;
                TextureViewerDialogManager.TargetTexturePath = this.FilePath;
            }
        }
    }
}
