﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.DataModelMaker.Core.Template;
using EffectMaker.Udd2Converter.Properties;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// UDD2.0定義からXamlファイルを生成するクラスです。
    /// </summary>
    internal class Udd2XamlGenerator
    {
        /// <summary>
        /// コントロールの配置先を表す列挙体です。
        /// </summary>
        private enum LayoutSide
        {
            Left,
            Right,
            ColorLeft,
        }

        /// <summary>
        /// UDD2.0定義
        /// </summary>
        private readonly UserDataDefinition uddInfo;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="uddInfo">パース済みのUDD2.0定義</param>
        public Udd2XamlGenerator(UserDataDefinition uddInfo)
        {
            this.uddInfo = uddInfo;
        }

        /// <summary>
        /// XamlによるUI定義ファイルを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデルおよびファイル名</param>
        /// <param name="outputPath">出力先フォルダパス</param>
        /// <returns>成功すればtrue,失敗すればfalse.</returns>
        public bool Generate(string dataModelName, string outputPath)
        {
            string code;
            var result = this.GenerateXamlCode(dataModelName, out code);
            if (result)
            {
                File.WriteAllText(Path.Combine(outputPath, dataModelName + ".xaml"), code);
            }

            return result;
        }

        /// <summary>
        /// XamlによるUIを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデル名</param>
        /// <param name="output">出力先インスタンス</param>
        /// <returns>成功すればtrue,失敗したらfalse.</returns>
        public bool GenerateXamlCode(string dataModelName, out string output)
        {
            var templateHelper = new TemplateHelper();
            templateHelper.ParseSimpleTemplate(Resources.Udd2_UI_Main);

            templateHelper.SetTemplateTagValue("Label", uddInfo.Name);
            templateHelper.SetTemplateTagValue("Resources", MakeResources(uddInfo));
            templateHelper.SetTemplateTagValue("Groups", MakeGroups(uddInfo));

            output = templateHelper.Compose();

            return true;
        }

        /// <summary>
        /// グループを生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeGroups(UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();
            int index = 0;

            foreach (var group in uddInfo.Layout.Groups)
            {
                var groupTemplate = new TemplateHelper();
                groupTemplate.ParseSimpleTemplate(
                    group.IsColorGroup ? Resources.Udd2_UI_ColorGroup : Resources.Udd2_UI_Group);
                groupTemplate.SetTemplateTagValue("Label", group.Name);
                groupTemplate.SetTemplateTagValue("DataContext", string.Format("Group{0}", index));
                groupTemplate.SetTemplateTagValue("ControlSets", MakeControls(group));
                ++index;
                builder.Append(groupTemplate.Compose());
            }

            return builder.ToString();
        }

        /// <summary>
        /// ルート要素のリソース定義を生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeResources(UserDataDefinition uddInfo)
        {
            var resourcesBuilder = new StringBuilder();

            foreach (var param in uddInfo.ParameterDefinitions)
            {
                if (param.Type == PropertyType.RadioButtons || param.Type == PropertyType.ComboBox)
                {
                    var itemsTemplate = new TemplateHelper();
                    itemsTemplate.ParseSimpleTemplate(Resources.Udd2_UI_KeyValuePairList);

                    var keyValueTemplate = new TemplateHelper();
                    keyValueTemplate.ParseSimpleTemplate(Resources.Udd2_UI_KeyValuePairItem);

                    var options = (OptionGroup)param.UISetting;
                    itemsTemplate.ResetTemplateTags();
                    itemsTemplate.SetTemplateTagValue("ResourceKey", string.Format("{0}Items", param.MemberName));

                    var keyValueBuilder = new StringBuilder();

                    foreach (var option in options.Options)
                    {
                        keyValueTemplate.ResetTemplateTags();
                        keyValueTemplate.SetTemplateTagValue("Label", option.Text);
                        keyValueTemplate.SetTemplateTagValue("Value", ((ulong)Math.Pow(2, option.Index)).ToString());
                        keyValueBuilder.Append(keyValueTemplate.Compose());
                    }

                    itemsTemplate.SetTemplateTagValue("Items", keyValueBuilder.ToString());
                    resourcesBuilder.Append(itemsTemplate.Compose());
                }
            }

            return resourcesBuilder.ToString();
        }

        /// <summary>
        /// コントロール群を生成します。
        /// </summary>
        /// <param name="group">グループ定義</param>
        /// <returns>生成コード</returns>
        private static string MakeControls(UIGroup group)
        {
            var builder = new StringBuilder();
            if (group.IsColorGroup)
            {
                // カラーグループはすべて左側のパネルに並べる
                foreach (var paramDef in group.Parameters)
                {
                    builder.Append(MakeControl(paramDef, LayoutSide.ColorLeft));
                }
            }
            else
            {
                // それ以外のグループは左右への割り振りを調整する
                int controlsNum = group.Parameters.Sum(p => p.UsingRows);
                int measuringNum = 0;
                var leftSide = new List<ParameterDefinition>();
                var rightSide = new List<ParameterDefinition>();
                foreach (var paramDef in group.Parameters)
                {
                    if (measuringNum * 2 < controlsNum)
                    {
                        leftSide.Add(paramDef);

                        // チェックボックス群だけで左側の許容域を超過してしまう場合の補正
                        if (paramDef.Type == PropertyType.CheckButtons
                            && paramDef.UsingRows * 2 > controlsNum)
                        {
                            var leftMax = (controlsNum + 1) / 2;
                            var option = (OptionGroup)paramDef.UISetting;
                            option.LeftSideNum = leftMax - measuringNum;
                            rightSide.Add(paramDef);
                            measuringNum += option.LeftSideNum;
                            continue;
                        }

                        measuringNum += paramDef.UsingRows;
                    }
                    else
                    {
                        rightSide.Add(paramDef);
                    }
                }

                var panelBuilder = new StringBuilder();
                var stackPanelTemplate = new TemplateHelper();
                stackPanelTemplate.ParseSimpleTemplate(Resources.Udd2_UI_StackPanel);
                foreach (var paramDef in leftSide)
                {
                    panelBuilder.Append(MakeControl(paramDef, LayoutSide.Left));
                }

                stackPanelTemplate.SetTemplateTagValue("ControlSets", panelBuilder.ToString());
                builder.Append(stackPanelTemplate.Compose(16));

                panelBuilder.Clear();
                stackPanelTemplate.ResetTemplateTags();
                foreach (var paramDef in rightSide)
                {
                    panelBuilder.Append(MakeControl(paramDef, LayoutSide.Right));
                }

                stackPanelTemplate.SetTemplateTagValue("ControlSets", panelBuilder.ToString());
                builder.Append(stackPanelTemplate.Compose(16));
            }

            return builder.ToString();
        }

        /// <summary>
        /// コントロール部を生成します。
        /// </summary>
        /// <param name="paramDef">パラメータ定義</param>
        /// <param name="side">左右どちらに配置するか</param>
        /// <returns>生成コード</returns>
        private static string MakeControl(ParameterDefinition paramDef, LayoutSide side)
        {
            var builder = new StringBuilder();
            var template = new TemplateHelper();
            int indent = side == LayoutSide.ColorLeft ? 16 : 0;

            if (paramDef.Type == PropertyType.CheckButtons)
            {
                // チェックボックスは1つの定義から複数の要素を生成するので特殊対応
                template.ParseSimpleTemplate(Resources.Udd2_UI_CheckBox);
                MakeCheckBox(builder, paramDef, template, side);
            }
            else
            {
                switch (paramDef.Type)
                {
                    case PropertyType.ColorRgba:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_ColorRgba);
                        break;
                    case PropertyType.ColorRgb:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_ColorRgb);
                        break;
                    case PropertyType.ColorA:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_ColorA);
                        break;
                    case PropertyType.ColorAnimeRgb:
                    case PropertyType.ColorAnimeA:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_ColorAnime);
                        template.SetTemplateTagValue("EnableRgb", (paramDef.Type == PropertyType.ColorAnimeRgb).ToString());
                        template.SetTemplateTagValue("EnableAlpha", (paramDef.Type == PropertyType.ColorAnimeA).ToString());
                        break;
                    case PropertyType.Slider1F:
                    case PropertyType.Slider1I:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_Slider);
                        AddSliderSetting(paramDef, template);
                        break;
                    case PropertyType.Slider2F:
                    case PropertyType.Slider2I:
                    case PropertyType.Slider3F:
                    case PropertyType.Slider3I:
                    case PropertyType.Slider4F:
                    case PropertyType.Slider4I:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_VectorSlider);
                        AddSliderSetting(paramDef, template);
                        break;
                    case PropertyType.RadioButtons:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_RadioButton);
                        template.SetTemplateTagValue("ResourceKey", string.Format("{0}Items", paramDef.MemberName));
                        break;
                    case PropertyType.ComboBox:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_ComboBox);
                        template.SetTemplateTagValue("AvailableItems", string.Format("{{Resource {0}Items}}", paramDef.MemberName));
                        break;
                    case PropertyType.StringComboBox:
                    case PropertyType.IntComboBox:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_ComboBox);
                        template.SetTemplateTagValue("AvailableItems", string.Format("{{Binding {0}Items}}", paramDef.MemberName));
                        break;
                    case PropertyType.TextBox:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_TextBox);
                        template.SetTemplateTagValue("TextBoxHeight", (paramDef.UsingRows * 14).ToString());
                        template.SetTemplateTagValue("ArrayLength", paramDef.ArraySize.ToString());
                        break;
                    case PropertyType.FilePathTextBox:
                        template.ParseSimpleTemplate(Resources.Udd2_UI_FilePathTextBox);
                        AddFilePathSetting(paramDef, template);
                        break;
                }

                template.SetTemplateTagValue("Label", paramDef.Name);
                template.SetTemplateTagValue("BindSource", paramDef.MemberName);

                builder.Append(template.Compose(indent));
            }

            if (!string.IsNullOrEmpty(paramDef.Comment))
            {
                var commentTemplate = new TemplateHelper();
                commentTemplate.ParseSimpleTemplate(Resources.Udd2_UI_Comment);
                commentTemplate.SetTemplateTagValue("Comment", paramDef.Comment);
                builder.Append(commentTemplate.Compose(indent));
            }

            return builder.ToString();
        }

        /// <summary>
        /// チェックボックス群を生成します。
        /// </summary>
        /// <param name="builder">コントロール生成用のビルダー</param>
        /// <param name="paramDef">チェックボックスのパラメータ定義</param>
        /// <param name="template">チェックボックスのテンプレート</param>
        /// <param name="side">左右どちらへの配置か</param>
        /// <returns></returns>
        private static void MakeCheckBox(StringBuilder builder, ParameterDefinition paramDef, TemplateHelper template, LayoutSide side)
        {
            int indent = side == LayoutSide.ColorLeft ? 16 : 0;
            var setting = (OptionGroup)paramDef.UISetting;
            int i = 0;
            foreach (var option in setting.Options)
            {
                if ((setting.LeftSideNum == -1 || side == LayoutSide.ColorLeft)
                    || (side == LayoutSide.Left && i < setting.LeftSideNum)
                    || (side == LayoutSide.Right && i >= setting.LeftSideNum && setting.LeftSideNum >= 0))
                {
                    template.ResetTemplateTags();
                    template.SetTemplateTagValue("Label", option.Text);
                    template.SetTemplateTagValue("BindSource", paramDef.MemberName);
                    template.SetTemplateTagValue("CheckBoxMask", ((ulong)Math.Pow(2, option.Index)).ToString());
                    builder.Append(template.Compose(indent));
                }

                ++i;
            }
        }

        /// <summary>
        /// テンプレートに対してスライダー設定情報を追加します。
        /// </summary>
        /// <param name="paramDef">スライダー設定を含むパラメータ定義</param>
        /// <param name="template">設定を追加したいテンプレート</param>
        private static void AddSliderSetting(ParameterDefinition paramDef, TemplateHelper template)
        {
            var setting = (SliderSetting)paramDef.UISetting;
            float[] deltaArray = { 0.0001f, 0.001f, 0.01f, 0.1f, 1.0f, 10.0f, 100.0f, 1000.0f, 10000.0f };

            template.SetTemplateTagValue("MinValue", setting.Minimum.ToString());
            template.SetTemplateTagValue("MaxValue", setting.Maximum.ToString());

            bool detected = false;
            int index = -1;
            for (int i = 0; i < deltaArray.Length; ++i)
            {
                if (Math.Abs(setting.Step - deltaArray[i]) < 0.00001f)
                {
                    index = i - 1;
                    detected = true;
                    break;
                }
            }

            if (!detected)
            {
                index = 3;
            }

            template.SetTemplateTagValue("DeltaLevel", index.ToString());

            if (paramDef.IsPrimitive)
            {
                template.SetTemplateTagValue("ShowValueTransformButtons", setting.EnableTransform.ToString());
            }

            template.SetTemplateTagValue("ControlMode", setting.IsTextBoxOnly ? "TextOnly" : "Normal");

            switch (paramDef.Type)
            {
                case PropertyType.Slider1I:
                case PropertyType.Slider2I:
                case PropertyType.Slider3I:
                case PropertyType.Slider4I:
                    template.SetTemplateTagValue("IntegerOnly", "True");
                    break;
                default:
                    template.SetTemplateTagValue("IntegerOnly", "False");
                    break;
            }
        }

        /// <summary>
        /// テンプレートに対してファイルパス設定情報を追加します。
        /// </summary>
        /// <param name="paramDef">ファイルパス設定を含むパラメータ定義</param>
        /// <param name="template">設定を追加したいテンプレート</param>
        private static void AddFilePathSetting(ParameterDefinition paramDef, TemplateHelper template)
        {
            var setting = (FilePathSetting)paramDef.UISetting;

            template.SetTemplateTagValue("ArrayLength", paramDef.ArraySize.ToString());
            template.SetTemplateTagValue("FileExtension", setting.FileExtension);
            template.SetTemplateTagValue("FileType", setting.FileType.ToString());
        }
    }
}
