﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Text;
using EffectMaker.DataModelMaker.Core.Template;
using EffectMaker.Udd2Converter.Properties;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// UDD2.0 定義からXPathテーブルファイルを生成するクラスです。
    /// </summary>
    public class Udd2XPathTableGenerator
    {
        /// <summary>
        /// UDD2.0 定義
        /// </summary>
        private readonly UserDataDefinition uddInfo;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="uddInfo">パース済みの UDD2.0 定義</param>
        public Udd2XPathTableGenerator(UserDataDefinition uddInfo)
        {
            this.uddInfo = uddInfo;
        }

        /// <summary>
        /// XPath テーブルファイルを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデルおよびファイル名</param>
        /// <param name="outputPath">出力先フォルダパス</param>
        /// <returns>成功したときは true、失敗したときは false を返します。</returns>
        public bool Generate(string dataModelName, string outputPath)
        {
            string code;

            bool result = this.GenerateXPathTable(dataModelName, out code);

            if (result)
            {
                File.WriteAllText(Path.Combine(outputPath, dataModelName + ".xml"), code);
            }

            return result;
        }

        /// <summary>
        /// XPath テーブルを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデル名</param>
        /// <param name="output">出力先インスタンス</param>
        /// <returns>成功したときは true、失敗したときは false を返します。</returns>
        public bool GenerateXPathTable(string dataModelName, out string output)
        {
            TemplateHelper templateHelper = new TemplateHelper();
            templateHelper.ParseSimpleTemplate(Resources.Udd2_XPT_Main);

            templateHelper.SetTemplateTagValue("SearchItems", MakeSearchItems(dataModelName, this.uddInfo));

            output = templateHelper.Compose();

            return true;
        }

        /// <summary>
        /// 検索項目を生成します。
        /// </summary>
        /// <param name="dataModelName">データモデル名</param>
        /// <param name="uddInfo">UDD2.0 定義</param>
        /// <returns>生成コード</returns>
        private static string MakeSearchItems(string dataModelName, UserDataDefinition uddInfo)
        {
            StringBuilder builder = new StringBuilder();

            TemplateHelper searchItemTemplate = new TemplateHelper();
            searchItemTemplate.ParseSimpleTemplate(Resources.Udd2_XPT_SearchItem);

            TemplateHelper enumDefinitionItemTemplate = new TemplateHelper();
            enumDefinitionItemTemplate.ParseSimpleTemplate(Resources.Udd2_XPT_EnumDefinitionItem);

            TemplateHelper optionItemTemplate = new TemplateHelper();
            optionItemTemplate.ParseSimpleTemplate(Resources.Udd2_XPT_OptionItem);

            string uiPathFormat = string.Empty;
            string xPathFormat = string.Empty;

            switch (uddInfo.Kind)
            {
                case UserDataKind.CustomAction:
                    uiPathFormat = "エミッタノード/カスタムアクション({0})タブ/{1}グループ";
                    xPathFormat = "//EmitterSetData/EmitterList/EmitterData/CustomActionData/Settings/{0}/{1}";
                    break;
                case UserDataKind.CustomShader:
                    uiPathFormat = "エミッタノード/Cstmシェーダ({0})タブ/{1}グループ";
                    xPathFormat = "//EmitterSetData/EmitterList/EmitterData/EmitterCustomShaderData/Settings/{0}/{1}";
                    break;
                case UserDataKind.CustomField:
                    throw new NotImplementedException();
                case UserDataKind.EmitterExtParams:
                    uiPathFormat = "エミッタノード/{0}タブ/{1}グループ";
                    xPathFormat = "//EmitterSetData/EmitterList/EmitterData/EmitterExtParams/EmitterExtParamsSettingData/{1}";
                    break;
            }

            foreach (UIGroup group in uddInfo.Layout.Groups)
            {
                foreach (ParameterDefinition paramDef in group.Parameters)
                {
                    // チェックボックスは1つの定義から複数の要素を生成するので特殊対応
                    if (paramDef.Type == PropertyType.CheckButtons)
                    {
                        string typeName = "BitTable";
                        string memberName = (paramDef.IsCustomShaderBuiltInMember ? "Shader" + paramDef.MemberName : paramDef.MemberName) + ".BitTable";

                        OptionGroup setting = (OptionGroup)paramDef.UISetting;

                        foreach (OptionGroup.OptionElement option in setting.Options)
                        {
                            searchItemTemplate.ResetTemplateTags();
                            searchItemTemplate.SetTemplateTagValue("UIPath", string.Format(uiPathFormat, uddInfo.Name, group.Name));
                            searchItemTemplate.SetTemplateTagValue("Title", string.Format("{0}を取得または設定します。", option.Text));
                            searchItemTemplate.SetTemplateTagValue("XPath", string.Format(xPathFormat, dataModelName, memberName));
                            searchItemTemplate.SetTemplateTagValue("Type", typeName);

                            optionItemTemplate.ResetTemplateTags();
                            optionItemTemplate.SetTemplateTagValue("Key", "FlagName");
                            optionItemTemplate.SetTemplateTagValue("Value", option.UniqueKey);

                            searchItemTemplate.SetTemplateTagValue("OptionItems", optionItemTemplate.Compose());

                            builder.Append(searchItemTemplate.Compose());
                        }
                    }
                    // ラジオボタンとコンボボックスは選択肢を列挙するため特殊対応
                    else if (paramDef.Type == PropertyType.RadioButtons || paramDef.Type == PropertyType.ComboBox)
                    {
                        string typeName = "BitTable";
                        string memberName = (paramDef.IsCustomShaderBuiltInMember ? "Shader" + paramDef.MemberName : paramDef.MemberName) + ".BitTable";

                        searchItemTemplate.ResetTemplateTags();
                        searchItemTemplate.SetTemplateTagValue("UIPath", string.Format(uiPathFormat, uddInfo.Name, group.Name));
                        searchItemTemplate.SetTemplateTagValue("Title", string.Format("{0}を取得または設定します。", paramDef.Name));
                        searchItemTemplate.SetTemplateTagValue("XPath", string.Format(xPathFormat, dataModelName, memberName));
                        searchItemTemplate.SetTemplateTagValue("Type", typeName);

                        // 選択肢を列挙する
                        {
                            var enumDefinitionItemsBuilder = new StringBuilder();

                            OptionGroup setting = (OptionGroup)paramDef.UISetting;

                            foreach (OptionGroup.OptionElement option in setting.Options)
                            {
                                enumDefinitionItemTemplate.ResetTemplateTags();
                                enumDefinitionItemTemplate.SetTemplateTagValue("Key", option.Text);
                                enumDefinitionItemTemplate.SetTemplateTagValue("Value", option.UniqueKey);

                                enumDefinitionItemsBuilder.Append(enumDefinitionItemTemplate.Compose());
                            }

                            searchItemTemplate.SetTemplateTagValue("EnumDefinitionItems", enumDefinitionItemsBuilder.ToString());
                        }

                        builder.Append(searchItemTemplate.Compose());
                    }
                    else
                    {
                        string typeName = GetTypeName(paramDef);
                        string memberName = (paramDef.IsCustomShaderBuiltInMember ? "Shader" + paramDef.MemberName : paramDef.MemberName);

                        searchItemTemplate.ResetTemplateTags();
                        searchItemTemplate.SetTemplateTagValue("UIPath", string.Format(uiPathFormat, uddInfo.Name, group.Name));
                        searchItemTemplate.SetTemplateTagValue("Title", string.Format("{0}を取得または設定します。", paramDef.Name));
                        searchItemTemplate.SetTemplateTagValue("XPath", string.Format(xPathFormat, dataModelName, memberName));
                        searchItemTemplate.SetTemplateTagValue("Type", typeName);

                        builder.Append(searchItemTemplate.Compose());
                    }
                }
            }

            return builder.ToString();
        }

        /// <summary>
        /// UDD 上の型情報をデータモデル上の型名に変換します。
        /// </summary>
        /// <param name="def">パラメータ定義</param>
        /// <returns>コード上での型名</returns>
        private static string GetTypeName(ParameterDefinition def)
        {
            string typeName = "int";

            switch (def.Type)
            {
                case PropertyType.ColorRgba:
                case PropertyType.ColorRgb:
                case PropertyType.ColorA:
                    typeName = "ColorRgba";
                    break;
                case PropertyType.ColorAnimeRgb:
                case PropertyType.ColorAnimeA:
                    typeName = "AnimationTableData";
                    break;
                case PropertyType.Slider1F:
                    typeName = "float";
                    break;
                case PropertyType.Slider2F:
                    typeName = "Vector2f";
                    break;
                case PropertyType.Slider2I:
                    typeName = "Vector2i";
                    break;
                case PropertyType.Slider3F:
                    typeName = "Vector3f";
                    break;
                case PropertyType.Slider3I:
                    typeName = "Vector3i";
                    break;
                case PropertyType.Slider4F:
                    typeName = "Vector4f";
                    break;
                case PropertyType.Slider4I:
                    typeName = "Vector4i";
                    break;
                case PropertyType.CheckButtons:
                    typeName = "BitTable";
                    break;
                case PropertyType.RadioButtons:
                    typeName = "BitTable";
                    break;
                case PropertyType.ComboBox:
                    typeName = "BitTable";
                    break;
                case PropertyType.TextBox:
                    typeName = "string";
                    break;
                case PropertyType.FilePathTextBox:
                    typeName = "string";
                    break;
            }

            return typeName;
        }
    }
}
