﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.Udd2Converter.Properties;
using EffectMaker.DataModelMaker.Core.Template;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// UDD2.0定義からデータモデルを生成するクラスです。
    /// </summary>
    internal class Udd2DataModelGenerator
    {
        /// <summary>
        /// コードブロックのインデント
        /// </summary>
        private const string Indent = "            ";

        /// <summary>
        /// UDD2.0のシステムバージョン
        /// </summary>
        private const uint SystemVersion = 4;

        /// <summary>
        /// UDD2.0定義
        /// </summary>
        private readonly UserDataDefinition uddInfo;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="uddInfo">パース済みのUDD2.0定義</param>
        public Udd2DataModelGenerator(UserDataDefinition uddInfo)
        {
            this.uddInfo = uddInfo;
        }

        /// <summary>
        /// データモデルソースファイルを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデルおよびファイル名</param>
        /// <param name="outputPath">出力先フォルダパス</param>
        /// <returns>成功すればtrue,失敗すればfalse.</returns>
        public bool Generate(string dataModelName, string outputPath)
        {
            string code;
            var result = this.GenerateSourceCode(dataModelName, out code);
            if (result)
            {
                File.WriteAllText(Path.Combine(outputPath, dataModelName + ".cs"), code);
            }

            return result;
        }

        /// <summary>
        /// データモデルソースを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデルおよびファイル名</param>
        /// <param name="output">ソースコード出力先インスタンス</param>
        /// <returns>成功すればtrue,失敗すればfalse.</returns>
        public bool GenerateSourceCode(string dataModelName, out string output)
        {
            output = "";
            var templateHelper = new TemplateHelper();
            templateHelper.ParseSimpleTemplate(Resources.Udd2_EDM_Main);
            templateHelper.SetTemplateTagValue("DataModelDescription", this.uddInfo.Name + "のデータモデルです。");
            templateHelper.SetTemplateTagValue("DataModelClassName", dataModelName);

            string baseClassName;
            if (this.uddInfo.Kind == UserDataKind.CustomAction)
            {
                baseClassName = "CustomActionSettingData";
            }
            else if (this.uddInfo.Kind == UserDataKind.CustomShader)
            {
                baseClassName = "EmitterCustomShaderSettingData";
            }
            else if (this.uddInfo.Kind == UserDataKind.CustomField)
            {
                baseClassName = "CustomFieldUserData";
            }
            else if (this.uddInfo.Kind == UserDataKind.EmitterExtParams)
            {
                baseClassName = "EmitterExtParamsSettingData";
            }
            else
            {
                return false;
            }

            templateHelper.SetTemplateTagValue("DataModelBaseClass", baseClassName);
            templateHelper.SetTemplateTagValue("DataModelBackingFields", MakeBackingFields(this.uddInfo));
            templateHelper.SetTemplateTagValue("DataModelConstructorCode", MakeConstructor(this.uddInfo));
            templateHelper.SetTemplateTagValue("DataModelProperties", MakeProperties(this.uddInfo));
            templateHelper.SetTemplateTagValue("DataModelCopyCode", MakeCopyCode(this.uddInfo));
            templateHelper.SetTemplateTagValue("DataModelWriteCode", MakeWriteCode(this.uddInfo));
            templateHelper.SetTemplateTagValue("DataModelReadCode", MakeReadCode(this.uddInfo));
            templateHelper.SetTemplateTagValue("DataModelUpdateCode", MakeUpdateCode(this.uddInfo));

            output = templateHelper.Compose();

            return true;
        }

        /// <summary>
        /// クラスインスタンスのメンバに対するバッキングフィールドの定義コードを生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeBackingFields(UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();
            var fieldTemplate = new TemplateHelper();
            fieldTemplate.ParseSimpleTemplate(Resources.Udd2_EDM_BackingField);

            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldComment", "このXMLが保存された時点のUDDシステムバージョン");
            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldType", "const uint");
            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldName", "UddSystemVersion");
            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldDefault", SystemVersion.ToString());
            builder.Append(fieldTemplate.Compose());

            fieldTemplate.ResetTemplateTags();
            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldComment", "このUDDのバージョン");
            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldType", "const int");
            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldName", "UddVersion");
            fieldTemplate.SetTemplateTagValue("DataModelBackingFieldDefault", uddInfo.Version.ToString());
            builder.Append(fieldTemplate.Compose());

            // バッキングフィールドはVector/Color型のみ生成します。
            foreach (var def in uddInfo.ParameterDefinitions.Where(def => def.IsPrimitive))
            {
                fieldTemplate.ResetTemplateTags();
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldComment", def.Name);
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldType", GetTypeName(def));
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldName", "_" + def.MemberName);
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldDefault", "null");
                builder.Append(fieldTemplate.Compose());
            }

            // ビットインデックスを操作するプロパティごとに文字列→インデックステーブルを生成します。
            foreach (var def in uddInfo.ParameterDefinitions.Where(def => def.UISetting is OptionGroup))
            {
                fieldTemplate.ResetTemplateTags();
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldComment", def.MemberName + "のビット桁テーブルです。");
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldType", "readonly Dictionary<string, int>");
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldName", def.MemberName + "IndexTable");

                var options = def.UISetting as OptionGroup;
                if (options == null)
                {
                    continue;
                }

                var tableBuilder = new StringBuilder();
                tableBuilder.AppendLine("new Dictionary<string, int>()");
                tableBuilder.AppendLine("        {");

                foreach (var option in options.Options)
                {
                    tableBuilder.AppendLine(string.Format("{0}{{ \"{1}\", {2} }},", Indent, option.UniqueKey, option.Index));
                }

                tableBuilder.Append("        }");
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldDefault", tableBuilder.ToString());
                builder.Append(fieldTemplate.Compose());

                // UI表示用文字列とインデックスのペア配列も作ります。
                fieldTemplate.ResetTemplateTags();
                fieldTemplate.SetTemplateTagValue(
                    "DataModelBackingFieldComment", def.MemberName + "のUI文字列とインデックスのペア配列です。");
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldType", "readonly ComboBoxItemType[]");
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldName", "_" + def.MemberName + "Items");

                var optionBuilder = new StringBuilder();
                optionBuilder.AppendLine("new ComboBoxItemType[]");
                optionBuilder.AppendLine("        {");

                foreach (var opt in options.Options)
                {
                    optionBuilder.AppendLine(string.Format(
                        "{0}new ComboBoxItemType(\"{1}\", {2}),", Indent, opt.Text, opt.Index));
                }

                optionBuilder.Append("        }");
                fieldTemplate.SetTemplateTagValue("DataModelBackingFieldDefault", optionBuilder.ToString());
                builder.Append(fieldTemplate.Compose());
            }

            XmlParseUtil.ChopDoubleNewLine(builder);
            return builder.ToString();
        }

        /// <summary>
        /// コンストラクタ内の初期化コードを生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeConstructor(UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();
            foreach (var def in uddInfo.ParameterDefinitions)
            {
                var color = def.UISetting as ColorSetting;
                var slider = def.UISetting as SliderSetting;
                var option = def.UISetting as OptionGroup;
                var text = def.UISetting as TextSetting;
                var filePath = def.UISetting as FilePathSetting;

                if (color != null)
                {
                    var tmpDefault = new float[4];

                    if (def.Type == PropertyType.ColorRgba)
                    {
                        tmpDefault = color.Default;
                    }
                    else if (def.Type == PropertyType.ColorRgb || def.Type == PropertyType.ColorAnimeRgb)
                    {
                        for (int i = 0; i < 3; ++i)
                        {
                            tmpDefault[i] = color.Default[i];
                        }

                        tmpDefault[3] = 1.0f;
                    }
                    else if (def.Type == PropertyType.ColorA)
                    {
                        for (int i = 0; i < 3; ++i)
                        {
                            tmpDefault[i] = 0.0f;
                        }

                        tmpDefault[3] = color.Default[0];
                    }
                    else if (def.Type == PropertyType.ColorAnimeA)
                    {
                        for (int i = 0; i < 3; ++i)
                        {
                            tmpDefault[i] = 1.0f;
                        }

                        tmpDefault[3] = color.Default[0];
                    }

                    if (def.IsAnimation)
                    {
                        builder.AppendLine(string.Format("{0}this.{1} = new AnimationTableData(0, {2}, {3}, {4}, {5});",
                            Indent,
                            def.MemberName,
                            GetFloatString(tmpDefault[0]),
                            GetFloatString(tmpDefault[1]),
                            GetFloatString(tmpDefault[2]),
                            GetFloatString(tmpDefault[3])));

                        // ループフラグ・ループフレーム・開始位置ランダムフラグの初期値はとりあえず決め打ち
                        builder.AppendLine(string.Format("{0}this.{1}EnableLoop = false;", Indent, def.MemberName));
                        builder.AppendLine(string.Format("{0}this.{1}LoopFrame = 100;", Indent, def.MemberName));
                        builder.AppendLine(string.Format("{0}this.{1}EnableStartRandom = false;", Indent, def.MemberName));
                    }
                    else
                    {
                        builder.AppendLine(string.Format("{0}this.{1} = new ColorRgba({2}, {3}, {4}, {5});",
                            Indent,
                            def.MemberName,
                            GetFloatString(tmpDefault[0]),
                            GetFloatString(tmpDefault[1]),
                            GetFloatString(tmpDefault[2]),
                            GetFloatString(tmpDefault[3])));
                    }
                }
                else if (slider != null)
                {
                    string typeName = GetTypeName(def);
                    string valueStr = string.Empty;

                    // 基本型で済む場合
                    if (typeName == "int" || typeName == "float")
                    {
                        valueStr = typeName == "int"
                            ? ((int)slider.Default[0]).ToString()
                            : GetFloatString(slider.Default[0]);
                        builder.AppendLine(string.Format("{0}this.{1} = {2};",
                            Indent,
                            def.MemberName,
                            valueStr));

                        continue;
                    }
                    else if (typeName.Contains("Vector"))
                    {
                        if (typeName.EndsWith("i"))
                        {
                            for (int i = 0; i < slider.Default.Length; ++i)
                            {
                                valueStr += ((int)slider.Default[i]).ToString();
                                if (i != slider.Default.Length - 1)
                                {
                                    valueStr += ", ";
                                }
                            }
                        }
                        else if (typeName.EndsWith("f"))
                        {
                            for (int i = 0; i < slider.Default.Length; ++i)
                            {
                                valueStr += GetFloatString(slider.Default[i]);
                                if (i != slider.Default.Length - 1)
                                {
                                    valueStr += ", ";
                                }
                            }
                        }
                    }

                    builder.AppendLine(string.Format("{0}this.{1} = new {2}({3});",
                        Indent,
                        def.MemberName,
                        typeName,
                        valueStr));
                }
                else if (option != null)
                {
                    string castExp =
                        (def.Type == PropertyType.CheckButtons || def.Type == PropertyType.RadioButtons || def.Type == PropertyType.ComboBox) ?
                        string.Format("({0})", GetTypeName(def)) : string.Empty;
                    builder.AppendLine(string.Format("{0}this.{1} = {2}{3};", Indent, def.MemberName, castExp, option.Default));
                }
                else if (text != null)
                {
                    builder.AppendLine(string.Format("{0}this.{1} = \"{2}\";", Indent, def.MemberName, text.Default));
                }
                else if (filePath != null)
                {
                    builder.AppendLine(string.Format("{0}this.{1} = \"{2}\";", Indent, def.MemberName, filePath.Default));
                }
                else
                {
                    throw new InvalidDataException("Undefined suitable UI setting: " + def.Name);
                }
            }

            XmlParseUtil.ChopDoubleNewLine(builder);
            return builder.ToString();
        }

        /// <summary>
        /// プロパティ定義コードを生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeProperties(UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();
            foreach (var def in uddInfo.ParameterDefinitions)
            {
                string memberName;
                if (uddInfo.Kind == UserDataKind.CustomShader &&
                    Udd2Parser.CustomShaderBuiltInNames.Contains(def.MemberName))
                {
                    // カスタムシェーダのビルドインデータはダミーで別名を設定して使わない（グループ分け対策）
                    memberName = string.Format("__dummy_{0}__", def.MemberName);
                }
                else
                {
                    memberName = def.MemberName;
                }

                var propTemplate = new TemplateHelper();
                bool isPrimitive = def.IsPrimitive;
                propTemplate.ParseSimpleTemplate(isPrimitive ?
                    Resources.Udd2_EDM_PrimitiveProperty : Resources.Udd2_EDM_AutoProperty);
                propTemplate.SetTemplateTagValue("DataModelPropertyComment", def.Name + "を取得または設定します。");
                propTemplate.SetTemplateTagValue("DataModelPropertyGuid", def.Guid.ToString());
                propTemplate.SetTemplateTagValue("DataModelPropertyType", GetTypeName(def));
                propTemplate.SetTemplateTagValue("DataModelPropertyName", memberName);
                propTemplate.SetTemplateTagValue(
                    "DataModelPropertyGroupIndex",
                    string.Format(
                        "Group{0}{1}",
                        def.GroupIndex,
                        def.Type == PropertyType.CheckButtons ? ",BitField" : string.Empty));
                if (isPrimitive)
                {
                    // ベクター型とアニメーションテーブルにはバッキングフィールドが付く
                    propTemplate.SetTemplateTagValue("DataModelBackingFieldName", "_" + memberName);
                }

                builder.Append(propTemplate.Compose());

                if (def.IsAnimation)
                {
                    // アニメーションデータにはループフラグ・ループフレーム・開始位置ランダムフラグが追加で付く
                    var additionalPropTemplate = new TemplateHelper();
                    additionalPropTemplate.ParseSimpleTemplate(Resources.Udd2_EDM_AutoProperty);

                    var settings = new Tuple<string, Guid, string, string>[3];
                    settings[0] = new Tuple<string, Guid, string, string>(
                        def.Name + "のループモードを取得または設定します。",
                        Guid.NewGuid(),
                        "bool",
                        memberName + "EnableLoop"
                    );
                    settings[1] = new Tuple<string, Guid, string, string>(
                        def.Name + "のループフレーム数を取得または設定します。",
                        Guid.NewGuid(),
                        "int",
                        memberName + "LoopFrame"
                    );
                    settings[2] = new Tuple<string, Guid, string, string>(
                        def.Name + "の開始位置ランダムモードを取得または設定します。",
                        Guid.NewGuid(),
                        "bool",
                        memberName + "EnableStartRandom"
                    );

                    foreach (var setting in settings)
                    {
                        def.AdditionalPropertyGuids[setting.Item4] = setting.Item2;

                        additionalPropTemplate.ResetTemplateTags();
                        additionalPropTemplate.SetTemplateTagValue("DataModelPropertyComment", setting.Item1);
                        additionalPropTemplate.SetTemplateTagValue("DataModelPropertyGuid", setting.Item2.ToString());
                        additionalPropTemplate.SetTemplateTagValue("DataModelPropertyType", setting.Item3);
                        additionalPropTemplate.SetTemplateTagValue("DataModelPropertyName", setting.Item4);
                        additionalPropTemplate.SetTemplateTagValue(
                            "DataModelPropertyGroupIndex", string.Format("Group{0}", def.GroupIndex));
                        builder.Append(additionalPropTemplate.Compose());
                    }
                }
            }

            foreach (var def in uddInfo.ParameterDefinitions.Where(p => p.IsComboBox))
            {
                var option = (OptionGroup)def.UISetting;
                var comboBoxItemTemplate = new TemplateHelper();
                comboBoxItemTemplate.ParseSimpleTemplate(Resources.Udd2_EDM_ComboBoxItems);
                comboBoxItemTemplate.SetTemplateTagValue("DataModelPropertyComment", def.Name + "の選択肢を取得します。");
                comboBoxItemTemplate.SetTemplateTagValue("DataModelPropertyGuid", option.Guid.ToString());
                comboBoxItemTemplate.SetTemplateTagValue("DataModelPropertyName", def.MemberName);
                builder.Append(comboBoxItemTemplate.Compose());
            }

            XmlParseUtil.ChopDoubleNewLine(builder);
            return builder.ToString();
        }

        /// <summary>
        /// コピーコードを生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeCopyCode(UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();
            foreach (var def in uddInfo.ParameterDefinitions)
            {
                if (uddInfo.Kind == UserDataKind.CustomShader &&
                    Udd2Parser.CustomShaderBuiltInNames.Contains(def.MemberName))
                {
                    // カスタムシェーダのビルドインデータは親クラスで処理されるのでスキップ
                    continue;
                }

                builder.AppendLine(string.Format("{0}this.{1} = srcDataModel.{1};", Indent, def.MemberName));
                if (def.IsAnimation)
                {
                    // アニメーションデータの付随データも忘れずにコピーに含めること
                    // 忘れるとコピペや保存時に値が反映されなくて悲しいことになる
                    builder.AppendLine(string.Format(
                        "{0}this.{1}EnableLoop = srcDataModel.{1}EnableLoop;", Indent, def.MemberName));
                    builder.AppendLine(string.Format(
                        "{0}this.{1}LoopFrame = srcDataModel.{1}LoopFrame;", Indent, def.MemberName));
                    builder.AppendLine(string.Format(
                        "{0}this.{1}EnableStartRandom = srcDataModel.{1}EnableStartRandom;", Indent, def.MemberName));
                }
            }

            XmlParseUtil.ChopDoubleNewLine(builder);
            return builder.ToString();
        }

        /// <summary>
        /// 書き込みコードを生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeWriteCode(UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();
            builder.AppendLine(string.Format("{0}this.WriteElement(context, \"UddSystemVersion\", {1});", Indent, SystemVersion));
            builder.AppendLine(string.Format("{0}this.WriteElement(context, \"UddVersion\", {1});", Indent, uddInfo.Version));

            foreach (var def in uddInfo.ParameterDefinitions)
            {
                if (uddInfo.Kind == UserDataKind.CustomShader &&
                    Udd2Parser.CustomShaderBuiltInNames.Contains(def.MemberName))
                {
                    string elemName, baseName;
                    Udd2Parser.GetElementAndBaseNames(def.MemberName, out elemName, out baseName);

                    // シェーダフラグ・スイッチのビットテーブルを書き出し
                    builder.AppendLine(MakeWriteCodeForBitTable(def, baseName, elemName));
                    continue;
                }

                if (def.Type == PropertyType.CheckButtons || def.Type == PropertyType.RadioButtons || def.Type == PropertyType.ComboBox)
                {
                    // ビットテーブルを書き出し
                    builder.AppendLine(MakeWriteCodeForBitTable(def));
                }
                else if (def.IsComboBox)
                {
                    // 選択状態に応じた文字列を書き出し（何も選択されていない場合はnullを許容する）
                    builder.AppendLine(string.Format(
                        "{0}this.WriteElement(context, \"{1}\", this.{1}IndexTable.FirstOrDefault(p => p.Value == this.{1}).Key);",
                        Indent,
                        def.MemberName));
                }
                else
                {
                    builder.AppendLine(string.Format(
                        "{0}this.WriteElement(context, \"{1}\", this.{1});", Indent, def.MemberName));
                    if (def.IsAnimation)
                    {
                        // アニメーションデータの追加プロパティを書き出し
                        builder.AppendLine(string.Format("{0}this.WriteElement(context, \"{1}EnableLoop\", this.{1}EnableLoop);", Indent, def.MemberName));
                        builder.AppendLine(string.Format("{0}this.WriteElement(context, \"{1}LoopFrame\", this.{1}LoopFrame);", Indent, def.MemberName));
                        builder.AppendLine(string.Format("{0}this.WriteElement(context, \"{1}EnableStartRandom\", this.{1}EnableStartRandom);", Indent, def.MemberName));
                    }
                }
            }

            XmlParseUtil.ChopDoubleNewLine(builder);
            return builder.ToString();
        }

        /// <summary>
        /// ビットテーブルを書き出します。
        /// </summary>
        /// <param name="def">パラメータ定義</param>
        /// <param name="baseName">カスタムシェーダビルドインパラメータの場合、Shaderが付かない基底クラスでのメンバ名</param>
        /// <param name="elemName">カスタムシェーダビルドインパラメータの場合、Shaderが付いたXML上でのエイリアス</param>
        /// <returns></returns>
        private static string MakeWriteCodeForBitTable(ParameterDefinition def, string baseName = null, string elemName = null)
        {
            var options = def.UISetting as OptionGroup;
            if (options == null)
            {
                return null;
            }

            // シェーダ関連の場合はbaseもelemも指定されるが、そうでない場合はどちらもメンバ名で揃えておく
            if (string.IsNullOrEmpty(baseName))
            {
                baseName = def.MemberName;
            }

            if (string.IsNullOrEmpty(elemName))
            {
                elemName = def.MemberName;
            }

            var templateHelper = new TemplateHelper();
            templateHelper.ParseSimpleTemplate(Resources.Udd2_EDM_BitTableWrite);
            templateHelper.SetTemplateTagValue("PropertyName", baseName);
            templateHelper.SetTemplateTagValue("ElementName", elemName);
            templateHelper.SetTemplateTagValue("TypeName", GetTypeName(def));

            return templateHelper.Compose();
        }

        /// <summary>
        /// 読み込みコードを生成します。
        /// </summary>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeReadCode(UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();

            // 処理の分岐用にローカルにバージョン情報を読みだしておく
            builder.AppendLine(string.Format(
                "{0}var _uddSystemVersion = this.ReadElement(context, \"UddSystemVersion\", UddSystemVersion);", Indent));
            builder.AppendLine(string.Format(
                "{0}var _uddVersion = this.ReadElement(context, \"UddVersion\", UddVersion);", Indent));

            foreach (var def in uddInfo.ParameterDefinitions)
            {
                if (uddInfo.Kind == UserDataKind.CustomShader &&
                    Udd2Parser.CustomShaderBuiltInNames.Contains(def.MemberName))
                {
                    // 基底クラスのメンバーに Shader が付加された要素から読み込み
                    string elemName, baseName;
                    Udd2Parser.GetElementAndBaseNames(def.MemberName, out elemName, out baseName);
                    builder.AppendLine(string.Format(
                        "{0}this.{1} = this.ReadElement(context, \"{2}\", this.{1});", Indent, baseName, elemName));

                    // ビットテーブルからの読み込み
                    builder.AppendLine(MakeReadCodeForBitTable(def, baseName, elemName));

                    continue;
                }

                if (def.IsComboBox)
                {
                    // 文字列で保存されてない時用の処理（読み込めなくてもエラーにはならない）
                    builder.AppendLine(string.Format(
                        "{0}var _tmp{1} = this.ReadElement(context, \"{1}\", \"\");", Indent, def.MemberName));

                    // UddSystemVersion >= 3 で文字列で保存されたものをテーブルでインデックス化して読みこむようにする
                    builder.AppendLine(string.Format(
                        "{0}this.{1} = _uddSystemVersion < 3 ? this.ReadElement(context, \"{1}\", this.{1}) : " +
                        "!string.IsNullOrEmpty(_tmp{1}) && this.{1}IndexTable.ContainsKey(_tmp{1}) ? " +
                        "this.{1}IndexTable[_tmp{1}] : this.{1};",
                        Indent,
                        def.MemberName));
                }
                else
                {
                    builder.AppendLine(string.Format(
                        "{0}this.{1} = this.ReadElement(context, \"{1}\", this.{1});", Indent, def.MemberName));
                    if (def.Type == PropertyType.CheckButtons || def.Type == PropertyType.RadioButtons || def.Type == PropertyType.ComboBox)
                    {
                        // ビットテーブルからの読み込み
                        builder.AppendLine(MakeReadCodeForBitTable(def));
                    }
                    else if (def.IsAnimation)
                    {
                        // アニメーション用追加データの読み込み
                        builder.AppendLine(string.Format(
                            "{0}this.{1}EnableLoop = this.ReadElement(context, \"{1}EnableLoop\", this.{1}EnableLoop);",
                            Indent,
                            def.MemberName));
                        builder.AppendLine(string.Format(
                            "{0}this.{1}LoopFrame = this.ReadElement(context, \"{1}LoopFrame\", this.{1}LoopFrame);",
                            Indent,
                            def.MemberName));
                        builder.AppendLine(string.Format(
                            "{0}this.{1}EnableStartRandom = this.ReadElement(context, \"{1}EnableStartRandom\", this.{1}EnableStartRandom);",
                            Indent,
                            def.MemberName));
                    }
                }
            }

            // アップデートコードの付加
            var updateDefinitions = uddInfo.ParameterDefinitions.Where(d => !string.IsNullOrEmpty(d.UpdateCode)).ToArray();
            if (updateDefinitions.Any())
            {
                builder.AppendLine();
                foreach (var def in updateDefinitions)
                {
                    builder.AppendLine(string.Format(
                        "{0}this.{1} = this.Update_{1}(context, _uddVersion, this.{1});", Indent, def.MemberName));
                }
            }

            XmlParseUtil.ChopDoubleNewLine(builder);
            return builder.ToString();
        }

        /// <summary>
        /// ビットテーブルの読み込み
        /// </summary>
        /// <param name="def">パラメータ定義</param>
        /// <param name="baseName">カスタムシェーダビルドインパラメータの場合、Shaderが付かない基底クラスでのメンバ名</param>
        /// <param name="elemName">カスタムシェーダビルドインパラメータの場合、Shaderが付いたXML上でのエイリアス</param>
        /// <returns></returns>
        private static string MakeReadCodeForBitTable(ParameterDefinition def, string baseName = null, string elemName = null)
        {
            var options = def.UISetting as OptionGroup;
            if (options == null)
            {
                return null;
            }

            // シェーダ関連の場合はbaseもelemも指定されるが、そうでない場合はどちらもメンバ名で揃えておく
            if (string.IsNullOrEmpty(baseName))
            {
                baseName = def.MemberName;
            }

            if (string.IsNullOrEmpty(elemName))
            {
                elemName = def.MemberName;
            }

            var templateHelper = new TemplateHelper();
            templateHelper.ParseSimpleTemplate(Resources.Udd2_EDM_BitTableRead);
            templateHelper.SetTemplateTagValue("PropertyName", baseName);
            templateHelper.SetTemplateTagValue("ElementName", elemName);
            templateHelper.SetTemplateTagValue("TypeName", GetTypeName(def));

            return templateHelper.Compose();
        }

        /// <summary>
        /// アップデートコードを生成します。
        /// </summary>
        /// <param name="uddInfo">ユーザーデータ定義</param>
        /// <returns>アップデートコードブロック</returns>
        private static string MakeUpdateCode(UserDataDefinition uddInfo)
        {
            var updateDefinitions = uddInfo.ParameterDefinitions.Where(d => !string.IsNullOrEmpty(d.UpdateCode)).ToArray();
            if (!updateDefinitions.Any())
            {
                return string.Empty;
            }

            var builder = new StringBuilder();
            var template = new TemplateHelper();
            template.ParseSimpleTemplate(Resources.Udd2_EDM_UpdateFunc);

            foreach (var def in updateDefinitions)
            {
                builder.AppendLine();
                template.ResetTemplateTags();
                template.SetTemplateTagValue("DataModelPropertyType", GetTypeName(def));
                template.SetTemplateTagValue("DataModelPropertyName", def.MemberName);
                template.SetTemplateTagValue("UpdateUserCode", def.UpdateCode);
                builder.Append(template.Compose());
            }

            return builder.ToString();
        }

        /// <summary>
        /// floatの値をリテラルとして解釈できる文字列に変換します。
        /// </summary>
        /// <param name="value">値</param>
        /// <returns>文字列</returns>
        private static string GetFloatString(float value)
        {
            string retVal = value.ToString();
            if (retVal.Contains("."))
            {
                retVal += "f";
            }

            return retVal;
        }

        /// <summary>
        /// UDD上の型情報をデータモデル上の型名に変換します。
        /// </summary>
        /// <param name="def">パラメータ定義</param>
        /// <returns>コード上での型名</returns>
        private static string GetTypeName(ParameterDefinition def)
        {
            string typeName;
            switch (def.Type)
            {
                case PropertyType.ColorRgba:
                case PropertyType.ColorRgb:
                case PropertyType.ColorA:
                    typeName = "ColorRgba";
                    break;
                case PropertyType.ColorAnimeRgb:
                case PropertyType.ColorAnimeA:
                    typeName = "AnimationTableData";
                    break;
                case PropertyType.Slider1F:
                    typeName = "float";
                    break;
                case PropertyType.Slider2F:
                    typeName = "Vector2f";
                    break;
                case PropertyType.Slider2I:
                    typeName = "Vector2i";
                    break;
                case PropertyType.Slider3F:
                    typeName = "Vector3f";
                    break;
                case PropertyType.Slider3I:
                    typeName = "Vector3i";
                    break;
                case PropertyType.Slider4F:
                    typeName = "Vector4f";
                    break;
                case PropertyType.Slider4I:
                    typeName = "Vector4i";
                    break;
                case PropertyType.CheckButtons:
                    typeName = def.IsCustomShaderBuiltInMember ? "ulong" : "uint";
                    break;
                case PropertyType.RadioButtons:
                    typeName = "uint";
                    break;
                case PropertyType.ComboBox:
                    typeName = "uint";
                    break;
                case PropertyType.TextBox:
                    typeName = "string";
                    break;
                case PropertyType.FilePathTextBox:
                    typeName = "string";
                    break;
                default:
                    typeName = "int";
                    break;
            }

            return typeName;
        }
    }
}
