﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.Foundation.Log;
using EffectMaker.Udd2Converter.Properties;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// 新形式のユーザーデータ定義ファイルからデータモデル、バイナリコンバータ、XamlUIを生成します。
    /// </summary>
    public class Udd2Converter : IUdd2Converter
    {
        /// <summary>
        /// パース結果の情報インスタンス
        /// </summary>
        private UserDataDefinition uddInfo = null;

        /// <summary>
        /// ユーザーデータ定義の対象を取得します。
        /// </summary>
        public string UserDataType
        {
            get
            {
                return this.uddInfo == null ? null : this.uddInfo.Kind.ToString();
            }
        }

        /// <summary>
        /// ユーザーデータ定義の名前を取得します。
        /// </summary>
        public string DefineName
        {
            get
            {
                return this.uddInfo == null ? null : this.uddInfo.Name;
            }
        }

        /// <summary>
        /// エラー報告を取得します。
        /// </summary>
        public string ErrorReport { get; private set; }

        /// <summary>
        /// データモデル名を取得します。
        /// </summary>
        public string DataModelName { get; private set; }

        /// <summary>
        /// データモデルのソースコードを取得します。
        /// </summary>
        public string DataModelCode { get; private set; }

        /// <summary>
        /// UIを定義するXamlコードを取得します。
        /// </summary>
        public string XamlCode { get; private set; }

        /// <summary>
        /// XPath テーブルを取得します。
        /// </summary>
        public string XPathTable { get; private set; }

        /// <summary>
        /// バイナリコンバート定義のソースコードを取得します。
        /// </summary>
        public string BinaryConversionInfoCode { get; private set; }

        /// <summary>
        /// ユーザーデータ定義を読み込みます。
        /// </summary>
        /// <param name="inputFilePath">UDD2.0定義ファイルのパス</param>
        public bool LoadDefinition(string inputFilePath)
        {
            bool result;
            this.ErrorReport = string.Empty;

            try
            {
                var uddConv = new Udd2Parser(File.ReadAllText(inputFilePath));
                this.uddInfo = uddConv.UsedDataDefinition;
                if (this.uddInfo == null)
                {
                    this.ErrorReport = "Definition is not loaded.";
                    return false;
                }

                this.DataModelName = "UDD2_" + Path.GetFileNameWithoutExtension(inputFilePath);

                result = true;
                this.ErrorReport = uddConv.ReportedMessage;
            }
            catch (Exception e)
            {
                Logger.Log(
                    LogLevels.Error,
                    "Udd2Converter.LoadDefinition : Definition load error at {0}.",
                    e.Message);
                this.ErrorReport = e.Message;
                result = false;
            }

            return result;
        }

        /// <summary>
        /// ユーザーデータ定義から各種コードを生成します。
        /// </summary>
        /// <returns>成功すればtrue,失敗したらfalse.</returns>
        public bool Convert()
        {
            if (this.uddInfo == null)
            {
                this.ErrorReport = "Definition is not loaded.";
                return false;
            }

            bool result = false;
            this.ErrorReport = string.Empty;

            try
            {
                var dmGen = new Udd2DataModelGenerator(this.uddInfo);
                string dmCode;
                result |= dmGen.GenerateSourceCode(this.DataModelName, out dmCode);
                this.DataModelCode = dmCode;

                string bciCode;
                var bciGen = new Udd2BinaryConversionInfoGenerator(this.uddInfo);
                result |= bciGen.GenerateSourceCode(this.DataModelName, out bciCode);
                this.BinaryConversionInfoCode = bciCode;

                string xamlCode;
                var xamlGen = new Udd2XamlGenerator(this.uddInfo);
                result |= xamlGen.GenerateXamlCode(this.DataModelName, out xamlCode);
                this.XamlCode = xamlCode;

                string xPathTable;
                var xPathGen = new Udd2XPathTableGenerator(this.uddInfo);
                result |= xPathGen.GenerateXPathTable(this.DataModelName, out xPathTable);
                this.XPathTable = xPathTable;
            }
            catch (Exception e)
            {
                Logger.Log(
                    LogLevels.Error,
                    "Udd2Converter.Convert : Code generate error at {0}.",
                    e.Message);
                this.ErrorReport = e.Message;
                result = false;
            }

            return result;
        }

        /// <summary>
        /// ユーザーデータ定義から各種ファイルを出力します。
        /// </summary>
        /// <param name="outputFolderPath">コンバート結果出力フォルダパス</param>
        /// <returns>成功すればtrue,失敗したらfalse.</returns>
        public bool ConvertToFile(string outputFolderPath)
        {
            if (this.uddInfo == null)
            {
                this.ErrorReport = "Definition is not loaded.";
                return false;
            }

            bool result = true;
            this.ErrorReport = string.Empty;

            try
            {
                var dmGen = new Udd2DataModelGenerator(this.uddInfo);
                result &= dmGen.Generate(this.DataModelName, outputFolderPath);

                var bciGen = new Udd2BinaryConversionInfoGenerator(this.uddInfo);
                result &= bciGen.Generate(this.DataModelName, outputFolderPath);

                var xamlGen = new Udd2XamlGenerator(this.uddInfo);
                result &= xamlGen.Generate(this.DataModelName, outputFolderPath);

                var xPathGen = new Udd2XPathTableGenerator(this.uddInfo);
                result &= xPathGen.Generate(this.DataModelName, outputFolderPath);
            }
            catch (Exception e)
            {
                Logger.Log(
                    LogLevels.Error,
                    "Udd2Converter.Convert : Code output error at {0}.",
                    e.Message);
                this.ErrorReport = e.Message;
                result = false;
            }

            return result;
        }
    }
}
