﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using EffectMaker.DataModelMaker.Generators;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;
using EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData;

namespace EffectMaker.ObsoleteUserDataConverter
{
    /// <summary>
    /// UDD, USDのコンバータです。
    /// プロジェクトの循環参照を避けるため、別DLLでの実装になっています。
    /// </summary>
    public class ObsoleteUserDataConverter : EffectMaker.BusinessLogic.UserData.IObsoleteUserDataConverter
    {
        /// <summary>
        /// USDファイルを新形式にコンバートします。
        /// </summary>
        /// <param name="usdFilePath">USDファイルパス</param>
        /// <param name="dataModelName">出力するデータモデルの名前</param>
        /// <returns>コンバートが成功したときtrueを返します。</returns>
        public bool ConvertUsd(string usdFilePath, string dataModelName, string outputFolderPath, string language)
        {
            Logger.Log(
                "ConsoleLogger",
                LogLevels.Information,
                "Converting .usd files...\n- File path :       {0}\n- Data model name : {1}\n- Output folder :   {2}\n- Language :   {3}\n",
                usdFilePath,
                dataModelName,
                outputFolderPath,
                language);

            using (var stream = new FileStream(usdFilePath, FileMode.Open, FileAccess.Read))
            {
                // Load user definitions.
                var def = SerializationHelper.LoadXmlDocSerializable<CustomShaderDefinition>(stream);
                def.PostLoad();

                // Convert environment variables and relative paths.
                string usdFolderPath = Path.GetDirectoryName(usdFilePath);
                for (int i = 0; i < def.VertexShaderPaths.Count; ++i)
                {
                    string resultPath = PathUtility.LoadEnvironments(def.VertexShaderPaths[i]);
                    if (Path.IsPathRooted(resultPath) == false)
                    {
                        resultPath = Path.GetFullPath(Path.Combine(usdFolderPath, resultPath));
                    }

                    def.VertexShaderPaths[i] = resultPath;
                }

                for (int i = 0; i < def.FragmentShaderPaths.Count; ++i)
                {
                    string resultPath = PathUtility.LoadEnvironments(def.FragmentShaderPaths[i]);
                    if (Path.IsPathRooted(resultPath) == false)
                    {
                        resultPath = Path.GetFullPath(Path.Combine(usdFolderPath, resultPath));
                    }

                    def.FragmentShaderPaths[i] = resultPath;
                }

                Logger.Log(
                    "ConsoleLogger",
                    LogLevels.Information,
                    "Load .usd file ==> Success!");

                // Convert data models, binary conversion info and UI Xaml.
                using (var generator = new UserDefinitionDataModelGenerator())
                {
                    generator.DataModelBaseClassName = "EmitterCustomShaderSettingData";
                    generator.SkipFirstCheckBoxGroup = true;     // EmitterCustomShaderSettingDataにFlagsデータがあるのでスキップする
                    generator.SkipFirstRadioButtonGroup = true;  // EmitterCustomShaderSettingDataにSwitch0, Switch1, ...データがあるのでスキップする

                    generator.Generate(def, dataModelName);

                    // Generate editor data models.
                    var editorDataModelGenerator = new DataModelGenerator(
                        generator.WorkspaceDefinition.EditorDataModelDefinition);

                    editorDataModelGenerator.Generate(outputFolderPath);

                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "Export data model ==> Success!");

                    // Generate binary conversion info.
                    var converterGenerator = new BinaryConversionInfoGenerator(
                        generator.WorkspaceDefinition.RuntimeDataModelDefinition);

                    converterGenerator.Generate(outputFolderPath);

                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "Export binary conversion info ==> Success!");

                    // Write UI Xaml file.
                    var xamlWriter = new UserDefinitionXamlWriter()
                    {
                        FirstCheckBoxGroupPropertyName = "Flags",
                    };

                    xamlWriter.Write(
                        def,
                        Path.Combine(outputFolderPath, dataModelName + ".xaml"),
                        language);

                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "Export UI Xaml file ==> Success!\n");

                    string usdFileName = Path.GetFileNameWithoutExtension(usdFilePath);

                    // Write UI Xaml file.
                    var customShaderUIDefWriter = new CustomShaderUIDefinitionWriter();

                    customShaderUIDefWriter.Write(
                        def,
                        dataModelName,
                        Path.Combine(outputFolderPath, usdFileName + ".xml"),
                        language);

                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "Export custom shader UI definition Xml file ==> Success!\n");
                }
            }

            return true;
        }

        /// <summary>
        /// UDDファイルを新形式にコンバートします。
        /// </summary>
        /// <param name="uddFilePath">UDDファイルパス</param>
        /// <param name="dataModelName">出力するデータモデルの名前</param>
        /// <returns>コンバートが成功したときtrueを返します。</returns>
        public bool ConvertUdd(string uddFilePath, string dataModelName, string outputFolderPath, string language)
        {
            Logger.Log(
                "ConsoleLogger",
                LogLevels.Information,
                "Converting .udd files...\n- File path :       {0}\n- Data model name : {1}\n- Output folder :   {2}\n- Language :   {3}\n",
                uddFilePath,
                dataModelName,
                outputFolderPath,
                language);

            using (var stream = new FileStream(uddFilePath, FileMode.Open, FileAccess.Read))
            {
                // Load user definitions.
                var def = SerializationHelper.LoadXmlDocSerializable<CustomActionDefinition>(stream);
                def.PostLoad();

                Logger.Log(
                    "ConsoleLogger",
                    LogLevels.Information,
                    "Load .usd file ==> Success!");

                // Convert data models, binary conversion info and UI Xaml.
                using (var generator = new UserDefinitionDataModelGenerator())
                {
                    generator.DataModelBaseClassName = "CustomActionSettingData";
                    generator.SkipFirstCheckBoxGroup = false;

                    generator.Generate(def, dataModelName);

                    // Generate editor data models.
                    var editorDataModelGenerator = new DataModelGenerator(
                        generator.WorkspaceDefinition.EditorDataModelDefinition);

                    editorDataModelGenerator.Generate(outputFolderPath);

                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "Export data model ==> Success!");

                    // Generate binary conversion info.
                    var converterGenerator = new BinaryConversionInfoGenerator(
                        generator.WorkspaceDefinition.RuntimeDataModelDefinition);

                    converterGenerator.Generate(outputFolderPath);

                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "Export binary conversion info ==> Success!");

                    // Write UI Xaml file.
                    var xamlWriter = new UserDefinitionXamlWriter();

                    xamlWriter.Write(
                        def,
                        Path.Combine(outputFolderPath, dataModelName + ".xaml"),
                        language);

                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "Export UI Xaml file ==> Success!\n");
                }
            }

            return true;
        }
    }
}
